from qgis.PyQt.QtCore import (QT_TRANSLATE_NOOP, QVariant, QCoreApplication)
from qgis.core import (
  QgsProcessing,
  QgsProcessingAlgorithm,
  QgsProcessingParameterFeatureSource,
  QgsProcessingParameterFeatureSink,
  QgsProcessingParameterString,
  QgsProcessingParameterBoolean,
  QgsFeatureRequest,
  QgsProcessingParameterField,
  QgsFeature,
  QgsFields,
  QgsField,
  QgsProcessingParameterNumber
  )

from ..algutil.hriskvar import PostProcessors
from ..algutil.hriskutil import HrUtil
from ..algutil.hriskfields import HrFields
from ..algutil.hriskpostprocessor import HrPostProcessor

class estimateriskofbuilding(QgsProcessingAlgorithm):
  PARAMETERS = {
    "BUILDING": {
      "ui_func": QgsProcessingParameterFeatureSource,
      "ui_args":{
        "description": QT_TRANSLATE_NOOP("estimateriskofbuilding","Building layer"),
        "types": [QgsProcessing.TypeVectorPolygon]
      }
    },
    "LDEN": {
      "ui_func": QgsProcessingParameterField,
      "ui_args":{
        "description": QT_TRANSLATE_NOOP("estimateriskofbuilding","Lden field"),
        "parentLayerParameterName": "BUILDING",
        "defaultValue": "LDEN_LAEQ_max"
      }
    },
    "LNIGHT": {
      "ui_func": QgsProcessingParameterField,
      "ui_args":{
        "description": QT_TRANSLATE_NOOP("estimateriskofbuilding","Lnight field"),
        "parentLayerParameterName": "BUILDING",
        "defaultValue": "LN_LAEQ_max"
      }
    },
    "POP": {
      "ui_func": QgsProcessingParameterField,
      "ui_args":{
        "description": QT_TRANSLATE_NOOP("estimateriskofbuilding","Population field"),
        "parentLayerParameterName": "BUILDING"
      }
    },
    "BASERISK_IHD_INC": {
      "ui_func": QgsProcessingParameterNumber,
      "ui_args":{
        "optional": True,
        "description": QT_TRANSLATE_NOOP("estimateriskofbuilding","Base-line risk of IHD incident, per 100000"),
        "type": QgsProcessingParameterNumber.Double
      }
    },
    "BASERISK_IHD_DEATH": {
      "ui_func": QgsProcessingParameterNumber,
      "ui_args":{
        "optional": True,
        "description": QT_TRANSLATE_NOOP("estimateriskofbuilding","Base-line risk of IHD death, per 100000"),
        "type": QgsProcessingParameterNumber.Double
      }
    },
    "FORMULA_HA": {
      "advanced": True,
      "ui_func": QgsProcessingParameterString,
      "ui_args":{
        "description": QT_TRANSLATE_NOOP("estimateriskofbuilding","Formula to calculate %HA"),
        "defaultValue": "0.78927 - 0.031162 * {Lden} + 0.000342 * {Lden} ** 2 if {Lden} > 40.0 else 0.0"
      }      
    },
    "FORMULA_HSD": {
      "advanced": True,
      "ui_func": QgsProcessingParameterString,
      "ui_args":{
        "description": QT_TRANSLATE_NOOP("estimateriskofbuilding","Formula to calculate %HSD"),
        "defaultValue": "0.194312 - 0.009336 * {Lnight} + 0.000126 * {Lnight} ** 2 if {Lnight} > 40.0 else 0.0"
      }      
    },
    "FORMULA_IHD": {
      "advanced": True,
      "ui_func": QgsProcessingParameterString,
      "ui_args":{
        "description": QT_TRANSLATE_NOOP("estimateriskofbuilding","Formula to calculate relative risk of IHD"),
        "defaultValue": "1.08 ** (({Lden} - 53.0) / 10) if {Lden} > 53.0 else 1.0"
      }
    },
    "OUTPUT": {
      "ui_func": QgsProcessingParameterFeatureSink,
      "ui_args": {
        "description": QT_TRANSLATE_NOOP("estimateriskofbuilding","Building with health risks")
      }
    }
  }
  
  def __init__(self) -> None:
    super().__init__()
    self.UTIL = HrUtil(self)
  
  
  def initAlgorithm(self, config):
    self.UTIL.initParameters()

  def processAlgorithm(self, parameters, context, feedback):       
    self.UTIL.registerProcessingParameters(parameters, context, feedback)
    self.CURRENT_PROCESS = self.UTIL.parseCurrentProcess(with_nm=True)
    
    pop_field = self.parameterAsString(parameters, "POP", context)
    Lden_field = self.parameterAsString(parameters, "LDEN", context)
    Lnight_field = self.parameterAsString(parameters, "LNIGHT", context)
    Lden_formula = f"ft['{Lden_field}']" 
    Lnight_formula = f"ft['{Lnight_field}']"
    
    risk_formula = {}
    
    # for IHD
    rel_risk_ihd_formula = self.parameterAsString(parameters, "FORMULA_IHD", context).format(Lden = Lden_formula, Lnight = Lnight_formula)
    ihd_inc_base = self.parameterAsDouble(parameters, "BASERISK_IHD_INC", context)
    ihd_death_base = self.parameterAsDouble(parameters, "BASERISK_IHD_DEATH", context)
    risk_formula |= {"relRiskIHD": rel_risk_ihd_formula}
      
    # for HA
    ha_formula = self.parameterAsString(parameters, "FORMULA_HA", context).format(Lden = Lden_formula, Lnight = Lnight_formula)
    risk_formula |= {"probHA": ha_formula}
                           
    # for HSD
    hsd_formula = self.parameterAsString(parameters, "FORMULA_HSD", context).format(Lden = Lden_formula, Lnight = Lnight_formula)
    risk_formula |= {"probHSD": hsd_formula}

    # calculate the risks
    bldg_layer = self.parameterAsSource(parameters, "BUILDING", context).materialize(QgsFeatureRequest(), feedback)
    
    risk_ests = {"pop": []} | {risk_key: [] for risk_key in risk_formula.keys()} 
    
    # risk estimation using the formula
    for ft in bldg_layer.getFeatures():
      risk_ests["pop"].append(ft[pop_field])
      for risk_key, formula in risk_formula.items():
        risk_ests[risk_key].append(eval(formula))
    
    # risk estimation using the results
    if ihd_inc_base >0:
      risk_ests["nIncIHD"] = [
        (rr - 1) * ihd_inc_base / 1e5 * pop
        for rr, pop in zip(risk_ests["relRiskIHD"], risk_ests["pop"])
      ]
    
    if ihd_death_base >0:
      risk_ests["nDeathIHD"] = [
        (rr - 1) * ihd_death_base / 1e5 * pop
        for rr, pop in zip(risk_ests["relRiskIHD"], risk_ests["pop"])
      ]
    
    risk_ests["nHA"] = [
      (prob * pop) / 100.0
      for prob, pop in zip(risk_ests["probHA"], risk_ests["pop"])
    ]
    risk_ests["nHSD"] = [
      (prob * pop) / 100.0
      for prob, pop in zip(risk_ests["probHSD"], risk_ests["pop"])
    ]
    
    del risk_ests["pop"]
    
    dest_id = self.UTIL.outputVectorLayer(
      vector_layer = bldg_layer,
      param_sink = "OUTPUT",
      fields_with_values = {
        "HISTORY": {
          "type": QVariant.String, 
          "value": self.CURRENT_PROCESS, 
          "append": True
        }
      } | {
        k: {"type": QVariant.Double,"value": v} 
        for k, v in risk_ests.items()
      }
    )
    
    
    PostProcessors[dest_id] = HrPostProcessor(
      history = [self.CURRENT_PROCESS],
      coloring = "none"
      )
    self.UTIL.registerPostProcessAlgorithm(context, PostProcessors)
          
    return {"OUTPUT": dest_id}

  def name(self):
    return self.__class__.__name__  

  def displayName(self):
    return self.tr("Estimate health risks of buildings")

  def group(self):
    return self.tr("Evaluate health risk")

  def groupId(self):
    return "healthrisk"
  
  def createInstance(self):
    return estimateriskofbuilding()

  # placing here is necessary, when employing pylupdate
  def tr(self, string):
    return QCoreApplication.translate(self.__class__.__name__, string)
