from qgis.PyQt.QtCore import (QCoreApplication, QT_TRANSLATE_NOOP, QVariant)
from qgis.core import (
  QgsProcessing,
  QgsProcessingAlgorithm,
  QgsProcessingParameterFeatureSource,
  QgsProcessingParameterFeatureSink,
  QgsProcessingParameterCrs,
  QgsProcessingParameterDistance,
  QgsProcessingParameterFeatureSource,
  QgsFeatureRequest,
  QgsFeature,
  QgsWkbTypes,
  QgsVectorLayer,
  QgsField
  )

from qgis import processing

from ..algutil.hriskvar import PostProcessors
from ..algutil.hriskutil import HrUtil
from ..algutil.hriskfields import HrFields
from ..algutil.hriskpostprocessor import HrPostProcessor
import numpy as np

class receivercrosssection(QgsProcessingAlgorithm):
  PARAMETERS = { 
    "RECEIVER_LINE": {
      "crs_reference": True, # this parameter is used as CRS reference
      "ui_func": QgsProcessingParameterFeatureSource,
      "ui_args":{
        "description": QT_TRANSLATE_NOOP("receivercrosssection","Receiver line layer"),
        "types": [QgsProcessing.TypeVectorLine]
      }
    },    
    
    "TARGET_CRS": {
      "ui_func": QgsProcessingParameterCrs,
      "ui_args": {
        "description": QT_TRANSLATE_NOOP("receivercrosssection","Target CRS (Cartesian coordinates)")
      }
    },
    "INTERVAL": {
      "ui_func": QgsProcessingParameterDistance,
      "ui_args": {
        "description": QT_TRANSLATE_NOOP("receivercrosssection","Horizontal / vertical interval between receivers"),
        "defaultValue": 2.0,
        "parentParameterName": "TARGET_CRS"
      }
    },
    "HEIGHT_MIN": {
      "ui_func": QgsProcessingParameterDistance,
      "ui_args": {
        "description": QT_TRANSLATE_NOOP("receivercrosssection","Minimum height of the receivers"),
        "defaultValue": 2.0,
        "parentParameterName": "TARGET_CRS"
      }
    },
    "HEIGHT_MAX": {
      "ui_func": QgsProcessingParameterDistance,
      "ui_args": {
        "description": QT_TRANSLATE_NOOP("receivercrosssection","Maximum height of the receivers"),
        "defaultValue": 20.0,
        "parentParameterName": "TARGET_CRS"
      }
    },
    "OUTPUT": {
      "ui_func": QgsProcessingParameterFeatureSink,
      "ui_args": {
        "description": QT_TRANSLATE_NOOP("receivercrosssection","Receivers of crosssection" )
      }
    }

  }
  
  def __init__(self) -> None:
    super().__init__()
    self.UTIL = HrUtil(self)
    
  def initAlgorithm(self, config):
    (extent, target_crs) = self.UTIL.getExtentAndCrsUsingCanvas()
    self.UTIL.setDefaultValue("TARGET_CRS", target_crs.authid())
    self.UTIL.initParameters()

  def processAlgorithm(self, parameters, context, feedback):   
    
    self.UTIL.registerProcessingParameters(parameters, context, feedback)
    self.CURRENT_PROCESS = self.UTIL.parseCurrentProcess()
    target_crs = self.parameterAsCrs(parameters, "TARGET_CRS", context)
    self.UTIL.checkCrsAsCartesian(target_crs)
    
    # list from HEIGHT_MIN to HEIGHT_MAX with INTERVAL
    receiver_heights = np.arange(
      self.parameterAsDouble(parameters, "HEIGHT_MIN", context),
      self.parameterAsDouble(parameters, "HEIGHT_MAX", context),
      self.parameterAsDouble(parameters, "INTERVAL", context)
    )
    
    receiver_line = self.parameterAsSource(parameters, "RECEIVER_LINE", context)
    receiver_fields = HrFields.fromQgsFieldList([
      QgsField("PK", QVariant.Int, "int"),
      QgsField("HISTORY", QVariant.String, "string")      
    ])
    
    receiver_interval = self.parameterAsDouble(parameters, "INTERVAL", context)
      
    line_layer_raw = processing.run(
      "native:reprojectlayer", 
      {
        "INPUT": receiver_line.materialize(QgsFeatureRequest(), feedback),
        "TARGET_CRS": target_crs,
        "OUTPUT": "TEMPORARY_OUTPUT"
      },
      context = context,
      is_child_algorithm = True,
    )["OUTPUT"]      
            
    line_layer_single = processing.run(
      "native:multiparttosingleparts",
      {
        "INPUT": line_layer_raw,
        "OUTPUT": "memory:singleparts"
      },
      context = context,
      is_child_algorithm = True,
    )["OUTPUT"]
    # line_layer_single = QgsVectorLayer(line_layer_single, "singleparts", "memory")

    line_layer_single = context.getMapLayer(line_layer_single)
    
    # set sink and add features with values
    (sink, dest_id) = self.parameterAsSink(
      parameters, "OUTPUT", context,
      receiver_fields, QgsWkbTypes.PointZ, target_crs
    )
      
    pk = 1
    for ft in line_layer_single.getFeatures():
      line_layer = QgsVectorLayer(f"LineString?crs={target_crs.authid()}", "temp", "memory")
      line_layer.dataProvider().addFeatures([ft])
      pnts_on_line = processing.run(
        "native:pointsalonglines",
        {
          "INPUT": line_layer,
          "DISTANCE": receiver_interval,
          "START_OFFSET": 0,
          "END_OFFSET": 0,
          "OUTPUT": "TEMPORARY_OUTPUT"
        },
        context = context,
        is_child_algorithm = True,
      )["OUTPUT"]
      
      for height in receiver_heights:
        
        # set z value
        pnt_z = processing.run(
          "native:setzvalue",
          {
            "INPUT": pnts_on_line,
            "Z_VALUE": float(height),
            "OUTPUT": "memory:dem"
          },
          context = context,
          is_child_algorithm = True
        )["OUTPUT"]
        pnt_z = context.getMapLayer(pnt_z)
        
        for pnt in pnt_z.getFeatures():
        
          ft_new = QgsFeature(receiver_fields)
          ft_new.setGeometry(pnt.geometry())
          ft_new["PK"] = pk
          ft_new["HISTORY"] = self.CURRENT_PROCESS
          sink.addFeature(ft_new)
          pk += 1
      
      PostProcessors[dest_id] = HrPostProcessor(history = [self.CURRENT_PROCESS])
      
    return {"OUTPUT": dest_id}
    
  def name(self):
    return self.__class__.__name__
  
  def displayName(self):
    return self.tr("Cross section")

  def group(self):
    return self.tr('Set receivers')

  def groupId(self):
    return 'receiver'

  def createInstance(self):
    return receivercrosssection()

  # placing here is necessary, when employing pylupdate
  def tr(self, string):
    return QCoreApplication.translate(self.__class__.__name__, string)
