from qgis.PyQt.QtCore import (QT_TRANSLATE_NOOP, QVariant)

from qgis.core import (
  QgsProcessing,
  QgsProcessingAlgorithm,
  QgsProcessingParameterFeatureSource,
  QgsProcessingParameterDistance,
  QgsProcessingParameterNumber,
  QgsProcessingParameterFeatureSink,
  QgsFeatureRequest
  )

from qgis import processing

from ..algutil.hriskutil import HrUtil
from ..algutil.hriskpostprocessor import HrPostProcessor
from ..algutil.hriskvar import PostProcessors
class heightsalonglines(QgsProcessingAlgorithm):
  

  PARAMETERS = {            
    "INPUT": {
      "crs_reference": True, # this parameter is used as CRS reference
      "ui_func": QgsProcessingParameterFeatureSource,
      "ui_args":{
        "description": QT_TRANSLATE_NOOP("heightsalonglines","Input layer"),
        "types": [QgsProcessing.TypeVectorLine]
      }
    },
    "INTERVAL": {
      "ui_func": QgsProcessingParameterDistance,
      "ui_args": {
        "description": QT_TRANSLATE_NOOP("heightsalonglines","Interval between points"),
        "defaultValue": 10.0,
        "parentParameterName": "INPUT"
      }
    },
    "HEIGHT": {
      "ui_func": QgsProcessingParameterNumber,
      "ui_args": {
        "description": QT_TRANSLATE_NOOP("heightsalonglines","Height (m)"),
        "type": QgsProcessingParameterNumber.Double,
        "minValue": -100.0, "defaultValue": 0.0, "maxValue": 10000.0
      }
    },
    "OUTPUT": {
      "ui_func": QgsProcessingParameterFeatureSink,
      "ui_args": {
        "description": QT_TRANSLATE_NOOP("heightsalonglines","Elevation points" )
      },
      "visibleByDefault": False
    }
  }
  
  def __init__(self) -> None:
    super().__init__()
    self.UTIL = HrUtil(self)
  
  def initAlgorithm(self, config):
    self.UTIL.initParameters()
    
  def processAlgorithm(self, parameters, context, feedback):  
    self.UTIL.registerProcessingParameters(parameters, context, feedback)
    self.CURRENT_PROCESS = self.UTIL.parseCurrentProcess()

    point_xy = processing.run(
      "native:pointsalonglines",
      {
        "INPUT": self.parameterAsSource(parameters, "INPUT", context).materialize(QgsFeatureRequest(), feedback),
        "INTERVAL": self.parameterAsDouble(parameters, "INTERVAL", context),
        "OUTPUT": "TEMPORARY_OUTPUT"
      },
      context = context,
      is_child_algorithm = True,
    )["OUTPUT"]
    
    point_xyz = processing.run(
      "native:setzvalue",
      {
        "INPUT": point_xy,
        "Z_VALUE": self.parameterAsDouble(parameters, "HEIGHT", context),
        "OUTPUT": "TEMPORARY_OUTPUT"
      },
      context = context,
      is_child_algorithm = True,
    )["OUTPUT"]
        
    point_elev = processing.run(
      "hrisk:initelevationpoint",
      {
          "INPUT": point_xyz,
          "TARGET_CRS": self.parameterAsCrs(parameters, "INPUT", context),
          "OVERWRITE": True,
          "OUTPUT": "TEMPORARY_OUTPUT"
      },
      context = context,
      is_child_algorithm = True,
    )["OUTPUT"]
    
    point_elev = context.getMapLayer(point_elev)
    new_fields = self.UTIL.newFieldsWithHistory(point_elev.fields())
    (sink, dest_id) = self.parameterAsSink(
      parameters, "OUTPUT", context, 
      new_fields, point_elev.wkbType(), point_elev.sourceCrs()
    )
    self.UTIL.addFeaturesWithHistoryToSink(
      sink, point_elev, new_fields, self.CURRENT_PROCESS
    )
    
    PostProcessors[dest_id] = HrPostProcessor(history = [self.CURRENT_PROCESS])
    
    return {"OUTPUT": dest_id}
  
  
  def name(self):
    return self.__class__.__name__
  
  def displayName(self):
    return self.tr("Elevation points along lines")

  def group(self):
    return self.tr("Initialize features")

  def groupId(self):
    return "initfeature"

  def createInstance(self):
    return heightsalonglines()
