from qgis.PyQt.QtCore import (QT_TRANSLATE_NOOP, QCoreApplication)
from qgis.core import (
  QgsRectangle,
  QgsProcessingAlgorithm,
  QgsReferencedRectangle,
  QgsCoordinateTransform,
  QgsProject,
  QgsCoordinateReferenceSystem,
  QgsProcessingParameterExtent,
  QgsProcessingParameterDistance,
  QgsProcessingParameterFeatureSink,
  QgsProcessingParameterCrs, 
  QgsProcessingParameterString,
  QgsProcessingParameterNumber
  )
from qgis import processing

from ..algutil.hriskutil import HrUtil, HrTileMapFetcher

class fetchany(QgsProcessingAlgorithm):
  
  # UIs
  PARAMETERS = {  
    "FETCH_EXTENT": {
      "ui_func": QgsProcessingParameterExtent,
      "ui_args":{
        "description": QT_TRANSLATE_NOOP("fetchany","Extent for fetching data")
      }
    },
    "TARGET_CRS": {
      "ui_func": QgsProcessingParameterCrs,
      "ui_args": {
        "description": QT_TRANSLATE_NOOP("fetchany","Target CRS (Cartesian coordinates)")
      }
    },
    "BUFFER": {
      "ui_func": QgsProcessingParameterDistance,
      "ui_args": {
        "description": QT_TRANSLATE_NOOP("fetchany","Buffer of the fetch area (using Target CRS)"),
        "defaultValue": 0.0,
        "parentParameterName": "TARGET_CRS"
      }
    },
    "TILEMAP_URL": {
      "ui_func": QgsProcessingParameterString,
      "ui_args": {
        "optional": True,
        "description": QT_TRANSLATE_NOOP("fetchany","Base-URL of the vector-tile map"),
        "defaultValue": "https://cyberjapandata.gsi.go.jp/xyz/experimental_bvmap/{z}/{x}/{y}.pbf|layername=building|geometrytype=Polygon"
      }
    },
    "TILEMAP_CRS": {
      "ui_func": QgsProcessingParameterCrs,
      "ui_args": {
        "optional": True,
        "description": QT_TRANSLATE_NOOP("fetchany","CRS of the vector-tile map"),        
        "defaultValue": "EPSG:3857" # must be specified as string, because optional parameter cannot be set as QgsCoordinateReferenceSystem
      }
    },
    "TILEMAP_ZOOM": {
      "ui_func": QgsProcessingParameterNumber,
      "ui_args": {
        "optional": True,
        "description": QT_TRANSLATE_NOOP("fetchany","Zoom level of the vector-tile map"),
        "type": QgsProcessingParameterNumber.Integer,
        "defaultValue": 16
      }
    },
    "OUTPUT": {
      "ui_func": QgsProcessingParameterFeatureSink,
      "ui_args": {
        "description": QT_TRANSLATE_NOOP("fetchany","Building")
      }
    }
  }  
  
  FETCH_CRS = QgsCoordinateReferenceSystem("EPSG:6668")
  
  def __init__(self):
    super().__init__()
    self.UTIL = HrUtil(self)
    
  # initialization of the algorithm
  def initAlgorithm(self, config):    
    (extent, target_crs) = self.UTIL.getExtentAndCrsUsingCanvas()
    self.UTIL.setDefaultValue("FETCH_EXTENT", extent)
    self.UTIL.setDefaultValue("TARGET_CRS", target_crs.authid())
    self.UTIL.initParameters()
  
  # execution of the algorithm
  def processAlgorithm(self, parameters, context, feedback):    
    import ptvsd
    ptvsd.debug_this_thread()
    
    self.UTIL.registerProcessingParameters(parameters, context, feedback)
    self.CURRENT_PROCESS = self.UTIL.parseCurrentProcess()
    # get target x-y CRS, to apply the buffer and determine the fetch area
    target_crs = self.parameterAsCrs(parameters, "TARGET_CRS", context)
    fetch_crs = self.FETCH_CRS
    tile_crs = self.parameterAsCrs(parameters, "TILEMAP_CRS", context)
    
    # check whether the target CRS is x-y coordinates
    self.UTIL.checkCrsAsCartesian(target_crs)
    
    # get the extent, using the target CRS
    fetch_extent = self.parameterAsExtent(
      parameters, "FETCH_EXTENT", context, 
      self.parameterAsCrs(parameters, "TARGET_CRS", context)
      )
    
    # get the buffer
    buffer = self.parameterAsDouble(parameters, "BUFFER",context)
    
    # get the fetch area, using the extent and buffer
    fetch_area_xy = QgsReferencedRectangle(
      QgsRectangle(
        fetch_extent.xMinimum() - buffer,
        fetch_extent.yMinimum() - buffer,
        fetch_extent.xMaximum() + buffer,
        fetch_extent.yMaximum() + buffer
      ),
      target_crs
    )
    
    # get the fetch area, using the fetch crs
    transform = QgsCoordinateTransform(target_crs, fetch_crs, QgsProject.instance())
    fetch_area = QgsReferencedRectangle(transform.transformBoundingBox(fetch_area_xy), fetch_crs)
    feedback.pushInfo(self.tr("Transform: ") + transform.instantiatedCoordinateOperationDetails().name)
    
    # unreasonable area checker
    if fetch_crs.isGeographic():
      if fetch_area.xMinimum() < -180 or fetch_area.yMinimum() < -90 or fetch_area.xMaximum() > 180 or fetch_area.yMaximum() > 90:
        feedback.reportError(self.tr("Unsolved problem in transforming coordinates is occurred. Save features, restart QGIS and try again."))
        raise Exception(self.tr("Unsolved problem in transforming coordinates is occurred. Save features, restart QGIS and try again."))
  
    
    fetcher = HrTileMapFetcher(
      url = self.parameterAsString(parameters, "TILEMAP_URL", context),
      crs = self.parameterAsCrs(parameters, "TILEMAP_CRS", context),
      zoom = self.parameterAsInt(parameters, "TILEMAP_ZOOM", context),
      geom_type = "LineString",
      fetch_area = fetch_area,
      feedback = feedback
    )
    
    fetch_args = fetcher.parseFetchArgs()      
    fetched_result = fetcher.fetchFeaturesFromTileConcurrently(fetch_args)
    for k in fetch_args:
      import re
      fetch_args[k]["url"] = re.search(r'^[^|]*', fetch_args[k]["url"]).group(0)
    fetch_results = self.UTIL.downloadFilesConcurrently(args = fetch_args)
    feedback.pushInfo(self.tr("The coordinates are transformed."))
    modified_result = {}
    for key, value in fetched_result.items():
      # constants
      EquatorLength = 40075016.68557849
      N_PIXELS_IN_GSI_VTILE = 4096
      n_pixels_all = N_PIXELS_IN_GSI_VTILE * 2 ** fetch_args[key]["zoom"]
      meter_per_tile  = EquatorLength / 2 ** fetch_args[key]["zoom"]
      meter_per_pixel = EquatorLength / n_pixels_all
      
      # affine transformation to obtain x and y for a given CRS
      modified_result[key] = processing.run(
        "native:affinetransform",
        {
          "INPUT": value,
          "DELTA_X":    fetch_args[key]["tx"]    * meter_per_tile - EquatorLength / 2,
          "DELTA_Y": - (fetch_args[key]["ty"]+1) * meter_per_tile + EquatorLength / 2,
          "SCALE_X": meter_per_pixel,
          "SCALE_Y": meter_per_pixel,
          "OUTPUT": "TEMPORARY_OUTPUT"
        },
        context = context,
        is_child_algorithm = True
      )["OUTPUT"]
    
    bldg_raw = processing.run(
      "native:mergevectorlayers",
      {
        "LAYERS": list(modified_result.values()),
        "CRS": tile_crs,
        "OUTPUT": "TEMPORARY_OUTPUT"
      },
      context = context,
      is_child_algorithm = True
    )["OUTPUT"]
    
      # transform
    bldg_transformed = processing.run(
    "native:reprojectlayer", 
    {
      "INPUT": bldg_raw,
      "TARGET_CRS": target_crs,
      "OUTPUT": "TEMPORARY_OUTPUT"
    },
    context = context,
    is_child_algorithm = True
    )["OUTPUT"]      
    
    
    feedback.pushInfo(self.tr("Snapping and dissolving are applied."))
    # snap geometry
    bldg_snap = processing.run(
      "native:snapgeometries", 
      {
        "INPUT": bldg_transformed,
        "REFERENCE_LAYER": context.getMapLayer(bldg_transformed),
        "TOLERANCE": 0.1,
        "BEHAVIOR": 0,
        "OUTPUT": "TEMPORARY_OUTPUT"
      },
      context = context,
      is_child_algorithm = True
    )["OUTPUT"]
      
    # dissolve
    bldg_dissolve = self.UTIL.dissolve(bldg_snap)
      
    # extract
    bldg_extract = processing.run(
      "native:extractbyextent",
      {
        "INPUT": bldg_dissolve,
        "EXTENT": fetch_area,
        "OUTPUT": "TEMPORARY_OUTPUT"
      },
      context = context,
      is_child_algorithm = True
    )["OUTPUT"]
            
    bldg_final = processing.run(
      "hrisk:initbuilding",{
        "INPUT": bldg_extract,
        "TARGET_CRS": target_crs,
        "OVERWRITE": True,
        "OUTPUT": "TEMPORARY_OUTPUT"
      },
      context = context,
      is_child_algorithm = True
    )["OUTPUT"]
      
      
    bldg_final_fts = context.getMapLayer(bldg_final)
    new_fields = self.UTIL.newFieldsWithHistory(bldg_final_fts.fields())
    (sink, dest_id) = self.parameterAsSink(
      parameters, "OUTPUT", context,
      new_fields, bldg_final_fts.wkbType(), target_crs
    )
    self.UTIL.addFeaturesWithHistoryToSink(
      sink, bldg_final_fts, new_fields,
      current_process=self.CURRENT_PROCESS
    )
    
    return {"OUTPUT": dest_id}
    
  def name(self):
    return self.__class__.__name__

  def displayName(self):
    return self.tr("Buildings (Ja)")
  
  def group(self):
    return self.tr('Fetch geometries (Ja)')

  def groupId(self):
    return 'fetchgeomja'

  def createInstance(self):
    return fetchany()

  # placing here is necessary, when employing pylupdate
  def tr(self, string):
    return QCoreApplication.translate(self.__class__.__name__, string)
