from qgis.PyQt.QtCore import (QCoreApplication, QT_TRANSLATE_NOOP, QVariant)
from qgis.core import (
  QgsProcessingAlgorithm,
  QgsProcessingParameterExtent,
  QgsProcessingParameterDistance,
  QgsProcessingParameterCrs, 
  QgsProcessingParameterVectorDestination,
  QgsProcessingParameterRasterDestination,
  QgsProcessingParameterNumber,
  QgsProcessingParameterBoolean,
  QgsFeature,
  QgsGeometry,
  QgsField,
  QgsRectangle,
  QgsReferencedRectangle,
  QgsWkbTypes,
  QgsProcessingParameterFeatureSink
  )
from qgis import processing
import datetime


from ..algutil.hriskvar import PostProcessors
from ..algutil.hriskutil import HrUtil
from ..algutil.hriskfields import HrFields
from ..algutil.hriskpostprocessor import HrPostProcessor

class fetchallja(QgsProcessingAlgorithm):
  PARAMETERS = {  
    "FETCH_EXTENT": {
      "ui_func": QgsProcessingParameterExtent,
      "ui_args":{
        "description": QT_TRANSLATE_NOOP("fetchallja","Extent for fetching data")
      }
    },
    "TARGET_CRS": {
      "ui_func": QgsProcessingParameterCrs,
      "ui_args": {
        "description": QT_TRANSLATE_NOOP("fetchallja","Target CRS (Cartesian coordinates)")
      }
    },
    "BUFFER": {
      "ui_func": QgsProcessingParameterDistance,
      "ui_args": {
        "description": QT_TRANSLATE_NOOP("fetchallja","Buffer of the fetch area (using Target CRS)"),
        "defaultValue": 0.0,
        "parentParameterName": "TARGET_CRS"
      }
    },
    
    "SET_RECEIVER_FACADE": {
      "ui_func": QgsProcessingParameterBoolean,
      "ui_args": {
        "description": QT_TRANSLATE_NOOP("fetchallja","Set receiver at building facade?"),
        "defaultValue": False
      }
    },
    
    "SET_RECEIVER_DELAUNAYGRID": {
      "ui_func": QgsProcessingParameterBoolean,
      "ui_args": {
        "description": QT_TRANSLATE_NOOP("fetchallja","Set receiver at delaynay grid point?"),
        "defaultValue": False
      }
    },
    
    "DELTA": {
      "advanced": True,
      "ui_func": QgsProcessingParameterNumber,
      "ui_args": {
        "description": QT_TRANSLATE_NOOP("fetchallja","Distance between receivers (m)"),
        "type": QgsProcessingParameterNumber.Double,
        "minValue": 1.0, "defaultValue": 3.0, "maxValue": 100.0
      },
      "n_mdl": "delta"
    },
    "HEIGHT": {
      "advanced": True,
      "ui_func": QgsProcessingParameterNumber,
      "ui_args": {
        "description": QT_TRANSLATE_NOOP("fetchallja","Height of receivers (m)"),
        "type": QgsProcessingParameterNumber.Double,
        "minValue": 0.1, "defaultValue": 4.0, "maxValue": 100.0
      },
      "n_mdl": "height"
    },    
    "DISTANCE": {
      "ui_func": QgsProcessingParameterNumber,
      "ui_args": {
        "description": QT_TRANSLATE_NOOP("fetchallja","Distance from the wall (m)"),
        "type": QgsProcessingParameterNumber.Double,
        "minValue": 0.01, "defaultValue": 0.1, "maxValue": 5.0
      },
      "n_mdl": "distance"
    },
    "MAX_CELL_DIST": {
      "advanced": True,
      "ui_func": QgsProcessingParameterNumber,
      "ui_args": {
        "description": QT_TRANSLATE_NOOP("fetchallja","Maximum distance used to split the domain into sub-domains (m)"),
        "type": QgsProcessingParameterNumber.Double,
        "minValue": 100.0, "defaultValue": 600.0, "maxValue": 2000.0
      },
      "n_mdl": "maxPropDist"
    },
    "ROAD_WIDTH": {
      "advanced": True,
      "ui_func": QgsProcessingParameterNumber,
      "ui_args": {
        "description": QT_TRANSLATE_NOOP("fetchallja","Road width (m), where no receivers will be set closer than it"),
        "type": QgsProcessingParameterNumber.Double,
        "minValue": 1.0, "defaultValue": 2.0, "maxValue": 20.0
      },
      "n_mdl": "roadWidth"
    },
    "MAX_AREA": {
      "advanced": True,
      "ui_func": QgsProcessingParameterNumber,
      "ui_args": {
        "description": QT_TRANSLATE_NOOP("fetchallja","Maximum trianglar area (m2)"),
        "type": QgsProcessingParameterNumber.Double,
        "minValue": 10.0, "defaultValue": 500.0, "maxValue": 10000.0
      },
      "n_mdl": "maxArea"
    },
    "FETCH_AREA": {
      "ui_func": QgsProcessingParameterFeatureSink,
      "ui_args": {
        "description": QT_TRANSLATE_NOOP("fetchallja","Fetch area" )
      },
      "visibleByDefault": False
    },
    "ROAD": {
      "ui_func": QgsProcessingParameterVectorDestination,
      "ui_args": {
        "description": QT_TRANSLATE_NOOP("fetchallja","Roads" )
      },
      "visibleByDefault": True
    },
    "BUILDING": {
      "ui_func": QgsProcessingParameterVectorDestination,
      "ui_args": {
        "description": QT_TRANSLATE_NOOP("fetchallja","Buildings" )
      },
      "visibleByDefault": True
    },    
    "RECEIVER_FACADE": {
      "ui_func": QgsProcessingParameterVectorDestination,
      "ui_args": {
        "description": QT_TRANSLATE_NOOP("fetchallja","Receivers at facade" )
      }     ,
      "visibleByDefault": False
    },    
    "RECEIVER_DELAUNAY": {
      "ui_func": QgsProcessingParameterVectorDestination,
      "ui_args": {
        "description": QT_TRANSLATE_NOOP("fetchallja","Receivers of Delaunay" )
      },
      "visibleByDefault": False
    },
    "TRIANGLE_DELAUNAY": {
      "ui_func": QgsProcessingParameterVectorDestination,
      "ui_args": {
        "description": QT_TRANSLATE_NOOP("fetchallja","Triangles of Delaunay" )
      },
      "visibleByDefault": False
    },    
    "DEM": {
      "ui_func": QgsProcessingParameterVectorDestination,
      "ui_args": {
        "description": QT_TRANSLATE_NOOP("fetchallja","Elevation points" )
      },
      "visibleByDefault": False
    },    
    "DEM_RASTER": {
      "ui_func": QgsProcessingParameterRasterDestination,
      "ui_args": {
        "description": QT_TRANSLATE_NOOP("fetchallja","Elevation raster" )
      },
      "visibleByDefault": False
    },
    "POP": {
      "ui_func": QgsProcessingParameterRasterDestination,
      "ui_args": {
        "description": QT_TRANSLATE_NOOP("fetchallja","Population" )
      },
      "visibleByDefault": True
    }
  }
  
  def __init__(self):
    super().__init__()
    self.UTIL = HrApi(self)
    
  def initAlgorithm(self, config):    
    (extent, target_crs) = self.UTIL.getExtentAndCrsUsingCanvas()
    self.UTIL.setDefaultValue("FETCH_EXTENT", extent)
    self.UTIL.setDefaultValue("TARGET_CRS", target_crs.authid())
    self.UTIL.initParameters()
    
  
  def processAlgorithm(self, parameters, context, feedback):
    
    self.UTIL.registerProcessingParameters(parameters, context, feedback)
    self.CURRENT_PROCESS = self.UTIL.parseCurrentProcess()
    # get target x-y CRS, to apply the buffer and determine the fetch area
    target_crs = self.parameterAsCrs(parameters, "TARGET_CRS", context)
    
    # check whether the target CRS is x-y coordinates
    self.UTIL.checkCrsAsCartesian(target_crs)
    
    # get the extent, using the target CRS
    fetch_extent = self.parameterAsExtent(
      parameters, "FETCH_EXTENT", context, 
      self.parameterAsCrs(parameters, "TARGET_CRS", context)
    )
    
    # get the buffer
    buffer = self.parameterAsDouble(parameters, "BUFFER",context)
    
    # get the fetch area, using the extent and buffer
    fetch_area = QgsReferencedRectangle(
      QgsRectangle(
        fetch_extent.xMinimum() - buffer,
        fetch_extent.yMinimum() - buffer,
        fetch_extent.xMaximum() + buffer,
        fetch_extent.yMaximum() + buffer
      ),
      target_crs
    )
    
    group_name = "geom_" + datetime.datetime.now().strftime("%Y-%m-%d %H:%M:%S")
    
    
    
    fa_flds = self.UTIL.newFieldsWithHistory(
      HrFields.fromQgsFieldList([
        QgsField("extent", QVariant.String)
      ])
    )
    (sink, dest_id_fa) = self.parameterAsSink(
      parameters, "FETCH_AREA", context, 
      fa_flds, QgsWkbTypes.Polygon, target_crs
    )
    
    ft = QgsFeature(fa_flds)
    ft.setGeometry(QgsGeometry.fromRect(fetch_area))
    ft["extent"] = fetch_area.toString()
    ft["history"] = self.CURRENT_PROCESS    
    sink.addFeature(ft)
    
    PostProcessors[dest_id_fa] = HrPostProcessor(
      history = [self.CURRENT_PROCESS], group=group_name,
      color_args={"coloring": "simple_bbox"},
      visibility=self.PARAMETERS["FETCH_AREA"]["visibleByDefault"]
    )
    
    feedback.setProgress(5)      
    
    feedback.pushInfo(self.tr("Fetch geometry of population"))
    
    pop = processing.run(
      "hrisk:fetchpopja",
      {
        "FETCH_EXTENT": self.parameterAsString(parameters, "FETCH_EXTENT", context), # Note that parameterAsExtent is NG because CRS is not included
        "TARGET_CRS": self.parameterAsCrs(parameters, "TARGET_CRS", context),
        "BUFFER": self.parameterAsDouble(parameters, "BUFFER",context),
        "OUTPUT": self.parameterAsOutputLayer(parameters, "POP", context)
      },
      context = context,
      feedback = feedback,
      is_child_algorithm = True
    )["OUTPUT"]
    
    
    PostProcessors[pop] = HrPostProcessor(
      history = [self.CURRENT_PROCESS], group=group_name,
      color_args = {"coloring": "single_band_pseudo_color", "theme": "Purples", "opacity": 0.5},
      set_min_to_zero = True,   
      visibility=self.PARAMETERS["POP"]["visibleByDefault"]
    )
    
    feedback.setProgress(10)    
    
    feedback.pushInfo(self.tr("Fetch geometry of roads"))
    
    road = processing.run(
      "hrisk:fetchroadja",
      {
        "FETCH_EXTENT": self.parameterAsString(parameters, "FETCH_EXTENT", context),# Note that parameterAsExtent is NG because CRS is not included
        "TARGET_CRS": self.parameterAsCrs(parameters, "TARGET_CRS", context),
        "BUFFER": self.parameterAsDouble(parameters, "BUFFER",context),
        "OUTPUT": self.parameterAsOutputLayer(parameters, "ROAD", context)
      },
      context = context,
      feedback = feedback,
      is_child_algorithm = True
    )["OUTPUT"]
    
    
    PostProcessors[road] = HrPostProcessor(
      history = [self.CURRENT_PROCESS], group=group_name,
      visibility=self.PARAMETERS["ROAD"]["visibleByDefault"]
    )
    
    feedback.setProgress(15)    
    
    feedback.pushInfo(self.tr("Fetch geometry of buildings and estimate the population"))
    
    
    bldg = processing.run(
      "hrisk:fetchbuildingja",
      {
        "FETCH_EXTENT": self.parameterAsString(parameters, "FETCH_EXTENT", context),# Note that parameterAsExtent is NG because CRS is not included
        "TARGET_CRS": self.parameterAsCrs(parameters, "TARGET_CRS", context),
        "BUFFER": self.parameterAsDouble(parameters, "BUFFER",context),
        "OUTPUT": "TEMPORARY_OUTPUT"
      },
      context = context,
      feedback = feedback,
      is_child_algorithm = True
    )["OUTPUT"]
    
    bldg_with_pop = processing.run(
      "hrisk:estimatepopulationofbuilding",
      {
        "BUILDING": bldg,
        "POP": pop,
        "OUTPUT": self.parameterAsOutputLayer(parameters, "BUILDING", context)
      },
      context = context,
      feedback = feedback,
      is_child_algorithm = True
    )["OUTPUT"]
    
    
    PostProcessors[bldg_with_pop] = HrPostProcessor(
      history = [self.CURRENT_PROCESS], group=group_name,
      visibility=self.PARAMETERS["BUILDING"]["visibleByDefault"]
    )
    
    feedback.setProgress(25)
    
    feedback.pushInfo(self.tr("Fetch geometry of DEM"))
    
    dem_processing = processing.run(
      "hrisk:fetchdemrasterja",
      {
        "FETCH_EXTENT": self.parameterAsString(parameters, "FETCH_EXTENT", context),
        "TARGET_CRS": self.parameterAsCrs(parameters, "TARGET_CRS", context),
        "BUFFER": self.parameterAsDouble(parameters, "BUFFER",context),
        "OUTPUT": self.parameterAsOutputLayer(parameters, "DEM", context),
        "OUTPUT_RASTER": self.parameterAsOutputLayer(parameters, "DEM_RASTER", context)
      },
      context = context,
      feedback = feedback,
      is_child_algorithm = True
    )
    dem = dem_processing["OUTPUT"]
    dem_raster = dem_processing["OUTPUT_RASTER"]
    
    PostProcessors[dem] = HrPostProcessor(
      history=[self.CURRENT_PROCESS], group = group_name,
      visibility=self.PARAMETERS["DEM"]["visibleByDefault"]
    )
    PostProcessors[dem_raster] = HrPostProcessor(
      history=[self.CURRENT_PROCESS], group = group_name,
      color_args = {"coloring": "single_band_pseudo_color", "theme": "Greens", "opacity": 0.8},
      set_min_to_zero = True,
      visibility=self.PARAMETERS["DEM_RASTER"]["visibleByDefault"]
    )  
    
    
    feedback.setProgress(50)    

    rcv_facade = None
    if self.parameterAsBoolean(parameters, "SET_RECEIVER_FACADE", context):    
      feedback.pushInfo(self.tr("Set receivers at building facade"))
      
      rcv_facade = processing.run(
        "hrisk:receiverfacade",
        {
          "BUILDING": bldg_with_pop,
          "SOURCE": road,
          "FENCE_EXTENT": self.parameterAsString(parameters, "FETCH_EXTENT", context),# Note that parameterAsExtent is NG because CRS is not included
          "DELTA": self.parameterAsDouble(parameters, "DELTA", context),
          "HEIGHT": self.parameterAsDouble(parameters, "HEIGHT", context),
          "OUTPUT": self.parameterAsOutputLayer(parameters, "RECEIVER_FACADE", context)
        },
        context = context,
        feedback = feedback,
        is_child_algorithm = True
      )["OUTPUT"]
      
      
      PostProcessors[rcv_facade] = HrPostProcessor(
        history=[self.CURRENT_PROCESS], group = group_name,
        visibility=self.PARAMETERS["RECEIVER_FACADE"]["visibleByDefault"]
      )
      
    feedback.setProgress(75) 
    
    rcv_delaunay = None
    tri_delaunay = None   
    if self.parameterAsBoolean(parameters, "SET_RECEIVER_DELAUNAYGRID", context):        
      feedback.pushInfo(self.tr("Set receivers of Delaunay grid"))
      
      
      delaunay_processing = processing.run(
        "hrisk:receiverdelaunaygrid",
        {
          "BUILDING": bldg_with_pop,
          "SOURCE": road,
          "FENCE_EXTENT": self.parameterAsString(parameters, "FETCH_EXTENT", context),# Note that parameterAsExtent is NG because CRS is not included
          "MAX_PROP_DIST": self.parameterAsDouble(parameters, "MAX_PROP_DIST", context),
          "ROAD_WIDTH": self.parameterAsDouble(parameters, "ROAD_WIDTH", context),
          "MAX_AREA": self.parameterAsDouble(parameters, "MAX_AREA", context),
          "HEIGHT": self.parameterAsDouble(parameters, "HEIGHT", context),
          "OUTPUT": self.parameterAsOutputLayer(parameters, "RECEIVER_DELAUNAY", context),
          "TRIANGLE": self.parameterAsOutputLayer(parameters, "TRIANGLE_DELAUNAY", context)
        },
        context = context,
        feedback = feedback,
        is_child_algorithm = True
      )
      
      rcv_delaunay = delaunay_processing["OUTPUT"]
      tri_delaunay = delaunay_processing["TRIANGLE"]
    
      PostProcessors[rcv_delaunay] = HrPostProcessor(
        history=[self.CURRENT_PROCESS], group = group_name,
        visibility=self.PARAMETERS["RECEIVER_DELAUNAY"]["visibleByDefault"]
      )
    
      PostProcessors[tri_delaunay] = HrPostProcessor(
        history=[self.CURRENT_PROCESS], group = group_name,
        visibility=self.PARAMETERS["TRIANGLE_DELAUNAY"]["visibleByDefault"]
      )
      
    feedback.setProgress(100)
    
    self.UTIL.registerPostProcessAlgorithm(context, PostProcessors)
    
    return {"FETCH_AREA": dest_id_fa, "BUILDING": bldg_with_pop, "ROAD": road, "DEM": dem, "DEM_RASTER": dem_raster, "POP": pop, "RECEIVER_FACADE": rcv_facade, "RECEIVER_DELAUNAY": rcv_delaunay, "TRIANGLE_DELAUNAY": tri_delaunay}

  def name(self):
    return self.__class__.__name__
    
  def displayName(self):
    return self.tr("All geometries and set receivers")

  def group(self):
    return self.tr('Fetch geometries (Ja)')

  def groupId(self):
    return 'fetchgeomja'

  def createInstance(self):
    return fetchallja()
