
from qgis.PyQt.QtCore import (QT_TRANSLATE_NOOP, QCoreApplication, QVariant)
from qgis.core import (
  QgsProcessing,
  QgsProcessingAlgorithm,
  QgsProcessingParameterString,
  QgsProcessingParameterBoolean,
  QgsProcessingParameterFeatureSource,
  QgsProcessingParameterNumber,
  QgsProcessingParameterFeatureSink,
  QgsFeatureRequest,
  QgsReferencedRectangle,
  QgsVectorLayer,
  QgsProcessingUtils,
  QgsField,
  QgsFields,
  QgsFeature,
  QgsGeometry,
  QgsProcessingParameterRasterLayer,
  QgsProcessingParameterFolderDestination
  )
from qgis import processing
import datetime
import os

from ..algutil.hriskvar import PostProcessors, HriskHome
from ..algutil.hriskutil import HrUtil
from ..algutil.hriskfields import HrFields
from ..algutil.hrisknoisemodelling import NoiseModelling
from ..algutil.hriskpostprocessor import HrPostProcessor

class formatnmresults(QgsProcessingAlgorithm):
  PARAMETERS = {
    "RECEIVERS_LEVEL": {
      "ui_func": QgsProcessingParameterFeatureSource,
      "ui_args":{
        "optional": True,
        "description": QT_TRANSLATE_NOOP("formatnmresults","RECEIVERS_LEVEL layer (output of NM5)"),
        "types": [QgsProcessing.TypeVectorPoint]
      }
    },
    "LDAY_GEOM": {
      "ui_func": QgsProcessingParameterFeatureSource,
      "ui_args":{
        "optional": True,
        "description": QT_TRANSLATE_NOOP("formatnmresults","LDAY_GEOM layer (output of NM4)"),
        "types": [QgsProcessing.TypeVectorPoint]
      }
    },
    "LEVENING_GEOM": {
      "ui_func": QgsProcessingParameterFeatureSource,
      "ui_args":{
        "optional": True,
        "description": QT_TRANSLATE_NOOP("formatnmresults","LEVENING_GEOM layer (output of NM4)"),
        "types": [QgsProcessing.TypeVectorPoint]
      }
    },
    "LNIGHT_GEOM": {
      "ui_func": QgsProcessingParameterFeatureSource,
      "ui_args":{
        "optional": True,
        "description": QT_TRANSLATE_NOOP("formatnmresults","LNIGHT_GEOM layer (output of NM4)"),
        "types": [QgsProcessing.TypeVectorPoint]
      }
    },
    "LDEN_GEOM": {
      "ui_func": QgsProcessingParameterFeatureSource,
      "ui_args":{
        "optional": True,
        "description": QT_TRANSLATE_NOOP("formatnmresults","LDEN_GEOM layer (output of NM4)"),
        "types": [QgsProcessing.TypeVectorPoint]
      }
    },
    "UNIQUE_FIELDS": {
      "advanced": True,
      "ui_func": QgsProcessingParameterString,
      "ui_args": {
        "description": QT_TRANSLATE_NOOP("formatnmresults","Field(s) to identify the geometry"),
        "defaultValue": "IDRECEIVER,IDSOURCE"
      }
    },
    "COLOR_FIELDS": {
      "advanced": True,
      "ui_func": QgsProcessingParameterString,
      "ui_args": {
        "description": QT_TRANSLATE_NOOP("formatnmresults","Field to be used for coloring the results"),
        "defaultValue": "LDEN_LAEQ,LAEQ"
      }
    },
    "GROUP_FIELDS": {
      "advanced": True,
      "ui_func": QgsProcessingParameterString,
      "ui_args": {
        "description": QT_TRANSLATE_NOOP("formatnmresults","Field(s) to group the results. Set none if not needed."),
        "defaultValue": "PERIOD"
      }
    },
    "RECEIVER_WITH_LEVEL": {
      "ui_func": QgsProcessingParameterFeatureSink,
      "ui_args": {
        "description": QT_TRANSLATE_NOOP("formatnmresults","Receiver with level" ),
        "defaultValue": QgsProcessing.TEMPORARY_OUTPUT
      },
      "visibleByDefault": True
    }
  }
  
  def __init__(self):
    super().__init__()
    self.UTIL = HrUtil(self)
  
  def initAlgorithm(self, config):
    self.UTIL.initParameters()

  def processAlgorithm(self, parameters, context, feedback):
    
    self.UTIL.registerProcessingParameters(parameters, context, feedback) 
    self.CURRENT_PROCESS = self.UTIL.parseCurrentProcess(with_nm=True)
    
    nm_results = {
      "receiver_level": self.parameterAsSource(parameters, "RECEIVERS_LEVEL", context),
      "LD": self.parameterAsSource(parameters, "LDAY_GEOM", context),
      "LE": self.parameterAsSource(parameters, "LEVENING_GEOM", context),
      "LN": self.parameterAsSource(parameters, "LNIGHT_GEOM", context),
      "LDEN": self.parameterAsSource(parameters, "LDEN_GEOM", context)
    }
    
    unique_keys = self.parameterAsString(parameters, "UNIQUE_FIELDS", context).split(",")
    group_keys = self.parameterAsString(parameters, "GROUP_FIELDS", context).split(",")
    color_keys = self.parameterAsString(parameters, "COLOR_FIELDS", context).split(",")
    
    nm_results_exist = {
      k: v.materialize(QgsFeatureRequest(), feedback)
      for k, v in nm_results.items() if v is not None
    }
    
    if len(nm_results_exist.keys()) == 0:
      feedback.reportError(self.tr("No results provided."))
      return {}
    
    if nm_results_exist.get("receiver_level") is None:
      feedback.pushInfo(self.tr("Results from NoiseModelling 4 were detected: Grouped using PERIOD field."))
      
      if "PERIOD" not in group_keys:
        group_keys.append("PERIOD")
      
      # make layers with the PERIOD field
      nm4_layers = []
      for key in ["LD", "LE", "LN", "LDEN"]:
        if key in nm_results_exist:
          lyr_with_period = processing.run(
            "native:fieldcalculator",
            {
              "INPUT": nm_results_exist[key],
              "FIELD_NAME": "PERIOD",
              "FIELD_TYPE": 2,
              "FIELD_LENGTH": 10,
              "FIELD_PRECISION": 0,
              "FORMULA": f"'{key[1:]}'",
              "OUTPUT": "TEMPORARY_OUTPUT"
            },
            context=context,
            is_child_algorithm=True
          )["OUTPUT"]
          nm4_layers.append(lyr_with_period)
      
      # merge layers
      nm_results_long_redundant = processing.run(
        "native:mergevectorlayers",
        {
          "LAYERS": nm4_layers,
          "OUTPUT": "TEMPORARY_OUTPUT"
        },
        context=context,
        is_child_algorithm=True
      )["OUTPUT"]

      nm_results_long = processing.run(
        "native:deletecolumn",
        {
          "INPUT": nm_results_long_redundant,
          "COLUMN": ["layer","path"],
          "OUTPUT": "TEMPORARY_OUTPUT"
        },
        context=context,
        is_child_algorithm=True
      )["OUTPUT"]

      nm_results_long = context.getMapLayer(nm_results_long)

    else:
      feedback.pushInfo(self.tr("Results from NoiseModelling 5 were detected."))
      nm_results_long = nm_results_exist["receiver_level"]
      
    
    nm_results_flds = nm_results_long.fields()
    
    unique_keys_ext = [uk for uk in unique_keys if uk in nm_results_flds.names()]
    group_keys_ext = [gk for gk in group_keys if gk in nm_results_flds.names()]

    if len(group_keys_ext) == 0:
      nm_results_wide = nm_results_long
      
    else:
      def field_name_parser(group_values, fld_name):
        return "L" + "_".join([str(gv) for gv in group_values]) + "_" + fld_name

      nm_results_wide_id = self.UTIL.pivotWider(
        nm_results_long,
        unique_keys=unique_keys_ext,
        group_keys=group_keys_ext,
        field_name_parser=field_name_parser
      )
      
      nm_results_wide = context.getMapLayer(nm_results_wide_id)
        
          
    fields_with_values = {
      "HISTORY": {
        "type": QVariant.String,
        "value": self.CURRENT_PROCESS,
        "append": True
      }
    }
    
    dest_id = self.UTIL.outputVectorLayer(
      vector_layer = nm_results_wide,
      param_sink = "RECEIVER_WITH_LEVEL",
      fields_with_values = fields_with_values
    )
    
    color_keys = self.parameterAsString(parameters, "COLOR_FIELDS", context).split(",")
    color_keys_exist = [ck for ck in color_keys if ck in nm_results_wide.fields().names()]
    if len(color_keys_exist) > 0:
      feedback.pushInfo(self.tr("Coloring the results by field: ") + color_keys_exist[0])
      color_args = {
        "coloring": "graduated_symbol",
        "attribute": color_keys_exist[0],
        "theme": "Default"
      }
    else:
      feedback.pushWarning(self.tr("The fields for coloring the results are not available: ") + ", ".join(color_keys))
      color_args = {"coloring": "none"}
    
    PostProcessors[dest_id] = HrPostProcessor(
      history = [self.CURRENT_PROCESS],
      **color_args
    )
    
    self.UTIL.registerPostProcessAlgorithm(context, PostProcessors)
    
    return {"OUTPUT": dest_id}
    

  
  def name(self):
    return self.__class__.__name__
  
  def displayName(self):
    return self.tr("Format NoiseModelling results")

  def group(self):
    return self.tr("Predict sound level")

  def groupId(self):
    return "soundlevel"

  def createInstance(self):
    return formatnmresults()


  # placing here is necessary, when employing pylupdate
  def tr(self, string):
    return QCoreApplication.translate(self.__class__.__name__, string)
