from qgis.PyQt.QtCore import (QT_TRANSLATE_NOOP, QVariant, QCoreApplication)
from qgis.core import (
  QgsProcessing,
  QgsProcessingAlgorithm,
  QgsProcessingParameterMultipleLayers,
  QgsProcessingParameterFeatureSink,
  QgsProcessingParameterBoolean,
  QgsProcessingParameterCrs
  )

from qgis import processing

from ..algutil.hriskutil import HrUtil
from ..algutil.hriskpostprocessor import HrPostProcessor
from ..algutil.hriskvar import PostProcessors

class mergeelevationpoint(QgsProcessingAlgorithm):
  PARAMETERS = {                  
    "INPUT": {
      "ui_func": QgsProcessingParameterMultipleLayers,
      "ui_args":{
        "description": QT_TRANSLATE_NOOP("mergeelevationpoint","Elevation point layers"),
        "layerType": QgsProcessing.TypeVectorPoint
      }
    },
    "TARGET_CRS": {
      "ui_func": QgsProcessingParameterCrs,
      "ui_args": {
        "description": QT_TRANSLATE_NOOP("mergeelevationpoint","Target CRS (Cartesian coordinates)")
      }
    },
    "KEEP_ATTRIBUTES": {
      "ui_func": QgsProcessingParameterBoolean,
      "ui_args":{
        "description" : QT_TRANSLATE_NOOP("mergeelevationpoint","Keep all attributes (or only the required fields)?"),
        "defaultValue": True
      }
    },
    "OUTPUT": {
      "ui_func": QgsProcessingParameterFeatureSink,
      "ui_args": {
        "description": QT_TRANSLATE_NOOP("mergeelevationpoint","Elevation point" )
      }
    }
  }
  
  def __init__(self):
    super().__init__()
    self.UTIL = HrUtil(self)
    
  def initAlgorithm(self, config):
    (_, target_crs) = self.UTIL.getExtentAndCrsUsingCanvas()
    self.UTIL.setDefaultValue("TARGET_CRS", target_crs.authid())
    self.UTIL.initParameters()
  
  
  def processAlgorithm(self, parameters, context, feedback):
    
    self.UTIL.registerProcessingParameters(parameters, context, feedback)
    target_crs = self.parameterAsCrs(parameters, "TARGET_CRS", context)
    
    # check whether the target CRS is x-y coordinates
    self.UTIL.checkCrsAsCartesian(target_crs)
    self.CURRENT_PROCESS = self.UTIL.parseCurrentProcess()

    layers = []
    for lyr in self.parameterAsLayerList(parameters, "INPUT", context):        
      layers.append(
        processing.run(
          "hrisk:initelevationpoint",
          {
            "INPUT": lyr,
            "TARGET_CRS": target_crs,
            "OVERWRITE": False,
            "OUTPUT": "TEMPORARY_OUTPUT"
          },
          context = context,
          is_child_algorithm = True
        )["OUTPUT"]
      )
    
    merged_layer = processing.run(
      "native:mergevectorlayers",
      {
        "LAYERS": layers,
        "OUTPUT": "TEMPORARY_OUTPUT"
      },
      context = context,
      is_child_algorithm = True
    )["OUTPUT"]
        
    merged_layer_fixed = context.getMapLayer(merged_layer)
        
    n = merged_layer_fixed.featureCount()
    fields_with_values = {
      "PK": {"type": QVariant.Int, "value": list(range(1, n+1))},
      "HISTORY": {"type": QVariant.String, "value": self.CURRENT_PROCESS, "append": True}
    }    
    
    dest_id = self.UTIL.outputVectorLayer(
      vector_layer= merged_layer_fixed,
      param_sink = "OUTPUT",
      fields_with_values= fields_with_values
    )
    
    PostProcessors[dest_id] = HrPostProcessor(history = [self.CURRENT_PROCESS])
              
    return {"OUTPUT": dest_id}
  
  def name(self):
    return self.__class__.__name__

  def displayName(self):
    return self.tr("Elevation point")

  def group(self):
    return self.tr("Merge features")

  def groupId(self):
    return "mergefeature"

  def createInstance(self):
    return mergeelevationpoint()

  # placing here is necessary, when employing pylupdate
  def tr(self, string):
    return QCoreApplication.translate(self.__class__.__name__, string)
