
from qgis.PyQt.QtCore import (QT_TRANSLATE_NOOP, QCoreApplication, QVariant)
from qgis.core import (
  QgsProcessing,
  QgsProcessingAlgorithm,
  QgsProcessingParameterFeatureSource,
  QgsProcessingParameterFeatureSink,
  QgsFeatureRequest,
  QgsVectorLayer,
  QgsProcessingParameterFolderDestination,
  QgsProcessingParameterBoolean,
  QgsProcessingParameterString
  )

from qgis import processing
import os

from ..algutil.hriskutil import HrUtil
from ..algutil.hrisknoisemodelling import NoiseModelling
from ..algutil.hriskpostprocessor import HrPostProcessor
from ..algutil.hriskvar import PostProcessors, HriskHome
class initroademissionfromtraffic(QgsProcessingAlgorithm):
  PARAMETERS = {  
    "INPUT": {
      "crs_reference": True, # this parameter is used as CRS reference
      "ui_func": QgsProcessingParameterFeatureSource,
      "ui_args":{
        "description": QT_TRANSLATE_NOOP("initroademissionfromtraffic","Road layer"),
        "types": [QgsProcessing.TypeVectorLine]
      },
      "nm_key":"roadGeomPath",
      "save_layer_get_path": True
    },
    "RUN_CALCULATION": {
      "advanced": True,
      "ui_func": QgsProcessingParameterBoolean,
      "ui_args": {
        "description": QT_TRANSLATE_NOOP("initroademissionfromtraffic","Run the calculation (or just prepare the files)?"),
        "defaultValue": True
      }
    },
    "OVERWRITE": {
      "advanced": True,
      "ui_func": QgsProcessingParameterBoolean,
      "ui_args": {
        "description": QT_TRANSLATE_NOOP("initroademissionfromtraffic","Overwrite results?"),
        "defaultValue": False
      }
    },
    "WORKING_DIRECTORY": {
      "advanced": True,
      "ui_func": QgsProcessingParameterFolderDestination,
      "ui_args": {
        "description": QT_TRANSLATE_NOOP("initroademissionfromtraffic","Working directory"),
        "defaultValue": QgsProcessing.TEMPORARY_OUTPUT,
      }
    },
    "SCRIPT_RUNNER": {
      "advanced": True,
      "ui_func": QgsProcessingParameterString,
      "ui_args": {
        "description": QT_TRANSLATE_NOOP("initroademissionfromtraffic","Script runner name"),
        "defaultValue": "init_roademission_from_traffic",
      }
    },
    "OUTPUT": {
      "ui_func": QgsProcessingParameterFeatureSink,
      "ui_args": {
        "description": QT_TRANSLATE_NOOP("initroademissionfromtraffic","Road" )
      }
    }
  }
  
  NM_OUTPUT = {"LW_ROADS": ["LW_ROADS"]}
  GROOVY_SCRIPT = os.path.join(HriskHome, "groovy", "initroademissionfromtraffic.groovy")
  
  def __init__(self):
    super().__init__()
    self.UTIL = HrUtil(self)
    self.NOISEMODELLING = {}
    
  def initAlgorithm(self, config):
    self.UTIL.initParameters()
    
  def processAlgorithm(self, parameters, context, feedback):   
    
    self.UTIL.registerProcessingParameters(parameters, context, feedback)
    self.CURRENT_PROCESS = self.UTIL.parseCurrentProcess(with_nm=True)
    target_crs = self.UTIL.parseCrs()
    nm_version = self.UTIL.getNoiseModellingVersion()
    
    self.UTIL.registerProcessingParameters(parameters, context, feedback)
    
    
    self.NOISEMODELLING = NoiseModelling(
      crs = target_crs,
      work_dir= self.parameterAsString(parameters, "WORKING_DIRECTORY", context),
      feedback = feedback,
      overwrite= self.parameterAsBool(parameters, "OVERWRITE", context)
    )
    
    
    self.NOISEMODELLING.initArgs(groovy_script = self.GROOVY_SCRIPT)
    
    for ui_key, ui_settings in self.PARAMETERS.items():
      if parameters.get(ui_key) is not None:
        value = parameters[ui_key]
      else:
        value = ui_settings["ui_args"].get("defaultValue", None)
      if value is not None:
        self.NOISEMODELLING.uiParametersToArg(ui_key, ui_settings, value)
    
    if nm_version[0] == "4":
      nm_runner = "org.noisemodelling.runner.Main"
    elif nm_version[0] == "5":
      nm_runner = "org.noise_planet.noisemodelling.runner.Main"
    
    self.NOISEMODELLING.setJavaCommand(
      nm_runner = nm_runner,
      script_name = self.parameterAsString(parameters, "SCRIPT_RUNNER", context),
    )
    
    if not self.parameterAsBool(parameters, "RUN_CALCULATION", context):
      return {"OUTPUT": None}      
    
    results = self.NOISEMODELLING.run()
    feedback.setProgress(100)
    
    # join to the source geom
    
    # first add level values to the receivers
    road_tr_layer = self.parameterAsSource(parameters, "INPUT", context).materialize(QgsFeatureRequest(), feedback)
    road_lw_layer = QgsVectorLayer(results.get(self.NM_OUTPUT["LW_ROADS"][0]))

    lw_in_road_tr_layer = [fld for fld in road_tr_layer.fields().names() if fld[:3].lower() in ["lwd", "lwe","lwn"]]
    
    if len(lw_in_road_tr_layer) > 0:
      road_tr_layer = processing.run(
        "native:deletecolumn",
        {
          "INPUT": road_tr_layer,
          "COLUMN": lw_in_road_tr_layer,
          "OUTPUT": "TEMPORARY_OUTPUT"
        },
        context = context,
        is_child_algorithm = True
      )["OUTPUT"]
      road_tr_layer = context.getMapLayer(road_tr_layer)
      
    road_joined = processing.run(
      "native:joinattributestable",
      {
        "INPUT": road_tr_layer,
        "FIELD": "PK",
        "INPUT_2": road_lw_layer,
        "FIELD_2": "PK",
        "FIELDS_TO_COPY": [fld for fld in road_lw_layer.fields().names() if fld not in ["PK"]],
        "METHOD": 0,
        "DISCARD_NONMATCHING": False,
        "PREFIX": "",
        "OUTPUT": "TEMPORARY_OUTPUT"
      },
      context = context,
      is_child_algorithm = True
    )["OUTPUT"]
    
    road_joined = context.getMapLayer(road_joined)
    
    if road_joined.featureCount() < 1:
      feedback.reportError(self.tr("The result of the NoiseModelling is empty."))
      raise Exception()
    
    fields_with_values = {
      "HISTORY": {
        "type": QVariant.String,
        "value": self.CURRENT_PROCESS,
        "append": True
      }
    }
    
    dest_id = self.UTIL.outputVectorLayer(
      vector_layer= road_joined,
      param_sink = "OUTPUT",
      fields_with_values = fields_with_values
    )
    
    PostProcessors[dest_id] = HrPostProcessor(history = [self.CURRENT_PROCESS])
    
    return {"OUTPUT": dest_id}
    

  def name(self):
    return self.__class__.__name__
  
  def displayName(self):
    return self.tr("Road emission calculated from traffic")

  def group(self):
    return self.tr("Initialize features")

  def groupId(self):
    return "initfeature"

  def createInstance(self):
    return initroademissionfromtraffic()
  
  # placing here is necessary, when employing pylupdate
  def tr(self, string):
    return QCoreApplication.translate(self.__class__.__name__, string)
