from qgis.PyQt.QtCore import (QT_TRANSLATE_NOOP, QCoreApplication, QVariant)
from qgis.core import (
  QgsProcessingAlgorithm,
  QgsProcessingParameterExtent,
  QgsProcessingParameterCrs,
  QgsProcessingParameterDistance,
  QgsProcessingParameterFeatureSink,
  QgsProject,
  QgsRectangle,
  QgsReferencedRectangle
  )
from qgis import processing

from ..algutil.hriskutil import HrUtil

class fetchroadosm(QgsProcessingAlgorithm):
  
  PARAMETERS = {  
    "FETCH_EXTENT": {
      "ui_func": QgsProcessingParameterExtent,
      "ui_args":{
        "description": QT_TRANSLATE_NOOP("fetchroadosm","Extent for fetching data")
      }
    },
    "TARGET_CRS": {
      "ui_func": QgsProcessingParameterCrs,
      "ui_args": {
        "description": QT_TRANSLATE_NOOP("fetchroadosm","Target CRS (Cartesian coordinates)")
      }
    },
    "BUFFER": {
      "ui_func": QgsProcessingParameterDistance,
      "ui_args": {
        "description": QT_TRANSLATE_NOOP("fetchroadosm","Buffer of the fetch area (using Target CRS)"),
        "defaultValue": 0.0,
        "parentParameterName": "TARGET_CRS"
      }
    },
    "OUTPUT": {
      "ui_func": QgsProcessingParameterFeatureSink,
      "ui_args": {
        "description": QT_TRANSLATE_NOOP("fetchroadosm","Road")
      }
    }
  }
  
  FETCH_TILE_FAMILY = "web_mercator"
  FETCH_BASE_URL = "https://lz4.overpass-api.de/api/interpreter"
  FETCH_TIMEOUT = 30
  FETCH_QUERY_KEY = "highway"
  FETCH_QUERY_VALUE = None
  FETCH_ZOOM = None
  FETCH_GEOM_TYPE = "LineString"
  
  def __init__(self):
    super().__init__()
    self.UTIL = HrUtil(self)
  
  def initAlgorithm(self, config):    
    (extent, target_crs) = self.UTIL.getExtentAndCrsUsingCanvas()
    self.UTIL.setDefaultValue("FETCH_EXTENT", extent)
    self.UTIL.setDefaultValue("TARGET_CRS", target_crs.authid())
    self.UTIL.initParameters()
  
  def processAlgorithm(self, parameters, context, feedback):
    
    self.UTIL.registerProcessingParameters(parameters, context, feedback)
    self.CURRENT_PROCESS = self.UTIL.parseCurrentProcess()
    # get target x-y CRS, to apply the buffer and determine the fetch area
    target_crs = self.parameterAsCrs(parameters, "TARGET_CRS", context)
    
    # check whether the target CRS is x-y coordinates
    self.UTIL.checkCrsAsCartesian(target_crs)
    
    # get the extent, using the target CRS
    fetch_extent = self.UTIL.asQgsReferencedRectangle(parameters["FETCH_EXTENT"], target_crs)
    
    # get the buffer
    buffer = self.parameterAsDouble(parameters, "BUFFER",context)
    
    # get the fetch area, using the extent and buffer
    fetch_area = QgsReferencedRectangle(
      QgsRectangle(
        fetch_extent.xMinimum() - buffer,
        fetch_extent.yMinimum() - buffer,
        fetch_extent.xMaximum() + buffer,
        fetch_extent.yMaximum() + buffer
      ),
      target_crs
    )
    
    fetch_area_tr = self.UTIL.transformExtent(fetch_area, target_crs, QgsProject.instance().crs())
    fetch_results = processing.run(
      "quickosm:downloadosmdataextentquery",
      {
        "KEY": self.FETCH_QUERY_KEY,
        "VALUE": self.FETCH_QUERY_VALUE,
        "TIMEOUT": self.FETCH_TIMEOUT,
        "SERVER": self.FETCH_BASE_URL,
        "EXTENT": fetch_area_tr,
        "OUTPUT": "TEMPORARY_OUTPUT"
      },
      # context = context, # not passed, to avoid adding a new layer
      feedback = feedback,
    )["OUTPUT_LINES"]
    
    road_raw = fetch_results
    
      
    road_transformed = processing.run(
    "native:reprojectlayer", 
    {
      "INPUT": road_raw,
      "TARGET_CRS": target_crs,
      "OUTPUT": "TEMPORARY_OUTPUT"
    },
    context = context,
    is_child_algorithm = True
    )["OUTPUT"]      
    
    road_dissolve = self.UTIL.dissolve(road_transformed)

    # Set road traffic fields
    road_final = processing.run(
      "hrisk:initroad",{
        "INPUT": road_dissolve,
        "TARGET_CRS": self.parameterAsCrs(parameters, "TARGET_CRS", context),
        "OVERWRITE": True,
        "OUTPUT": "TEMPORARY_OUTPUT"
      },
      context = context,
      is_child_algorithm = True
    )["OUTPUT"]
    
    road_final_fts = context.getMapLayer(road_final)
    
    fields_with_values = {
      "HISTORY": {
        "type": QVariant.String, 
        "value": self.CURRENT_PROCESS,
        "append": False
        }
    }
          
    dest_id = self.UTIL.outputVectorLayer(
      vector_layer= road_final_fts,
      param_sink = "OUTPUT",
      fields_with_values= fields_with_values
    )
    return {"OUTPUT": dest_id}
  
  def name(self):
    return self.__class__.__name__

  def displayName(self):
    return self.tr("Road centerline (OSM)")

  def group(self):
    return self.tr('Fetch geometries')

  def groupId(self):
    return 'fetchgeometry'

  def createInstance(self):
    return fetchroadosm()

  # placing here is necessary, when employing pylupdate
  def tr(self, string):
    return QCoreApplication.translate(self.__class__.__name__, string)
