from qgis.PyQt.QtCore import (QCoreApplication, QT_TRANSLATE_NOOP, QVariant)
from qgis.core import (
  QgsCoordinateReferenceSystem,
  QgsProcessingAlgorithm,
  QgsProcessingParameterExtent,
  QgsProcessingParameterDistance,
  QgsProcessingParameterCrs, 
  QgsProcessingParameterFeatureSink,
  QgsProcessingParameterRasterDestination,
  QgsProcessingParameterString,
  QgsProcessingParameterBoolean,
  QgsProcessingParameterNumber,
  QgsReferencedRectangle,
  QgsRectangle,
  QgsRasterLayer
  )
from qgis import processing

from ..algutil.hriskutil import HrUtil
from ..algutil.hrisktile import HrQgsTile
from ..algutil.hriskpostprocessor import HrPostProcessor
from ..algutil.hriskvar import PostProcessors

import requests
import re
import os
import shutil

class fetchdemrastersrtm(QgsProcessingAlgorithm):
  
  PARAMETERS = {  
    "FETCH_EXTENT": {
      "ui_func": QgsProcessingParameterExtent,
      "ui_args":{
        "description": QT_TRANSLATE_NOOP("fetchdemrastersrtm","Extent for fetching data")
      }
    },
    "TARGET_CRS": {
      "ui_func": QgsProcessingParameterCrs,
      "ui_args": {
        "description": QT_TRANSLATE_NOOP("fetchdemrastersrtm","Target CRS (Cartesian coordinates)")
      }
    },
    "BUFFER": {
      "ui_func": QgsProcessingParameterDistance,
      "ui_args": {
        "description": QT_TRANSLATE_NOOP("fetchdemrastersrtm","Buffer of the fetch area (using Target CRS)"),
        "defaultValue": 0.0,
        "parentParameterName": "TARGET_CRS"
      }
    },
    "WARP": {
      "ui_func": QgsProcessingParameterBoolean,
      "ui_args": {
        "description": QT_TRANSLATE_NOOP("fetchdemrastersrtm","Warp the DEM raster to the target CRS?"),
        "defaultValue": True
      }
    },
    "CLIP": {
      "ui_func": QgsProcessingParameterBoolean,
      "ui_args": {
        "description": QT_TRANSLATE_NOOP("fetchdemrastersrtm","Clip the DEM raster?"),
        "defaultValue": True
      }
    },
    "VECTORIZE": {
      "ui_func": QgsProcessingParameterBoolean,
      "ui_args": {
        "description": QT_TRANSLATE_NOOP("fetchdemrastersrtm","Output elevation points as vector layer"),
        "defaultValue": False
      }
    },
    "LOAD_CREDENTIAL": {
      "ui_func": QgsProcessingParameterBoolean,
      "ui_args": {
        "description": QT_TRANSLATE_NOOP("fetchdemrastersrtm","Use saved username and password to login SRTM data system"),
        "defaultValue": True
      }
    },
    "USERNAME": {
      "ui_func": QgsProcessingParameterString,
      "ui_args": {
        "description": QT_TRANSLATE_NOOP("fetchdemrastersrtm","Username to login SRTM data system"),
        "optional": True
      }
    },
    "PASSWORD": {
      "ui_func": QgsProcessingParameterString,
      "ui_args": {
        "description": QT_TRANSLATE_NOOP("fetchdemrastersrtm","Password to login SRTM data system"),
        "optional": True
      }
    },
    "SAVE_CREDENTIAL": {
      "ui_func": QgsProcessingParameterBoolean,
      "ui_args": {
        "description": QT_TRANSLATE_NOOP("fetchdemrastersrtm","Save (overwrite) username and password to login SRTM data system"),
        "defaultValue": False
      }
    },
    "MAX_DOWNLOAD": {
      "ui_func": QgsProcessingParameterNumber,
      "ui_args": {
        "description": QT_TRANSLATE_NOOP("fetchdemrastersrtm","Maximum number of download"),
        "type": QgsProcessingParameterNumber.Integer,
        "defaultValue": 4
      }
    },
    "OUTPUT": {
      "ui_func": QgsProcessingParameterFeatureSink,
      "ui_args": {
        "description": QT_TRANSLATE_NOOP("fetchdemrastersrtm","Elevation points")
      }
    },
    "OUTPUT_RASTER": {
      "ui_func": QgsProcessingParameterRasterDestination,
      "ui_args": {
        "description": QT_TRANSLATE_NOOP("fetchdemrastersrtm","Elevation raster" )
      }
    }
  }  
  
  FETCH_BASE_URL = "https://e4ftl01.cr.usgs.gov/DP109/SRTM/SRTMGL1.003/2000.02.11/{lat_str}{long_str}.SRTMGL1.hgt.zip"
  FETCH_LOGIN_URL = "https://urs.earthdata.nasa.gov"
  FETCH_CRS = QgsCoordinateReferenceSystem("EPSG:4326")
  FETCH_TILE_PARAMETERS = {"xunit": 1.0, "yunit": 1.0, "xorig": 0.0, "yorig": 0.0}
    
  def __init__(self):
    super().__init__()
    self.UTIL = HrUtil(self)
  
  # initialize of the algorithm  
  def initAlgorithm(self, config):    
    (extent, target_crs) = self.UTIL.getExtentAndCrsUsingCanvas()
    self.UTIL.setDefaultValue("FETCH_EXTENT", extent)
    self.UTIL.setDefaultValue("TARGET_CRS", target_crs.authid())
    self.UTIL.initParameters()

  # execution of the algorithm
  def processAlgorithm(self, parameters, context, feedback): 
    
    self.UTIL.registerProcessingParameters(parameters, context, feedback)
    self.CURRENT_PROCESS = self.UTIL.parseCurrentProcess()
    # get target x-y CRS, to apply the buffer and determine the fetch area
    target_crs = self.parameterAsCrs(parameters, "TARGET_CRS", context)
    
    # check whether the target CRS is x-y coordinates
    self.UTIL.checkCrsAsCartesian(target_crs)
    
    # get the extent, using the target CRS
    fetch_extent = self.UTIL.asQgsReferencedRectangle(parameters["FETCH_EXTENT"], target_crs)
    
    # get the buffer
    buffer = self.parameterAsDouble(parameters, "BUFFER",context)
    
    # get the fetch area, using the extent and buffer
    fetch_area = QgsReferencedRectangle(
      QgsRectangle(
        fetch_extent.xMinimum() - buffer,
        fetch_extent.yMinimum() - buffer,
        fetch_extent.xMaximum() + buffer,
        fetch_extent.yMaximum() + buffer
      ),
      target_crs
    )
    
    tile = HrQgsTile(
      crs = self.FETCH_CRS,
      **self.FETCH_TILE_PARAMETERS
    )
    fetch_area_tile = self.UTIL.transformExtent(fetch_area, target_crs, self.FETCH_CRS)
    long_lat_list = tile.cellXyIdx(fetch_area_tile)
    
    
    fetch_args = {}
    for i, (long, lat) in enumerate(long_lat_list):
      long_str = f"{long:+04}".replace("+","E").replace("-","W")
      lat_str  = f"{lat:+03}".replace("+","N").replace("-","S")
      fetch_args[f"{i+1}/{len(long_lat_list)}"] = {
        "url": self.FETCH_BASE_URL.format(lat_str = lat_str, long_str = long_str)
      }
          
    if len(fetch_args) > self.parameterAsInt(parameters, "MAX_DOWNLOAD", context):
      feedback.reportError(self.tr("Too many downloads are required: ") + str(len(fetch_args)))
      raise Exception(self.tr("Too many downloads are required: ") + str(len(fetch_args)))
    
    # set login data and login
    srtm_session = requests.Session()
    srtm_token = re.search(
      r'<form[^>]*id="login".*?>[\s\S]*?<input[^>]*name="authenticity_token"[^>]*value="([^"]*)"[^>]*>', 
      srtm_session.get(self.FETCH_LOGIN_URL).text
    ).group(1)
    
    if self.parameterAsBool(parameters, "LOAD_CREDENTIAL", context):
      try:
        with open(os.path.join(os.path.dirname(__file__), "srtm_credential"), "r") as f:
          line1 = f.readline().strip()
          line2 = f.readline().strip()
          username = re.search(r"username: (.*)", line1).group(1)
          password = re.search(r"password: (.*)", line2).group(1)
      except:
        raise Exception(self.tr("Failed to load the credential"))
    else:
      username = self.parameterAsString(parameters, "USERNAME", context)
      password = self.parameterAsString(parameters, "PASSWORD", context)
    
    if self.parameterAsBool(parameters, "SAVE_CREDENTIAL", context):
      with open(os.path.join(os.path.dirname(__file__), "srtm_credential"), "w") as f:
        f.write("username: " + username + "\n")
        f.write("password: " + password + "\n")
    
    srtm_session.request(
      method = "POST",
      url = self.FETCH_LOGIN_URL + "/login",
      data = {
        "utf8": "✓",
        "authenticity_token": srtm_token,
        "username": username,
        "password": password
      }
    )
    
    fetch_results = self.UTIL.downloadFilesConcurrently(
      args=fetch_args, session=srtm_session,
      parallel=False
    )
    
    files_unpack = []
    for file in fetch_results.values():
      _, extracted_files = self.UTIL.extractArchive(file, extension=".zip", pattern = "\\.hgt$")
      files_unpack += extracted_files
    
    ras_path = self.parameterAsOutputLayer(parameters, "OUTPUT_RASTER", context)
    ras_path_tmp = os.path.join(os.path.dirname(ras_path), "tmp.tif")
    
    processing.run(
      "gdal:merge",{
        "INPUT": files_unpack,
        "OUTPUT": ras_path
      },
      is_child_algorithm = True # do not pass the context, because this output is set as the final output
    )
    
    
    # warp the raster to the target CRS
    if self.parameterAsBoolean(parameters, "WARP", context):
      feedback.pushInfo("Raster is being warped...")
      ras = QgsRasterLayer(ras_path)
      nx = ras.width()
      ny = ras.height()
      ras_extent = self.UTIL.transformExtent(ras.extent(), ras.crs(), target_crs)
      lx = ras_extent.xMaximum() - ras_extent.xMinimum() 
      ly = ras_extent.yMaximum() - ras_extent.yMinimum()
      target_res = round((lx + ly)/(nx + ny),3)
      processing.run(
        "gdal:warpreproject",
        {
          "INPUT": ras_path,
          "RESAMPLING": 1,
          "SOURCE_CRS": self.FETCH_CRS,
          "TARGET_CRS": target_crs.authid(),
          "TARGET_RESOLUTION": target_res,
          "EXTRA" : "-srcnodata nan",
          "NODATA": -9999,
          "OUTPUT": ras_path_tmp
        },
        # context = context, # do not use context
        is_child_algorithm = True
      )
      shutil.move(ras_path_tmp, ras_path)
      
    else:
      fetch_area = self.UTIL.transformExtent(fetch_area, target_crs, self.FETCH_CRS)
    
    clip = self.parameterAsBoolean(parameters, "CLIP", context)
    vectorize = self.parameterAsBoolean(parameters, "VECTORIZE", context)
    if clip or vectorize:
      feedback.pushInfo("Raster is being clipped...")
      proj_win = self.UTIL.transformExtent(fetch_area, target_crs, QgsRasterLayer(ras_path).crs())
      processing.run(
        "gdal:cliprasterbyextent",
        {
          "INPUT": ras_path,
          "PROJWIN": proj_win,
          "OUTPUT": ras_path_tmp
        },
        is_child_algorithm = True
      )
    
    if clip:
      shutil.copy(ras_path_tmp, ras_path)

    
    
    if vectorize:
      feedback.pushInfo("Raster is being vectorized...")
    
      # create vector layer from the raster
      elev_raw = processing.run(
        "native:pixelstopoints",{
          "INPUT_RASTER": ras_path_tmp,
          "RASTER_BAND": 1,
          "FIELD_NAME": "alti",
          "OUTPUT": "TEMPORARY_OUTPUT"
        },
        context = context,
        is_child_algorithm = True
      )["OUTPUT"]
      
      # CRS transform    
      elev_tr = processing.run(
        "native:reprojectlayer", 
        {
          "INPUT": elev_raw,
          "TARGET_CRS": target_crs,
          "OUTPUT": "TEMPORARY_OUTPUT"
        },
        context = context,
        is_child_algorithm = True
      )["OUTPUT"]      
              
      # substitute self constant with the fetched vector layer
      elev_final = processing.run(
        "hrisk:initelevationpoint",{
          "INPUT": elev_tr,
          "FIELD_USE_AS_HEIGHT": "alti",
          "TARGET_CRS": target_crs,
          "OVERWRITE": True,
          "OUTPUT": "TEMPORARY_OUTPUT"
        },
        context = context,
        is_child_algorithm = True
      )["OUTPUT"]
      
      elev_final = context.getMapLayer(elev_final)
      
      fields_with_values = {
        "HISTORY": {
          "type": QVariant.String, 
          "value": self.CURRENT_PROCESS,
          "append": False
          }
      }
      
      dest_id = self.UTIL.outputVectorLayer(
        vector_layer= elev_final,
        param_sink = "OUTPUT",
        fields_with_values= fields_with_values
      )
      
      PostProcessors[dest_id] = HrPostProcessor(
        history=[self.CURRENT_PROCESS]
      )
      
    else:
      dest_id = None
    
    
    # register history and postprocessor for the raster output
    PostProcessors[ras_path] = HrPostProcessor(
      history=[self.CURRENT_PROCESS], 
      color_args = {"coloring": "single_band_pseudo_color", "theme": "Greens", "opacity": 0.8},
      set_min_to_zero = False
    )  
    
    self.UTIL.registerPostProcessAlgorithm(context, PostProcessors)  
    
    return {"OUTPUT": dest_id, "OUTPUT_RASTER": ras_path}   
    
  def name(self):
    return self.__class__.__name__
  
  def displayName(self):
    return self.tr("Elevation points (SRTM)")

  def group(self):
    return self.tr('Fetch geometries')

  def groupId(self):
    return 'fetchgeometry'

  def createInstance(self):
    return fetchdemrastersrtm()

  # placing here is necessary, when employing pylupdate
  def tr(self, string):
    return QCoreApplication.translate(self.__class__.__name__, string)
