from qgis.PyQt.QtCore import (QT_TRANSLATE_NOOP, QCoreApplication)

from qgis.core import (
  QgsProcessingParameterEnum,
  QgsProcessingAlgorithm,
  QgsProject,
  QgsProcessingUtils,
  QgsProcessingParameterBoolean,
  QgsProcessingParameterString,
  QgsVectorLayer
  )

from qgis import processing

import os
import json
import datetime
import hashlib
import shutil
import uuid

from ..algutil.hriskutil import HrUtil
from ..algutil.hriskpostprocessor import HrPostProcessor
from ..algutil.hriskvar import PostProcessors, HriskHome

class testalgorithms(QgsProcessingAlgorithm):
  

  PARAMETERS = {                  
    "CASE": {
      "ui_func": QgsProcessingParameterEnum,
      "ui_args": {
        "description": QT_TRANSLATE_NOOP("testalgorithms","Test case" ),
        "options": [],
        "allowMultiple": True,
        "defaultValue": 0
      }
    },
    "ADD_OUTPUTS": {
      "advanced": True,
      "ui_func": QgsProcessingParameterBoolean,
      "ui_args": {
        "description": QT_TRANSLATE_NOOP("testalgorithms","Add output layers to the current project?" ),
        "defaultValue": False
      }
    },
    "UPDATE_HASH": {
      "advanced": True,
      "ui_func": QgsProcessingParameterBoolean,
      "ui_args": {
        "description": QT_TRANSLATE_NOOP("testalgorithms","Update SHA256 hash?" ),
        "defaultValue": False
      }
    },
    "PASSWORD_UPDATE": {
      "advanced": True,
      "ui_func": QgsProcessingParameterString,
      "ui_args": {
        "description": QT_TRANSLATE_NOOP("testalgorithms","Password for updating the hash" ),
        "defaultValue": "password"
      }
    }
  }
  
  SCENARIOS = []
  DELIMITER = "::::::::::::::"
  
  SCENARIOS_JSON = os.path.join(os.path.dirname(__file__),"test-scenarios.json")
  
  def __init__(self):
    super().__init__()
    self.UTIL = HrUtil(self)
    
  def insertDelimiter(self, feedback):
    feedback.pushInfo("")
    feedback.pushInfo("::::::::::::::::::")
    feedback.pushInfo("")
  
  def initAlgorithm(self, config):
    with open(self.SCENARIOS_JSON, "r") as f:
      self.SCENARIOS = json.load(f)
    self.PARAMETERS["CASE"]["ui_args"]["options"] = [item["description"] for item in self.SCENARIOS]
    self.UTIL.initParameters()
  
  def parseArgs(self, arg, tmp_dir:str):
    if isinstance(arg, list):
      return [self.parseArgs(x, tmp_dir) for x in arg]
    elif isinstance(arg, str) and "{path}" in arg:
      return os.path.join(
        HriskHome,
        os.path.normpath(arg.replace("{path}", "", 1)) 
      )
    elif isinstance(arg, str) and "{tmp}" in arg:
      return os.path.join(
        tmp_dir, os.path.normpath(arg.replace("{tmp}", "", 1)) 
      )
    else:
      return arg
  
  def runTestAlg(self, alg:str, args:dict, targets:list, tmp_dir:str, feedback, context):
    args_parsed = {}
    for arg_name, arg_value in args.items():
      args_parsed[arg_name] = self.parseArgs(arg_value, tmp_dir)
    
    processing.run(
      alg, args_parsed, feedback=feedback, context=context
    )
    
    hash_sha256 = {}
    for tg in targets:
      
      # delete HISTORY attribute
      test_dir = os.path.join(os.path.dirname(args_parsed[tg]), str(uuid.uuid4())[:6])
      if not os.path.exists(test_dir):
        os.mkdir(test_dir)
        
      tmp_path = os.path.join(test_dir, os.path.basename(args_parsed[tg]))
      try:
        vl = QgsVectorLayer(args_parsed[tg])
        column_delete = []
        if "HISTORY" in vl.fields().names():
          column_delete.append("HISTORY")
        if "layer" in vl.fields().names():
          column_delete.append("layer")
        if "path" in vl.fields().names():
          column_delete.append("path")
          
        if len(column_delete) > 0:
          processing.run(
            "native:deletecolumn",
            {
              "INPUT": args_parsed[tg],
              "COLUMN": column_delete,
              "OUTPUT": tmp_path
            },
            context = context
          )
        else:
          shutil.copy(args_parsed[tg], tmp_path)
      except:
        shutil.copy(args_parsed[tg], tmp_path)
        feedback.pushInfo(f"Failed to delete HISTORY attribute in {tg}")
            
      with open(tmp_path, "rb") as f:
        hash_tg = hashlib.sha256(f.read()).hexdigest()
        hash_sha256.update({tg: hash_tg})
        feedback.pushInfo(f"{tg} SHA256: " + hash_tg)
            
    return args_parsed, hash_sha256
    
  def processAlgorithm(self, parameters, context, feedback):
    
    self.UTIL.registerProcessingParameters(parameters, context, feedback)  
    pswd = self.parameterAsString(parameters, "PASSWORD_UPDATE", context)
    if pswd == "jtagu":
      is_update_hash = self.parameterAsBool(parameters, "UPDATE_HASH", context)
    else:
      is_update_hash = False
      
    hashtest_summary = []
        
    for sc_idx in self.parameterAsEnums(parameters, "CASE", context):
      scenario = self.SCENARIOS[sc_idx]
     
      tmp_dir = os.path.normpath(os.path.dirname(QgsProcessingUtils.generateTempFilename("")))
      if not os.path.exists(tmp_dir):
        os.makedirs(tmp_dir)
        
      for sc_key, sc_val in scenario.items():
        if isinstance(sc_val, dict):
          try:
            if sc_val["type"] == "test":
              
              feedback.pushInfo("")
              feedback.pushInfo("-----------------------")
              feedback.pushInfo("[[ TEST ]]: " + sc_val["description"])
              feedback.pushInfo("-----------------------")
              
              test_result, test_hash = self.runTestAlg(
                sc_val["alg"], sc_val["args"], sc_val["targets"], tmp_dir, 
                feedback, context
              )
              
              for tg in sc_val["targets"]:
                if not is_update_hash:
                  if test_hash[tg] == sc_val["sha256"][tg]:
                    feedback.pushInfo("")
                    feedback.pushInfo(f"Hash check PASSED: {tg} in {sc_val['description']}")
                    feedback.pushInfo(f"SHA256 hash: {test_hash[tg]}")
                    feedback.pushInfo("")
                    hashtest_summary.append(f"Hash check PASSED: {test_hash[tg]} in {sc_val['description']}")
                  else:
                    feedback.pushInfo("")
                    feedback.pushWarning(f"Hash check FAILED: {tg} in {sc_val['description']}")
                    feedback.pushInfo(f"SHA256 hash: {test_hash[tg]} (test) vs {sc_val['sha256'][tg]} (reference)]")
                    feedback.pushInfo(f"The data/algorithm may be updated. Check the time stamp of the reference data: {sc_val['timestamp']}")
                    feedback.pushInfo("In the case of testing fetch algorithms, the data may be updated.")
                    hashtest_summary.append(f"Hash check FAILED: {tg} in {sc_val['description']}")

                if self.parameterAsBool(parameters, "ADD_OUTPUTS", context):
                  context.addLayerToLoadOnCompletion(
                    test_result[tg], context.LayerDetails(tg, QgsProject.instance(), "")
                  )
                
            if is_update_hash:
              scenario[sc_key].update({"sha256": test_hash})
              scenario[sc_key].update({"timestamp": str(datetime.datetime.now())})
                
          except Exception as e:
            feedback.reportError(f"Error in testing {sc_key}.")
            raise(e)
        
      if not is_update_hash:
        feedback.pushInfo("")
        feedback.pushInfo("-----------------------")
        feedback.pushInfo("[[ TEST SUMMARY ]]")
        feedback.pushInfo("-----------------------")
        for hashtest in hashtest_summary:
          feedback.pushInfo(hashtest)
      
        feedback.pushInfo("")
    
    if is_update_hash:
      shutil.copyfile(self.SCENARIOS_JSON, self.SCENARIOS_JSON + ".bak")
      with open(self.SCENARIOS_JSON, "w") as f:
        json.dump(self.SCENARIOS, f, indent=2)
      feedback.pushInfo(f"Updated: {self.SCENARIOS_JSON}")
      feedback.pushInfo(f"Backup: {self.SCENARIOS_JSON}.bak")
      
    self.UTIL.registerPostProcessAlgorithm(context, PostProcessors)
    
    return {}


  def name(self):
    return self.__class__.__name__
  
  def displayName(self):
    return self.tr("Test algorithms")

  def group(self):
    return self.tr("Configurations")

  def groupId(self):
    return "config"

  def createInstance(self):
    return testalgorithms()

  # placing here is necessary, when employing pylupdate
  def tr(self, string):
    return QCoreApplication.translate(self.__class__.__name__, string)
