from qgis.PyQt.QtCore import (QT_TRANSLATE_NOOP, QCoreApplication)

from qgis.core import (
  QgsProcessingAlgorithm,
  QgsProcessingParameterBoolean,
  QgsProcessingParameterString
  )

import os
import platform
import subprocess
import shutil

from ..algutil.hriskutil import HrUtil

class installcomponents(QgsProcessingAlgorithm):
  

  PARAMETERS = {                  
    "INSTALL_NM": {
      "ui_func": QgsProcessingParameterBoolean,
      "ui_args": {
        "description": QT_TRANSLATE_NOOP("installcomponents","Download and Install NoiseModelling" ),
        "defaultValue": True
      }
    },
    "INSTALL_JAVA": {
      "ui_func": QgsProcessingParameterBoolean,
      "ui_args": {
        "description": QT_TRANSLATE_NOOP("installcomponents","Download and Install Java environment required for NoiseModelling" ),
        "defaultValue": True
      }
    },
    "URL_NM": {
      "advanced": True,
      "ui_func": QgsProcessingParameterString,
      "ui_args": {
        "description": QT_TRANSLATE_NOOP("installcomponents","Path of NoiseModelling" ),
        "defaultValue": "https://github.com/Universite-Gustave-Eiffel/NoiseModelling/releases/download/v5.0.1/NoiseModelling_without_gui-5.0.1.zip"
        # "defaultValue": "https://github.com/Universite-Gustave-Eiffel/NoiseModelling/releases/download/v4.0.5/NoiseModelling_without_gui.zip"
      }
    },
    "URL_JAVA": {
      "advanced": True,
      "ui_func": QgsProcessingParameterString,
      "ui_args": {
        "description": QT_TRANSLATE_NOOP("installcomponents","Path of Java environment" ),
        "defaultValue": ""
      }
    },
    "PATH_NM": {
      "advanced": True,
      "ui_func": QgsProcessingParameterString,
      "ui_args": {
        "description": QT_TRANSLATE_NOOP("installcomponents","Path of NoiseModelling" ),
        "defaultValue": ""
      }
    },
    "PATH_JAVA": {
      "advanced": True,
      "ui_func": QgsProcessingParameterString,
      "ui_args": {
        "description": QT_TRANSLATE_NOOP("installcomponents","Path of Java environment" ),
        "defaultValue": ""
      }
    }
  }
  
  NM_TARGET = "wps_scripts"
  JAVA_TARGET = "java.exe" if platform.system() == 'Windows' else "java"
    
  def __init__(
    self, 
    # nm_version = "4.0.5", 
    nm_version = "5.0.1", 
    java_version = "11.0.2"
    ):
    super().__init__()
    
    pf = platform.system()
    self.UTIL = HrUtil(self)
        
    if pf == 'Windows':
      self.PARAMETERS["URL_JAVA"]["ui_args"]["defaultValue"]  = "https://download.java.net/java/GA/jdk11/9/GPL/openjdk-11.0.2_windows-x64_bin.zip"
      self.PARAMETERS["PATH_NM"]["ui_args"]["defaultValue"]  = os.path.join(os.path.normpath(os.environ['APPDATA']), "H-RISK", "NoiseModelling", nm_version)
      self.PARAMETERS["PATH_JAVA"]["ui_args"]["defaultValue"] = os.path.join(os.path.normpath(os.environ['APPDATA']), "H-RISK", "Java", java_version)
    elif pf == 'Darwin':
      self.PARAMETERS["URL_JAVA"]["ui_args"]["defaultValue"]  = "https://download.java.net/java/GA/jdk11/9/GPL/openjdk-11.0.2_osx-x64_bin.tar.gz"
      self.PARAMETERS["PATH_NM"]["ui_args"]["defaultValue"]  = os.path.join(os.path.expanduser("~"), "Applications", "H-RISK", "NoiseModelling", nm_version)
      self.PARAMETERS["PATH_JAVA"]["ui_args"]["defaultValue"] = os.path.join(os.path.expanduser("~"), "Applications", "H-RISK", "Java", java_version)
    elif pf == 'Linux':
      self.PARAMETERS["URL_JAVA"]["ui_args"]["defaultValue"]  = "https://download.java.net/java/GA/jdk11/9/GPL/openjdk-11.0.2_linux-x64_bin.tar.gz"
      self.PARAMETERS["PATH_NM"]["ui_args"]["defaultValue"]  = os.path.join(os.path.expanduser("~"), "H-RISK", "NoiseModelling", nm_version)
      self.PARAMETERS["PATH_JAVA"]["ui_args"]["defaultValue"] = os.path.join(os.path.expanduser("~"), "H-RISK", "Java", java_version)
  
  def initAlgorithm(self, config):
    self.UTIL.initParameters()
  
  
  def setEnvVariable(self, key: str, value: str) -> None:
    try:
      if platform.system() == 'Windows':  
        subprocess.call(["setx", key, value])
      else:  # Handle Linux/macOS
        # Detect the current shell
        shell = os.environ.get('SHELL', '')
        if 'zsh' in shell:  # For Zsh
          config_file = '~/.zshrc'
        elif 'bash' in shell:  # For Bash
          config_file = '~/.bashrc'
        else:  # Fallback to Bash
          config_file = '~/.bashrc'

        # Expand the tilde (~) in the path
        config_file = os.path.expanduser(config_file)

        # Ensuring the config file exists else we create one.
        if not os.path.exists(config_file):
          with open(config_file, 'w') as f:
            pass  # Create an empty file if it doesn't exist

        # Removing existing export statement for the variable
        if platform.system() == 'Darwin':  # macOS
          subprocess.run(f"sed -i '' '/^export {key}=/d' {config_file}", shell=True, check=True)
        else:  # Linux
          subprocess.run(f"sed -i '/^export {key}=/d' {config_file}", shell=True, check=True)

        # Append the new export statement
        subprocess.run(f'echo \'export {key}="{value}"\' >> {config_file}', shell=True, check=True)

    except subprocess.CalledProcessError as e:
      raise Exception(f"Error running command: {e.cmd}, Exit Code: {e.returncode}, Output: {e.output}") from e
    except Exception as e:
      raise Exception(f"Error(s) occurred while setting environment variables: {e}")
    
  
  def downloadAndExtractComponents(
    self, parameters:dict, context, feedback, 
    component_name:str, component_url:str, component_dir:str, component_target:str, component_target_depth:int = 2,
    component_env_key:str = None):
    
    try:
      feedback.pushInfo(f"Downloading {component_name}...")
      
      download_path = self.UTIL.downloadFile(url = component_url, feedback_progress=True)
      
      # extract NoiseModelling and get paths
      feedback.pushInfo(f"Extracting {component_name}...")
      
        
      extracted_dirs, extracted_files = self.UTIL.extractArchive(download_path)
      
      # find a file of target
      component_core_dir = None
      for path in extracted_files:
        if component_target in path:
          component_core_dir = path
          for _ in range(component_target_depth):
            component_core_dir = os.path.dirname(component_core_dir)
          break
      if component_core_dir is None:
        feedback.reportError(self.tr(f"The target file was not found in the extracted files."), fatalError=True)
        raise Exception(self.tr(f"The target file was not found in the extracted files."))
      
      # initialize the component_dir
      if not os.path.exists(component_dir):
        os.makedirs(component_dir)
      else:
        shutil.rmtree(component_dir, ignore_errors=True)
        os.makedirs(component_dir)
      
      # move extracted files to the target folder
      for path in extracted_dirs + extracted_files:
        if os.path.dirname(path) == component_core_dir:
          src = path
          dst = os.path.join(component_dir, os.path.basename(path))
          shutil.move(src, dst)
        
      # set a environment variable of NOISEMODELLING_HOME
      feedback.pushInfo(f"Setting an environment variable of {component_name}...")
      
      if component_env_key is not None:
        self.setEnvVariable(component_env_key, component_dir)
        feedback.pushInfo(f"{component_env_key}: " + component_dir)
      
      return {component_env_key: component_dir}
        
    except Exception as e:
      # if any error occurred
      feedback.reportError(f"Error(s) occurred at installing {component_name}.", fatalError=True)
      raise e
    
  
  def processAlgorithm(self, parameters, context, feedback):
    self.UTIL.registerProcessingParameters(parameters, context, feedback)
    
    prc_output = {}
    
    # if installing NoiseModelling
    if self.parameterAsBoolean(parameters, "INSTALL_NM", context):
      prc_output = prc_output | self.downloadAndExtractComponents(
        parameters, context, feedback, 
        component_name = "NoiseModelling", 
        component_url = self.parameterAsString(parameters, "URL_NM", context), 
        component_dir = self.parameterAsString(parameters, "PATH_NM", context),
        component_target = self.NM_TARGET,
        component_target_depth = 2, component_env_key = "NOISEMODELLING_HOME")
      
    
    # if installing Java
    if self.parameterAsBoolean(parameters, "INSTALL_Java", context):
      prc_output = prc_output | self.downloadAndExtractComponents(
        parameters, context, feedback, 
        component_name = "Java environment", 
        component_url = self.parameterAsString(parameters, "URL_JAVA", context), 
        component_dir = self.parameterAsString(parameters, "PATH_JAVA", context),
        component_target = self.JAVA_TARGET,
        component_target_depth = 2, component_env_key = "JAVA_FOR_NOISEMODELLING")
    
    if len(prc_output.values()) > 0:        
      feedback.pushInfo(self.tr("Restart QGIS to apply changes of Environments."))
        
    return {k: os.path.normpath(v).replace(os.path.sep, "/") for k, v in prc_output.items()}
  
  
  def createInstance(self):
    return installcomponents()


  def name(self):
    return self.__class__.__name__

  def displayName(self):
    return self.tr("Install required components")

  def group(self):
    return self.tr("Configurations")

  def groupId(self):
    return "config"

  def createInstance(self):
    return installcomponents()

  # placing here is necessary, when employing pylupdate
  def tr(self, string):
    return QCoreApplication.translate(self.__class__.__name__, string)
