/**
 * Isosurface Creation from Noise Calculation Results
 *
 * This Groovy script creates sound level contour lines (isosurfaces) from noise 
 * calculation results. It processes result tables and triangle mesh data to generate
 * smoothed isosurface polygons at specified noise levels.
 *
 * Dependencies:
 * - H2GIS database for spatial data processing
 * - NoiseModelling framework for acoustic calculations
 * - SLF4J and Log4j for logging
 * 
 * @author Junta Tagusari
 * @version 1.0
 */
 
import org.h2gis.api.ProgressVisitor;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import java.sql.Connection;
import java.nio.file.Path;
import java.nio.file.Paths;
import org.apache.log4j.PropertyConfigurator;
import java.io.FileWriter;
import java.io.PrintWriter;

title = 'Create Isosurface'
description = 'Create isosurface (sound-level contour) using results'

inputs = [
  resultGeomPath:[
    name : "Path of the result file",
    title : "Path of the result file",
    description : "Path of the result file",
    type : String.class
  ],
  resultTableField:[
    name : "Field of result table",
    title : "Field of result table",
    description : "Field of result table",
    type : String.class
  ],
  triangleGeomPath:[
    name : "Path of the triangle file",
    title : "Path of the triangle file",
    description : "Path of the triangle file, of which file name must be 'triangles'",
    type : String.class
  ],
  isoClass         : [
    name: 'Iso levels in dB',
    title: 'Iso levels in dB',
    description: 'Separation of sound levels for isosurfaces. ' +
            '</br> </br> <b> Default value : 35.0,40.0,45.0,50.0,55.0,60.0,65.0,70.0,75.0,80.0,200.0 </b>',
    min: 0, max: 1,
    type: String.class
  ],
  smoothCoefficient: [
    name: 'Polygon smoothing coefficient',
    title: 'Polygon smoothing coefficient',
    description: 'This coefficient (Bezier curve coefficient) will smooth generated isosurfaces. If equal to 0, it disables the smoothing step.' +
            '</br> </br> <b> Default value : 1.0 </b>',
    min: 0, max: 1,
    type: Double.class
  ],
  inputSRID: [
    name: 'Projection identifier',
    title: 'Projection identifier',
    description: 'Original projection identifier (also called SRID) of your table. It should be an EPSG code, a integer with 4 or 5 digits (ex: 3857 is Web Mercator projection). ' +
            '</br>  All coordinates will be projected from the specified EPSG to WGS84 coordinates. ' +
            '</br> This entry is optional because many formats already include the projection and you can also import files without geometry attributes.</br> ' +
            '</br> <b> Default value : 4326 </b> ',
    type: Integer.class,
    min: 0, max: 1
  ],
  noiseModellingHome : [
    name: "Path of NOISEMODELLING_HOME",
    title: "Path of NOISEMODELLING_HOME",
    description: "Path of NOISEMODELLING_HOME",
    type : String.class
  ],
  exportDir : [
    name: "Path of export directory",
    title: "Path of export directory",
    description: "Path of export directory",
    min        : 0, max: 1,
    type : String.class
  ]
]

outputs = [
  result: [
    name: 'Result output string', 
    title: 'Result output string', 
    description: 'This type of result does not allow the blocks to be linked together.', 
    type: String.class
  ]
]

/**
 * Main execution function for isosurface creation
 * This function orchestrates the isosurface generation workflow:
 * 1. Clear existing database tables
 * 2. Import result and triangle mesh data
 * 3. Generate isosurface contours at specified noise levels
 * 4. Export results to files
 * 
 * @param connection H2GIS database connection
 * @param input Map containing input parameters (resultGeomPath, triangleGeomPath, isoClass, etc.)
 * @return List of result objects containing table names and export paths
 */
def exec(Connection connection, input) {
  Logger logger = LoggerFactory.getLogger("org.hrisk")
  
  try {
    logger.info("Starting noise calculation process")

    // Initialize the NoiseModelling utility class
    def noiseModelling = new NoiseModelling(connection)
    
    // Step 1: Clear any existing data in the database
    noiseModelling.clearDatabase()

    // set input tables
    String resultTable =  noiseModelling.importTable(input["resultGeomPath"], input["inputSRID"])
    String triangleTable =  noiseModelling.importTable(input["triangleGeomPath"], input["inputSRID"])


    // Get list of tables before running the noise calculation
    def tables_before_alg = noiseModelling.tableNames()
    logger.info("Current tables: " + tables_before_alg)

    // run calculation
    Map args = [
        "resultTable": resultTable, 
        "resultTableField": input["resultTableField"],
        "isoClass": input["isoClass"],
        "smoothCoefficient": input["smoothCoefficient"]
      ].findAll{ it.value!=null }

    noiseModelling.runScript(
      "Acoustic_Tools/Create_Isosurface",
      args
    )
    
    
    // Get list of tables after running the noise calculation
    def tables_after_alg = noiseModelling.tableNames()
    logger.info("Current tables: " + tables_after_alg)

    // Identify the new tables created by the calculation
    def result_tables = tables_after_alg - tables_before_alg
    logger.info("Result tables created by noise calculation: " + result_tables)

    // Step 4: Export all result tables to GeoJSON files
    results = []
    for (tbl in result_tables){
      geom_export = noiseModelling.exportTables(tbl, input["exportDir"])
      results.add(
        [
          "tableName": tbl,
          "exportPath": geom_export
        ]
      )
    }

    logger.info("Noise calculation process completed successfully")
    
    return results
  
  
  } finally {
    
  }
}

