from qgis.PyQt.QtCore import (QCoreApplication, QT_TRANSLATE_NOOP)

from qgis.core import (
  QgsCoordinateReferenceSystem,
  QgsProcessingParameterExtent,
  QgsProcessingParameterDistance,
  QgsProcessingParameterCrs, 
  QgsProcessingParameterRasterDestination,
  QgsProcessingParameterNumber,
  QgsRectangle,
  QgsReferencedRectangle,
  QgsProcessingAlgorithm
  )
from qgis import processing

from ..algutil.hriskutil import HrUtil
from ..algutil.hrisktile import HrQgsTile
from ..algutil.hriskpostprocessor import HrPostProcessor
from ..algutil.hriskvar import PostProcessors

import os

class fetchpopghs(QgsProcessingAlgorithm):
  
  PARAMETERS = {  
    "FETCH_EXTENT": {
      "ui_func": QgsProcessingParameterExtent,
      "ui_args":{
        "description": QT_TRANSLATE_NOOP("fetchpopghs","Extent for fetching data")
      }
    },
    "TARGET_CRS": {
      "ui_func": QgsProcessingParameterCrs,
      "ui_args": {
        "description": QT_TRANSLATE_NOOP("fetchpopghs","Target CRS (Cartesian coordinates)")
      }
    },
    "BUFFER": {
      "ui_func": QgsProcessingParameterDistance,
      "ui_args": {
        "description": QT_TRANSLATE_NOOP("fetchpopghs","Buffer of the fetch area (using Target CRS)"),
        "defaultValue": 0.0,
        "parentParameterName": "TARGET_CRS"
      }
    },
    "MAX_DOWNLOAD": {
      "ui_func": QgsProcessingParameterNumber,
      "ui_args": {
        "description": QT_TRANSLATE_NOOP("fetchpopghs","Maximum number of download"),
        "type": QgsProcessingParameterNumber.Integer,
        "defaultValue": 4
      }
    },
    "OUTPUT": {
      "ui_func": QgsProcessingParameterRasterDestination,
      "ui_args": {
        "description": QT_TRANSLATE_NOOP("fetchpopghs","Population" )
      }
    }
  }  
  
  FETCH_BASE_URL = "https://jeodpp.jrc.ec.europa.eu/ftp/jrc-opendata/GHSL/GHS_POP_GLOBE_R2023A/GHS_POP_E2020_GLOBE_R2023A_4326_3ss/V1-0/tiles/GHS_POP_E2020_GLOBE_R2023A_4326_3ss_V1_0_{lat_str}_{long_str}.zip"
  FETCH_CRS = QgsCoordinateReferenceSystem("EPSG:4326")
  FETCH_GEOM_TYPE = "Point"
  FETCH_TILE_FAMILY = "long_lat"
  FETCH_TILE_PARAMETERS = {"xunit": 10.0, "yunit": 10.0, "xorig": 0.0, "yorig": 0.0}

  def __init__(self):
    super().__init__()
    self.UTIL = HrUtil(self)
    
  # initialize of the algorithm  
  def initAlgorithm(self, config):    
    (extent, target_crs) = self.UTIL.getExtentAndCrsUsingCanvas()
    self.UTIL.setDefaultValue("FETCH_EXTENT", extent)
    self.UTIL.setDefaultValue("TARGET_CRS", target_crs.authid())
    self.UTIL.initParameters()

  # execution of the algorithm
  def processAlgorithm(self, parameters, context, feedback): 
        
    self.UTIL.registerProcessingParameters(parameters, context, feedback)
    self.CURRENT_PROCESS = self.UTIL.parseCurrentProcess()
    # get target x-y CRS, to apply the buffer and determine the fetch area
    target_crs = self.parameterAsCrs(parameters, "TARGET_CRS", context)
    
    # check whether the target CRS is x-y coordinates
    self.UTIL.checkCrsAsCartesian(target_crs)
    
    # get the extent, using the target CRS
    fetch_extent = self.UTIL.asQgsReferencedRectangle(parameters["FETCH_EXTENT"], target_crs)
    
    # get the buffer
    buffer = self.parameterAsDouble(parameters, "BUFFER",context)
    
    # get the fetch area, using the extent and buffer
    fetch_area = QgsReferencedRectangle(
      QgsRectangle(
        fetch_extent.xMinimum() - buffer,
        fetch_extent.yMinimum() - buffer,
        fetch_extent.xMaximum() + buffer,
        fetch_extent.yMaximum() + buffer
      ),
      target_crs
    )
    
    tile = HrQgsTile(
      crs = self.FETCH_CRS,
      **self.FETCH_TILE_PARAMETERS
    )
    fetch_area_tile = self.UTIL.transformExtent(fetch_area, target_crs, self.FETCH_CRS)
    long_lat_list = tile.cellXyIdx(fetch_area_tile)
    
    fetch_args = {}
    for i, (long_idx, lat_idx) in enumerate(long_lat_list):
      long_str = f"C{long_idx + 18 + 1}"
      lat_str  = f"R{9 - lat_idx}"
      fetch_args[f"{i+1}/{len(long_lat_list)}"] = {
        "url": self.FETCH_BASE_URL.format(lat_str = lat_str, long_str = long_str)
      }
          
    if len(fetch_args) > self.parameterAsInt(parameters, "MAX_DOWNLOAD", context):
      feedback.reportError(self.tr("Too many downloads are required: ") + str(len(fetch_args)))
      raise Exception(self.tr("Too many downloads are required: ") + str(len(fetch_args)))

    fetch_results = self.UTIL.downloadFilesConcurrently(
      args=fetch_args, parallel=False
    )
    
    files_unpack = []
    for file in fetch_results.values():
      _, extracted_files = self.UTIL.extractArchive(file, extension=".zip", pattern = "\\.tif$")
      files_unpack += extracted_files
    
    
    ras_path = self.parameterAsOutputLayer(parameters, "OUTPUT", context)
    
    processing.run(
      "gdal:merge",{
        "INPUT": files_unpack,
        "OUTPUT": ras_path
      },
      context = context,
      is_child_algorithm = True
    )
    
    # register history and postprocessor for the raster output
    PostProcessors[ras_path] = HrPostProcessor(
      history=[self.CURRENT_PROCESS], 
      color_args = {"coloring": "single_band_pseudo_color", "theme": "Purples", "opacity": 0.5},
      set_min_to_zero = True
    )
    
    self.UTIL.registerPostProcessAlgorithm(context, PostProcessors)  
    return {"OUTPUT": ras_path.replace(os.path.sep, "/")}   
    
  def name(self):
    return self.__class__.__name__
  
  def displayName(self):
    return self.tr("Population (GHS)")

  def group(self):
    return self.tr('Fetch geometries')

  def groupId(self):
    return 'fetchgeometry'

  def createInstance(self):
    return fetchpopghs()


  # placing here is necessary, when employing pylupdate
  def tr(self, string):
    return QCoreApplication.translate(self.__class__.__name__, string)
