from qgis.PyQt.QtCore import (QT_TRANSLATE_NOOP, QCoreApplication, QVariant)
from qgis.core import (
  QgsRectangle,
  QgsProcessingAlgorithm,
  QgsReferencedRectangle,
  QgsProcessingParameterExtent,
  QgsProcessingParameterCrs,
  QgsProcessingParameterDistance,
  QgsProcessingParameterFeatureSink,
  QgsProcessingParameterNumber,
  QgsProject
  )
from qgis import processing

from ..algutil.hriskutil import HrUtil

class fetchbuildingosm(QgsProcessingAlgorithm):
  
  PARAMETERS = {  
    "FETCH_EXTENT": {
      "ui_func": QgsProcessingParameterExtent,
      "ui_args":{
        "description": QT_TRANSLATE_NOOP("fetchbuildingosm","Extent for fetching data")
      }
    },
    "TARGET_CRS": {
      "ui_func": QgsProcessingParameterCrs,
      "ui_args": {
        "description": QT_TRANSLATE_NOOP("fetchbuildingosm","Target CRS (Cartesian coordinates)")
      }
    },
    "BUFFER": {
      "ui_func": QgsProcessingParameterDistance,
      "ui_args": {
        "description": QT_TRANSLATE_NOOP("fetchbuildingosm","Buffer of the fetch area (using Target CRS)"),
        "defaultValue": 0.0,
        "parentParameterName": "TARGET_CRS"
      }
    },
    "HEIGHT": {
      "ui_func": QgsProcessingParameterNumber,
      "ui_args":{
        "optional": True,
        "description" : QT_TRANSLATE_NOOP("fetchbuildingja","Default height of buildings"),
        "defaultValue": 8.0
      }
    },
    "OUTPUT": {
      "ui_func": QgsProcessingParameterFeatureSink,
      "ui_args": {
        "description": QT_TRANSLATE_NOOP("fetchbuildingosm","Building")
      }
    }
  }
  
  
  FETCH_TILE_FAMILY = "web_mercator"
  FETCH_BASE_URL = "https://lz4.overpass-api.de/api/interpreter"
  FETCH_TIMEOUT = 30
  FETCH_QUERY_KEY = "building"
  FETCH_QUERY_VALUE = None
  FETCH_ZOOM = None
  FETCH_GEOM_TYPE = "Polygon"
  
  
  def __init__(self):
    super().__init__()
    self.UTIL = HrUtil(self)
    
  def initAlgorithm(self, config):    
    (extent, target_crs) = self.UTIL.getExtentAndCrsUsingCanvas()
    self.UTIL.setDefaultValue("FETCH_EXTENT", extent)
    self.UTIL.setDefaultValue("TARGET_CRS", target_crs.authid())
    self.UTIL.initParameters()
  
  def processAlgorithm(self, parameters, context, feedback):
    
    self.UTIL.registerProcessingParameters(parameters, context, feedback)
    self.CURRENT_PROCESS = self.UTIL.parseCurrentProcess()
    # get target x-y CRS, to apply the buffer and determine the fetch area
    target_crs = self.parameterAsCrs(parameters, "TARGET_CRS", context)
    
    # check whether the target CRS is x-y coordinates
    self.UTIL.checkCrsAsCartesian(target_crs)
    
    # get the extent, using the target CRS
    fetch_extent = self.UTIL.asQgsReferencedRectangle(parameters["FETCH_EXTENT"], target_crs)
    
    # get the buffer
    buffer = self.parameterAsDouble(parameters, "BUFFER",context)
    
    # get the fetch area, using the extent and buffer
    fetch_area = QgsReferencedRectangle(
      QgsRectangle(
        fetch_extent.xMinimum() - buffer,
        fetch_extent.yMinimum() - buffer,
        fetch_extent.xMaximum() + buffer,
        fetch_extent.yMaximum() + buffer
      ),
      target_crs
    )
    
    fetch_area_tr = self.UTIL.transformExtent(fetch_area, target_crs, QgsProject.instance().crs())
    
    fetch_results = processing.run(
      "quickosm:downloadosmdataextentquery",
      {
        "KEY": self.FETCH_QUERY_KEY,
        "VALUE": self.FETCH_QUERY_VALUE,
        "TIMEOUT": self.FETCH_TIMEOUT,
        "SERVER": self.FETCH_BASE_URL,
        "EXTENT": fetch_area_tr,
        "OUTPUT": "TEMPORARY_OUTPUT"
      },
      # context = context, # not passed, to avoid adding a new layer
      feedback = feedback,
    )["OUTPUT_MULTIPOLYGONS"]
    
    bldg_raw = fetch_results
    
      # transform
    bldg_transformed = processing.run(
    "native:reprojectlayer", 
    {
      "INPUT": bldg_raw,
      "TARGET_CRS": target_crs,
      "OUTPUT": "TEMPORARY_OUTPUT"
    },
    context = context,
    is_child_algorithm = True
    )["OUTPUT"]      
    
    
    
    feedback.pushInfo(self.tr("Snapping and dissolving are applied."))
    # snap geometry
    bldg_snap = processing.run(
      "native:snapgeometries", 
      {
        "INPUT": bldg_transformed,
        "REFERENCE_LAYER": context.getMapLayer(bldg_transformed),
        "TOLERANCE": 0.1,
        "BEHAVIOR": 0,
        "OUTPUT": "TEMPORARY_OUTPUT"
      },
      context = context,
      is_child_algorithm = True
    )["OUTPUT"]
      
    # dissolve
    bldg_dissolve = self.UTIL.dissolve(bldg_snap)
      
    
    bldg_final = processing.run(
      "hrisk:initbuilding",{
        "INPUT": bldg_dissolve,
        "TARGET_CRS": target_crs,
        "OVERWRITE": True,
        "HEIGHT": self.parameterAsDouble(parameters, "HEIGHT", context),
        "OUTPUT": "TEMPORARY_OUTPUT"
      },
      context = context,
      is_child_algorithm = True
    )["OUTPUT"]
    
    
    bldg_final_fts = context.getMapLayer(bldg_final)   
    
    fields_with_values = {
      "HISTORY": {
        "type": QVariant.String, 
        "value": self.CURRENT_PROCESS,
        "append": False
        }
    }
    dest_id = self.UTIL.outputVectorLayer(
      vector_layer= bldg_final_fts,
      param_sink = "OUTPUT",
      fields_with_values= fields_with_values
    )
    
    return {"OUTPUT": dest_id}
  
  def name(self):
    return self.__class__.__name__

  def displayName(self):
    return self.tr("Building (OSM)")

  def group(self):
    return self.tr('Fetch geometries')

  def groupId(self):
    return 'fetchgeometry'

  def createInstance(self):
    return fetchbuildingosm()

  # placing here is necessary, when employing pylupdate
  def tr(self, string):
    return QCoreApplication.translate(self.__class__.__name__, string)
