/**
 * Road Emission from Traffic Calculation Script
 * 
 * This Groovy script calculates road noise emissions (LW) based on traffic data.
 * It integrates with the NoiseModelling framework to process road geometry files
 * and generate noise emission data for acoustic modeling purposes.
 * 
 * Dependencies:
 * - H2GIS database for spatial data processing
 * - NoiseModelling framework for acoustic calculations
 * - SLF4J and Log4j for logging
 * 
 * @author Junta Tagusari
 * @version 1.0
 */

// Import required libraries for database connectivity, file handling, and logging
import org.h2gis.api.ProgressVisitor;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import java.sql.Connection;
import java.nio.file.Path;
import java.nio.file.Paths;
import org.apache.log4j.PropertyConfigurator;
import java.io.FileWriter;
import java.io.PrintWriter;


title = 'Road emission from traffic'
description = 'Road emission (LW) is obtained from the information on the traffic'

inputs = [
  roadGeomPath:[
    name : "Path of the road file",
    title : "Path of the road file",
    description : "Path of the road file",
    min        : 0, max: 1,
    type : String.class
  ],
  inputSRID: [
    name: 'Projection identifier',
    title: 'Projection identifier',
    description: 'Original projection identifier (also called SRID) of your table. It should be an EPSG code, a integer with 4 or 5 digits (ex: 3857 is Web Mercator projection). ' +
            '</br>  All coordinates will be projected from the specified EPSG to WGS84 coordinates. ' +
            '</br> This entry is optional because many formats already include the projection and you can also import files without geometry attributes.</br> ' +
            '</br> <b> Default value : 4326 </b> ',
    type: Integer.class,
    min: 0, max: 1
  ],
  noiseModellingHome : [
    name: "Path of NOISEMODELLING_HOME",
    title: "Path of NOISEMODELLING_HOME",
    description: "Path of NOISEMODELLING_HOME",
    type : String.class
  ],
  exportDir : [
    name: "Path of export directory",
    title: "Path of export directory",
    description: "Path of export directory",
    min        : 0, max: 1,
    type : String.class
  ]
]

outputs = [
  result: [
    name: 'Result output string', 
    title: 'Result output string', 
    description: 'This type of result does not allow the blocks to be linked together.', 
    type: String.class
  ]
]

/**
 * Main execution function for road emission calculation from traffic data
 * This function orchestrates the road emission calculation workflow:
 * 1. Clear existing database tables
 * 2. Import road geometry data
 * 3. Calculate road noise emissions based on traffic information
 * 4. Export results to files
 * 
 * @param connection H2GIS database connection
 * @param input Map containing input parameters (roadGeomPath, inputSRID, exportDir)
 * @return List of result objects containing table names and export paths
 */
def exec(Connection connection, input) {
  Logger logger = LoggerFactory.getLogger("org.hrisk")
  
  try {
    logger.info("Starting noise calculation process")
    
    // Initialize the NoiseModelling utility class
    def noiseModelling = new NoiseModelling(connection)
    
    // Step 1: Clear any existing data in the database
    noiseModelling.clearDatabase()

    // Step 2: Import the road geometry data into the database
    String tableRoads = noiseModelling.importTable(input["roadGeomPath"], input["inputSRID"])

    // Get list of tables before running the noise calculation
    def tables_before_alg = noiseModelling.tableNames()
    logger.info("Current tables: " + tables_before_alg)

    // Step 3: Execute the road emission calculation
    Map args = [
        "tableRoads": tableRoads
      ].findAll{ it.value!=null }

    noiseModelling.runScript("NoiseModelling/Road_Emission_from_Traffic", args)

    // Get list of tables after running the noise calculation
    def tables_after_alg = noiseModelling.tableNames()
    logger.info("Current tables: " + tables_after_alg)

    // Identify the new tables created by the calculation
    def result_tables = tables_after_alg - tables_before_alg
    logger.info("Result tables created by noise calculation: " + result_tables)

    // Step 4: Export all result tables to GeoJSON files
    results = []
    for (tbl in result_tables){
      geom_export = noiseModelling.exportTables(tbl, input["exportDir"])
      results.add(
        [
          "tableName": tbl,
          "exportPath": geom_export
        ]
      )
    }

    logger.info("Noise calculation process completed successfully")

    // Return the list of exported files with their table names and paths
    return results
  
  } finally { 
    // Cleanup completed - no additional resources to close
  }
}

