from qgis.PyQt.QtCore import (QT_TRANSLATE_NOOP, QVariant, QCoreApplication)
from qgis.core import (
  QgsProcessing,
  QgsProcessingAlgorithm,
  QgsProcessingParameterFeatureSource,
  QgsProcessingParameterFeatureSink,
  QgsFeatureRequest,
  QgsProcessingParameterField,
  QgsFeature,
  QgsField,
  QgsFields
  )

from qgis import processing
from ..algutil.hriskvar import PostProcessors
from ..algutil.hriskutil import HrUtil
from ..algutil.hriskfields import HrFields
from ..algutil.hriskpostprocessor import HrPostProcessor

class estimatepopulationofbuildingplg(QgsProcessingAlgorithm):
  PARAMETERS = {
    "BUILDING": {
      "ui_func": QgsProcessingParameterFeatureSource,
      "ui_args":{
        "description": QT_TRANSLATE_NOOP("estimatepopulationofbuildingplg","Building layer"),
        "types": [QgsProcessing.TypeVectorPolygon]
      }
    },
    "BUILDING_WEIGHT": {
      "ui_func": QgsProcessingParameterField,
      "ui_args":{
        "description": QT_TRANSLATE_NOOP("estimatepopulationofbuilding","Weight field of the Building layer to assign the population; if not provided, the areas are used."),
        "parentLayerParameterName": "BUILDING",
        "optional": True
      }
    },
    "POP": {
      "ui_func": QgsProcessingParameterFeatureSource,
      "ui_args":{
        "description": QT_TRANSLATE_NOOP("estimatepopulationofbuildingplg","Population layer"),
        "types": [QgsProcessing.TypeVectorPolygon]
      }
    },
    "PK_FIELD": {
      "ui_func": QgsProcessingParameterField,
      "ui_args":{
        "description": QT_TRANSLATE_NOOP("estimatepopulationofbuildingplg","Primary key field"),
        "parentLayerParameterName": "POP"
      }
    },
    "POP_FIELD": {
      "ui_func": QgsProcessingParameterField,
      "ui_args":{
        "description": QT_TRANSLATE_NOOP("estimatepopulationofbuildingplg","Population field"),
        "parentLayerParameterName": "POP"
      }
    },
    "OUTPUT": {
      "ui_func": QgsProcessingParameterFeatureSink,
      "ui_args": {
        "description": QT_TRANSLATE_NOOP("estimatepopulationofbuildingplg","Building with population")
      }
    }
  }
  

  
  def __init__(self) -> None:
    super().__init__()
    self.UTIL = HrUtil(self)
    self.FIELDS = HrFields.fromQgsFieldList([
      # QgsField("pkPlg",      QVariant.Int),
      QgsField("nBldgPlg",   QVariant.Int),
      QgsField("wgtBldg",    QVariant.Double),
      QgsField("wgtBldgPlg", QVariant.Double),
      QgsField("popPlg",     QVariant.Int),
      QgsField("pop",        QVariant.Double)
    ])
  
  def initAlgorithm(self, config):
    self.UTIL.initParameters()
    

  def cmptBldgPntLayer(self, bldg_layer, pop_plg, join_fields, context):
    
    bldg_cent_pnt = processing.run(
      "native:centroids",
      {
        "INPUT": bldg_layer,
        "OUTPUT": "TEMPORARY_OUTPUT"
      },
      context = context,
      is_child_algorithm = True,
    )["OUTPUT"]
    
    bldg_cent_pnt_transformed = processing.run(
      "native:reprojectlayer",
      {
        "INPUT": bldg_cent_pnt,
        "TARGET_CRS" : pop_plg.crs(),
        "OUTPUT" : "TEMPORARY_OUTPUT"
      },
      context = context,
      is_child_algorithm = True,
    )["OUTPUT"]
    
    # sampling the population from polygon
    bldg_cent_pnt_pop = processing.run(
      "native:joinattributesbylocation",
      {
        "DISCARD_NONMATCHING": False,
        "INPUT": bldg_cent_pnt_transformed,
        "JOIN": pop_plg,
        "JOIN_FIELDS": join_fields,
        "METHOD": 0, # intersect
        "PREDICATE": [0], # equals
        "PREFIX": "plg",
        "OUTPUT" : "TEMPORARY_OUTPUT"
      },
      context = context,
      is_child_algorithm = True,
    )["OUTPUT"]
    
    
    return bldg_cent_pnt_pop
  
  
  def processAlgorithm(self, parameters, context, feedback):   
    
    self.UTIL.registerProcessingParameters(parameters, context, feedback)
    self.CURRENT_PROCESS = self.UTIL.parseCurrentProcess(with_nm=True)
    
    pop = self.parameterAsSource(parameters, "POP", context).materialize(QgsFeatureRequest(), feedback) 
    bldg_plg = self.parameterAsSource(parameters, "BUILDING", context).materialize(QgsFeatureRequest(), feedback) 
    plg_pk_fld = self.parameterAsString(parameters, "PK_FIELD", context)
    plg_pop_fld = self.parameterAsString(parameters, "POP_FIELD", context)
    self.FIELDS.append(QgsField("pkPlg", pop.fields().field(plg_pk_fld).type()))
    
    bldg_pnt = self.cmptBldgPntLayer(bldg_plg, pop, [plg_pk_fld, plg_pop_fld], context)
    bldg_pnt = context.temporaryLayerStore().mapLayers()[bldg_pnt]        
    
    bldg_fld_wgt = self.parameterAsFields(parameters, "BUILDING_WEIGHT", context)
    bldg_fld_wgt = None if len(bldg_fld_wgt) == 0 else bldg_fld_wgt[0]
    
    if bldg_fld_wgt is not None:
      wgt_type = bldg_plg.fields()[bldg_plg.fields().indexOf(bldg_fld_wgt)].type()
      if wgt_type != QVariant.Double and wgt_type != QVariant.Int:
        feedback.reportError(self.tr("The weight field of the building layer must be number (double or integer)."))
        raise Exception(self.tr("The weight field of the building layer must be number (double or integer)."))
      
    plg_pk_fld = "plg" + plg_pk_fld
    plg_pop_fld = "plg" + plg_pop_fld
    
    # initialize the fields
    bldg_fields = QgsFields(bldg_plg.fields())
    for fld in self.FIELDS:
      if fld.name() in bldg_fields.names():
        bldg_fields.remove(bldg_fields.lookupField(fld.name()))
      bldg_fields.append(fld)
    new_fields = self.UTIL.newFieldsWithHistory(bldg_fields)
          
    # create sink
    (sink, dest_id) = self.parameterAsSink(
      parameters, "OUTPUT", context,
      new_fields, bldg_plg.wkbType(), bldg_plg.sourceCrs()
    )
    
      
    # first loop is to get the worldmesh information from ft_pnt
    plg_pk_ft = []
    plg_info = {}    
    for ft_plg, ft_pnt in zip(bldg_plg.getFeatures(), bldg_pnt.getFeatures()):
      
      plg_pk = ft_pnt[plg_pk_fld]
      plg_pk_ft.append(plg_pk)
      
      # accumlate the information regarding the worldmesh
      if plg_info.get(plg_pk) == None:
        plg_info[plg_pk] = {"popPlg": ft_pnt[plg_pop_fld], "nBldgPlg": 0,"wgtBldgPlg": 0.0}
      plg_info[plg_pk]["nBldgPlg"] += 1
      if bldg_fld_wgt is not None:
        plg_info[plg_pk]["wgtBldgPlg"] += float(ft_plg[bldg_fld_wgt])
      else:
        plg_info[plg_pk]["wgtBldgPlg"] += ft_plg.geometry().area()
    
    # second loop is to set the information to the output layer      
    for ft_plg, ft_pnt, plg_pk in zip(bldg_plg.getFeatures(), bldg_pnt.getFeatures(), plg_pk_ft):
      ft_new = QgsFeature(bldg_fields)
      ft_new.setGeometry(ft_plg.geometry())
      for field_name in ft_plg.fields().names():
        if field_name == "HISTORY":
          if len(ft_plg["HISTORY"]) > 0:
            ft_new["HISTORY"] = ft_plg["HISTORY"] + ";" + self.CURRENT_PROCESS
          else:
            ft_new["HISTORY"] = self.CURRENT_PROCESS
        elif not field_name in self.FIELDS.names():
          ft_new[field_name] = ft_plg[field_name]
          
      ft_new["pkPlg"]        = ft_pnt[plg_pk_fld]
      ft_new["nBldgPlg"]     = plg_info[plg_pk]["nBldgPlg"]
      ft_new["wgtBldgPlg"]   = plg_info[plg_pk]["wgtBldgPlg"]
      if bldg_fld_wgt is not None:
        ft_new["wgtBldg"]   = float(ft_plg[bldg_fld_wgt])
      else:
        ft_new["wgtBldg"]     = ft_new.geometry().area()
        
      ft_new["popPlg"]      = ft_pnt[plg_pop_fld]
      if ft_pnt[plg_pop_fld] is not None:
        ft_new["pop"]     = float(ft_pnt[plg_pop_fld]) / ft_new["wgtBldgPlg"] * ft_new["wgtBldg"]
      sink.addFeature(ft_new)    
      
    PostProcessors[dest_id] = HrPostProcessor(history = [self.CURRENT_PROCESS])
    self.UTIL.registerPostProcessAlgorithm(context, PostProcessors)
    
    return {"OUTPUT": dest_id}
  
  def name(self):
    return self.__class__.__name__
  def displayName(self):
    return self.tr("Estimate populations of buildings using Polygon")

  def group(self):
    return self.tr("Evaluate health risk")

  def groupId(self):
    return "healthrisk"
  
  def createInstance(self):
    return estimatepopulationofbuildingplg()

  # placing here is necessary, when employing pylupdate
  def tr(self, string):
    return QCoreApplication.translate(self.__class__.__name__, string)
