from qgis.PyQt.QtCore import (QT_TRANSLATE_NOOP, QVariant, QCoreApplication)
from qgis.core import (
  QgsProcessing,
  QgsProcessingAlgorithm,
  QgsProcessingParameterFeatureSource,
  QgsProcessingParameterFeatureSink,
  QgsFeatureRequest,
  QgsProcessingParameterRasterLayer,
  QgsProcessingParameterField,
  QgsFeature,
  QgsField,
  QgsFields
  )

from qgis import processing
from ..algutil.hriskvar import PostProcessors
from ..algutil.hriskutil import HrUtil
from ..algutil.hriskfields import HrFields
from ..algutil.hriskpostprocessor import HrPostProcessor
from ..worldmesh import (
  cal_meshcode_ex1m_16  
)

class estimatepopulationofbuilding(QgsProcessingAlgorithm):
  PARAMETERS = {
    "BUILDING": {
      "ui_func": QgsProcessingParameterFeatureSource,
      "ui_args":{
        "description": QT_TRANSLATE_NOOP("estimatepopulationofbuilding","Building layer"),
        "types": [QgsProcessing.TypeVectorPolygon]
      }
    },
    "BUILDING_WEIGHT": {
      "ui_func": QgsProcessingParameterField,
      "ui_args":{
        "description": QT_TRANSLATE_NOOP("estimatepopulationofbuilding","Weight field of the Building layer to assign the population; if not provided, the areas are used."),
        "parentLayerParameterName": "BUILDING",
        "optional": True
      }
    },
    "POP": {
      "ui_func": QgsProcessingParameterRasterLayer,
      "ui_args":{
        "description": QT_TRANSLATE_NOOP("estimatepopulationofbuilding","Population layer"),
      }
    },
    "OUTPUT": {
      "ui_func": QgsProcessingParameterFeatureSink,
      "ui_args": {
        "description": QT_TRANSLATE_NOOP("estimatepopulationofbuilding","Building with population")
      }
    }
  }
  
  FIELDS = HrFields.fromQgsFieldList([
    QgsField("meshIdx",     QVariant.Int),
    QgsField("meshCode",    QVariant.String),
    QgsField("nBldgMesh",   QVariant.Int),
    QgsField("wgtBldg",     QVariant.Double),
    QgsField("wgtBldgMesh", QVariant.Double),
    QgsField("popMesh",     QVariant.Int),
    QgsField("pop",         QVariant.Double)
  ])
  
  def __init__(self) -> None:
    super().__init__()
    self.UTIL = HrUtil(self)
  
  def initAlgorithm(self, config):
    self.UTIL.initParameters()
    

  def cmptBldgPntLayer(self, bldg_layer, pop_raster, context, feedback):
    
    bldg_cent_pnt = processing.run(
      "native:centroids",
      {
        "INPUT": bldg_layer,
        "OUTPUT": "TEMPORARY_OUTPUT"
      },
      context = context,
      is_child_algorithm = True,
    )["OUTPUT"]
    
    bldg_cent_pnt_transformed = processing.run(
      "native:reprojectlayer",
      {
        "INPUT": bldg_cent_pnt,
        "TARGET_CRS" : pop_raster.crs(),
        "OUTPUT" : "TEMPORARY_OUTPUT"
      },
      context = context,
      is_child_algorithm = True,
    )["OUTPUT"]
    
    # sampling the population from raster
    # the population is stored in the first band (1), named "pop1"
    bldg_cent_pnt_pop = processing.run(
      "native:rastersampling",
      {
        "INPUT": bldg_cent_pnt_transformed,
        "RASTERCOPY": pop_raster,
        "COLUMN_PREFIX": "pop",
        "OUTPUT" : "TEMPORARY_OUTPUT"
      },
      context = context,
      is_child_algorithm = True,
    )["OUTPUT"]
    
    return bldg_cent_pnt_pop
  
  
  def processAlgorithm(self, parameters, context, feedback):    
    
    self.UTIL.registerProcessingParameters(parameters, context, feedback)
    self.CURRENT_PROCESS = self.UTIL.parseCurrentProcess()
    
    pop = self.parameterAsRasterLayer(parameters, "POP", context)
    bldg_plg = self.parameterAsSource(parameters, "BUILDING", context).materialize(QgsFeatureRequest(), feedback) 
    bldg_pnt = self.cmptBldgPntLayer(bldg_plg, pop, context, feedback)
    bldg_pnt_fts = context.temporaryLayerStore().mapLayers()[bldg_pnt]
    
    bldg_fld_wgt = self.parameterAsFields(parameters, "BUILDING_WEIGHT", context)
    bldg_fld_wgt = None if len(bldg_fld_wgt) == 0 else bldg_fld_wgt[0]
    
    if bldg_fld_wgt is not None:
      wgt_type = bldg_plg.fields()[bldg_plg.fields().indexOf(bldg_fld_wgt)].type()
      if wgt_type != QVariant.Double and wgt_type != QVariant.Int:
        feedback.reportError(self.tr("The weight field of the building layer must be number (double or integer)."))
        raise Exception(self.tr("The weight field of the building layer must be number (double or integer)."))

    
    # initialize the fields
    bldg_fields = QgsFields(bldg_plg.fields())
    for fld in self.FIELDS:
      if fld.name() in bldg_fields.names():
        bldg_fields.remove(bldg_fields.lookupField(fld.name()))
      bldg_fields.append(fld)
    new_fields = self.UTIL.newFieldsWithHistory(bldg_fields)
      
    # define a function to get the worldmesh index
    def getMeshIdx(x, y):
      x_idx = (x - pop.extent().xMinimum()) // pop.rasterUnitsPerPixelX()
      y_idx = (y - pop.extent().yMinimum()) // pop.rasterUnitsPerPixelY()
      idx = y_idx * pop.width() + x_idx
      idx = int(idx) if idx > 0 else -1
      return int(idx)
    
    # create sink
    (sink, dest_id) = self.parameterAsSink(
      parameters, "OUTPUT", context,
      new_fields, bldg_plg.wkbType(), bldg_plg.sourceCrs()
    )
      
    # first loop is to get the worldmesh information from ft_pnt
    mesh_idx_ft = []
    mesh_info = {}
    for ft_plg, ft_pnt in zip(bldg_plg.getFeatures(), bldg_pnt_fts.getFeatures()):
      
      mesh_idx = getMeshIdx(ft_pnt.geometry().vertexAt(0).x(), ft_pnt.geometry().vertexAt(0).y())
      mesh_idx_ft.append(mesh_idx)
      
      # accumlate the information regarding the worldmesh
      if mesh_info.get(mesh_idx) == None:
        mesh_info[mesh_idx] = {"meshCode":None, "popMesh": ft_pnt["pop1"], "nBldgMesh": 0,"wgtBldgMesh": 0.0}
        if pop.crs().isGeographic():
          mesh_info[mesh_idx]["meshCode"] = cal_meshcode_ex1m_16(ft_pnt.geometry().vertexAt(0).y(), ft_pnt.geometry().vertexAt(0).x())
      mesh_info[mesh_idx]["nBldgMesh"] += 1
      if bldg_fld_wgt is not None:
        mesh_info[mesh_idx]["wgtBldgMesh"] += float(ft_plg[bldg_fld_wgt])
      else:
        mesh_info[mesh_idx]["wgtBldgMesh"] += ft_plg.geometry().area()
    
    
    # second loop is to set the information to the output layer      
    for ft_plg, ft_pnt, mesh_idx in zip(bldg_plg.getFeatures(), bldg_pnt_fts.getFeatures(), mesh_idx_ft):
      ft_new = QgsFeature(bldg_fields)
      ft_new.setGeometry(ft_plg.geometry())
      for field_name in ft_plg.fields().names():
        if field_name == "HISTORY":
          if len(ft_plg[field_name]) > 0:
            ft_new[field_name] = ft_plg[field_name] + ";" + self.CURRENT_PROCESS
          else:
            ft_new[field_name] = self.CURRENT_PROCESS
        elif not field_name in self.FIELDS.names():
          ft_new[field_name] = ft_plg[field_name]
          
      ft_new["meshIdx"]      = mesh_idx
      ft_new["meshCode"]     = mesh_info[mesh_idx]["meshCode"]
      ft_new["nBldgMesh"]    = mesh_info[mesh_idx]["nBldgMesh"]
      ft_new["wgtBldgMesh"]  = mesh_info[mesh_idx]["wgtBldgMesh"]
      
      if bldg_fld_wgt is not None:
        ft_new["wgtBldg"]    = float(ft_plg[bldg_fld_wgt])
      else:
        ft_new["wgtBldg"]    = ft_new.geometry().area()
        
      ft_new["popMesh"]      = ft_pnt["pop1"]
      if ft_pnt["pop1"] is not None:
        ft_new["pop"]     = float(ft_pnt["pop1"]) / ft_new["wgtBldgMesh"] * ft_new["wgtBldg"]
        
      sink.addFeature(ft_new)    
      
    PostProcessors[dest_id] = HrPostProcessor(
      history = [self.CURRENT_PROCESS],
      coloring = "none"
      )
    self.UTIL.registerPostProcessAlgorithm(context, PostProcessors)
    
    return {"OUTPUT": dest_id}
  
  def name(self):
    return self.__class__.__name__
  
  def displayName(self):
    return self.tr("Estimate populations of buildings using Raster")

  def group(self):
    return self.tr("Evaluate health risk")

  def groupId(self):
    return "healthrisk"
  
  def createInstance(self):
    return estimatepopulationofbuilding()

  # placing here is necessary, when employing pylupdate
  def tr(self, string):
    return QCoreApplication.translate(self.__class__.__name__, string)
