from qgis.PyQt.QtCore import (QT_TRANSLATE_NOOP, QCoreApplication, QVariant)
from qgis.core import (
  QgsProcessing,
  QgsProcessingAlgorithm,
  QgsProcessingParameterFeatureSource,
  QgsProcessingParameterFeatureSink,
  QgsProcessingParameterField,
  QgsFields,
  QgsField,
  QgsProcessingParameterBoolean,
  QgsFeatureRequest,
  QgsProcessingParameterNumber,
  QgsFeature
  )


from ..algutil.hriskutil import HrUtil
from ..algutil.hriskpostprocessor import HrPostProcessor
from ..algutil.hriskvar import PostProcessors

class estimatelevelofbuilding(QgsProcessingAlgorithm):
  PARAMETERS = {
    
    "BUILDING": {
      "ui_func": QgsProcessingParameterFeatureSource,
      "ui_args":{
        "description": QT_TRANSLATE_NOOP("estimatelevelofbuilding","Building layer"),
        "types": [QgsProcessing.TypeVectorPolygon]
      }
    },
    "BUILDING_BKEY": {
      "ui_func": QgsProcessingParameterField,
      "ui_args":{
        "description": QT_TRANSLATE_NOOP("estimatelevelofbuilding","Building ID Field of the Building layer"),
        "parentLayerParameterName": "BUILDING"
      }
    },
    "RECEIVER": {
      "ui_func": QgsProcessingParameterFeatureSource,
      "ui_args":{
        "description": QT_TRANSLATE_NOOP("estimatelevelofbuilding","Receiver layer"),
        "types": [QgsProcessing.TypeVectorPoint]
      }
    },
    "RECEIVER_BKEY": {
      "ui_func": QgsProcessingParameterField,
      "ui_args":{
        "description": QT_TRANSLATE_NOOP("estimatelevelofbuilding","Building ID Field of the Receiver layer"),
        "parentLayerParameterName": "RECEIVER"
      }
    },
    "RECEIVER_RKEY": {
      "ui_func": QgsProcessingParameterField,
      "ui_args":{
        "description": QT_TRANSLATE_NOOP("estimatelevelofbuilding","Receiver ID Field of the Receiver layer"),
        "parentLayerParameterName": "RECEIVER"
      }
    },                          
    "LEVEL": {
      "ui_func": QgsProcessingParameterFeatureSource,
      "ui_args":{
        "description": QT_TRANSLATE_NOOP("estimatelevelofbuilding","Sound level layer"),
        "types": [QgsProcessing.TypeVectorPoint]
      }
    },
    "LEVEL_RKEY": {
      "ui_func": QgsProcessingParameterField,
      "ui_args":{
        "description": QT_TRANSLATE_NOOP("estimatelevelofbuilding","Receiver ID Field of the Sound level layer"),
        "parentLayerParameterName": "LEVEL"
      }
    },
    "LEVEL_ASSIGN": {
      "ui_func": QgsProcessingParameterField,
      "ui_args":{
        "description": QT_TRANSLATE_NOOP("estimatelevelofbuilding","Sound level field(s) of the Sound level layer"),
        "parentLayerParameterName": "LEVEL",
        "allowMultiple": True
      }
    },
    "LEVEL_ASSIGN_COLOR": {
      "ui_func": QgsProcessingParameterField,
      "ui_args":{
        "description": QT_TRANSLATE_NOOP("estimatelevelofbuilding","Field to color the results"),
        "parentLayerParameterName": "LEVEL"
      }
    },
    "SUBTRACTION_TO_CANCEL_REFLECTION": {
      "ui_func": QgsProcessingParameterNumber,
      "ui_args": {
        "description": QT_TRANSLATE_NOOP("estimatelevelofbuilding","Subtraction of sound level to cancel the reflection from the facade (dB)"),
        "type": QgsProcessingParameterNumber.Double,
        "minValue": 0.0, "defaultValue": 3.0, "maxValue": 10.0
      }
    },
    "OUTPUT": {
      "ui_func": QgsProcessingParameterFeatureSink,
      "ui_args": {
        "description": QT_TRANSLATE_NOOP("estimatelevelofbuilding","Building with facade level")
      }
    }
  }
  
  def __init__(self) -> None:
    super().__init__()
    self.UTIL = HrUtil(self)
      
  def initAlgorithm(self, config):
    self.UTIL.initParameters()


  def processAlgorithm(self, parameters, context, feedback):  
        
    self.UTIL.registerProcessingParameters(parameters, context, feedback)
    self.CURRENT_PROCESS = self.UTIL.parseCurrentProcess()
    
    # variables definition
    bldg_layer = self.parameterAsSource(parameters, "BUILDING", context).materialize(QgsFeatureRequest(), feedback)
    bldg_bkey = self.parameterAsString(parameters, "BUILDING_BKEY", context)
    
    rcv_layer = self.parameterAsSource(parameters, "RECEIVER", context).materialize(QgsFeatureRequest(), feedback)
    rcv_bkey = self.parameterAsString(parameters, "RECEIVER_BKEY", context)
    rcv_rkey = self.parameterAsString(parameters, "RECEIVER_RKEY", context)
    
    res_layer = self.parameterAsSource(parameters, "LEVEL", context).materialize(QgsFeatureRequest(), feedback)
    res_rkey = self.parameterAsString(parameters, "LEVEL_RKEY", context)
    res_assign = self.parameterAsFields(parameters, "LEVEL_ASSIGN", context)
    res_subtraction = self.parameterAsDouble(parameters, "SUBTRACTION_TO_CANCEL_REFLECTION", context)
    res_color_field = self.parameterAsString(parameters, "LEVEL_ASSIGN_COLOR", context)    
    
    # 1. make receiver_pk(rcv_rkey) to build_pk(rcv_bkey) mapping
    rkey_bkey_map = {}
    for ft in rcv_layer.getFeatures():
      if rkey_bkey_map.get(ft[rcv_rkey]) is not None:
        msg = self.tr("Receiver ID is not unique")
        feedback.reportError(msg)
        raise Exception(msg)
      rkey_bkey_map[ft[rcv_rkey]] = ft[rcv_bkey]
    
    # 2. link level to build_pk 
    bldg_results = {}
    for ft in res_layer.getFeatures():
      res_bval = rkey_bkey_map[ft[res_rkey]]
      if res_bval not in bldg_results:
        bldg_results[res_bval] = {k:[] for k in res_assign + [res_rkey]}
      for key in res_assign + [res_rkey]:
        val = ft[key]
        if val is not None:
          bldg_results[res_bval][key].append(val)
    
    # compute max, min, index
    result_ests = {"num_receivers": []}
    result_flds = {"num_receivers": QVariant.Int}
    for ra in res_assign:
      result_ests |= {
        f"{ra}_max": [],
        f"{ra}_max_idx": [],
        f"{ra}_min": [],
        f"{ra}_min_idx": []
      }
      result_flds |= {
        f"{ra}_max": QVariant.Double,
        f"{ra}_max_idx": QVariant.Int,
        f"{ra}_min": QVariant.Double,
        f"{ra}_min_idx": QVariant.Int
      }
    
    for ft in bldg_layer.getFeatures():
      bval = ft[bldg_bkey]
      results_all = bldg_results[bval]
      result_ests["num_receivers"].append(len(results_all[res_rkey]))
      rval = results_all[res_rkey]
      for key, value in results_all.items():
        if key == res_rkey:
          continue
        try:
          result_ests[f"{key}_max"].append(max(value) - res_subtraction)
          result_ests[f"{key}_max_idx"].append(rval[value.index(max(value))])
          result_ests[f"{key}_min"].append(min(value) - res_subtraction)
          result_ests[f"{key}_min_idx"].append(rval[value.index(min(value))])
        except:
          result_ests[f"{key}_max"].append(None)
          result_ests[f"{key}_max_idx"].append(None)
          result_ests[f"{key}_min"].append(None)
          result_ests[f"{key}_min_idx"].append(None)
    
    dest_id = self.UTIL.outputVectorLayer(
      vector_layer = bldg_layer,
      param_sink = "OUTPUT",
      fields_with_values = {
        "HISTORY": {
          "type": QVariant.String, 
          "value": self.CURRENT_PROCESS, 
          "append": True
        }
      } | {
        k: {"type": v, "value": result_ests[k]} 
        for k, v in result_flds.items()
      }
    )
    
    # set the postprocessing
    PostProcessors[dest_id] = HrPostProcessor(
        history = [self.CURRENT_PROCESS], 
        coloring = "graduated_symbol", 
        attribute = res_color_field + "_max", 
        theme ="Default"
    )
    
    return {"OUTPUT": dest_id}
          
  
  def name(self):
    return self.__class__.__name__
  
  def displayName(self):
    return self.tr("Assign calculated levels to buildings")

  def group(self):
    return self.tr("Evaluate health risk")

  def groupId(self):
    return "healthrisk"

  def createInstance(self):
    return estimatelevelofbuilding()

  # placing here is necessary, when employing pylupdate
  def tr(self, string):
    return QCoreApplication.translate(self.__class__.__name__, string)
