from qgis.PyQt.QtCore import (QT_TRANSLATE_NOOP, QCoreApplication)
from qgis.core import (
  QgsProcessingAlgorithm,
  QgsProcessingParameterExtent,
  QgsProcessingParameterCrs,
  QgsProcessingParameterDistance,
  QgsProcessingParameterFeatureSink,
  QgsCoordinateReferenceSystem,
  QgsProcessingParameterNumber,
  QgsRectangle,
  QgsReferencedRectangle
  )
from qgis import processing

from ..algutil.hriskutil import HrUtil
from ..algutil.hrisktile import WebMercatorTile


class fetchroadja(QgsProcessingAlgorithm):
  
  # UIs
  PARAMETERS = {  
    "FETCH_EXTENT": {
      "ui_func": QgsProcessingParameterExtent,
      "ui_args":{
        "description": QT_TRANSLATE_NOOP("fetchroadja","Extent for fetching data")
      }
    },
    "TARGET_CRS": {
      "ui_func": QgsProcessingParameterCrs,
      "ui_args": {
        "description": QT_TRANSLATE_NOOP("fetchroadja","Target CRS (Cartesian coordinates)")
      }
    },
    "BUFFER": {
      "ui_func": QgsProcessingParameterDistance,
      "ui_args": {
        "description": QT_TRANSLATE_NOOP("fetchroadja","Buffer of the fetch area (using Target CRS)"),
        "defaultValue": 0.0,
        "parentParameterName": "TARGET_CRS"
      }
    },        
    "MAX_DOWNLOAD": {
      "ui_func": QgsProcessingParameterNumber,
      "ui_args": {
        "description": QT_TRANSLATE_NOOP("fetchroadja","Maximum number of download"),
        "type": QgsProcessingParameterNumber.Integer,
        "defaultValue": 100
      }
    },
    "OUTPUT": {
      "ui_func": QgsProcessingParameterFeatureSink,
      "ui_args": {
        "description": QT_TRANSLATE_NOOP("fetchroadja","Road")
      }
    }
  }
  
  FETCH_BASE_URL = "https://cyberjapandata.gsi.go.jp/xyz/experimental_rdcl/{z}/{x}/{y}.geojson"
  FETCH_CRS = QgsCoordinateReferenceSystem("EPSG:6668")
  FETCH_ZOOM = 16
  FETCH_GEOM_TYPE = "LineString"
  
  def __init__(self):
    super().__init__()
    self.UTIL = HrUtil(self)
    
  # initialization of the algorithm
  def initAlgorithm(self, config):    
    (extent, target_crs) = self.UTIL.getExtentAndCrsUsingCanvas()
    self.UTIL.setDefaultValue("FETCH_EXTENT", extent)
    self.UTIL.setDefaultValue("TARGET_CRS", target_crs.authid())
    self.UTIL.initParameters()
  
  # execution of the algorithm
  def processAlgorithm(self, parameters, context, feedback):   
    
    self.UTIL.registerProcessingParameters(parameters, context, feedback)
    self.CURRENT_PROCESS = self.UTIL.parseCurrentProcess()
    # get target x-y CRS, to apply the buffer and determine the fetch area
    target_crs = self.parameterAsCrs(parameters, "TARGET_CRS", context)
    
    # check whether the target CRS is x-y coordinates
    self.UTIL.checkCrsAsCartesian(target_crs)
    
    # get the extent, using the target CRS
    fetch_extent = self.parameterAsExtent(
      parameters, "FETCH_EXTENT", context, 
      self.parameterAsCrs(parameters, "TARGET_CRS", context)
      )
    
    # get the buffer
    buffer = self.parameterAsDouble(parameters, "BUFFER",context)
    
    # get the fetch area, using the extent and buffer
    fetch_area = QgsReferencedRectangle(
      QgsRectangle(
        fetch_extent.xMinimum() - buffer,
        fetch_extent.yMinimum() - buffer,
        fetch_extent.xMaximum() + buffer,
        fetch_extent.yMaximum() + buffer
      ),
      target_crs
    )
    
    tile = WebMercatorTile(zoom = self.FETCH_ZOOM)
    
    tiles_list = tile.cellXyIdx(fetch_area)
    
    fetch_args = {}
    for i, (tx, ty) in enumerate(tiles_list):
      url_parsed = self.FETCH_BASE_URL.format(
        z = self.FETCH_ZOOM, x = tx, y = ty
      )
      
      fetch_args[f"{i+1}/{len(tiles_list)}"] = {
        "url": url_parsed,
        "zoom": self.FETCH_ZOOM,
        "tx": tx,
        "ty": ty,
        "geom_type": self.FETCH_GEOM_TYPE
      }
    
    if len(fetch_args) > self.parameterAsInt(parameters, "MAX_DOWNLOAD", context):
      feedback.reportError(self.tr("Too many downloads are required: ") + str(len(fetch_args)))
      raise Exception(self.tr("Too many downloads are required: ") + str(len(fetch_args)))
          
    fetch_results = self.UTIL.downloadFilesConcurrently(args = fetch_args)
        
    road_raw = processing.run(
      "native:mergevectorlayers",
      {
        "LAYERS": list(fetch_results.values()),
        "CRS": self.FETCH_CRS,
        "OUTPUT": "TEMPORARY_OUTPUT"
      },
      context = context,
      is_child_algorithm = True
    )["OUTPUT"]

    # transform and dissolve
    road_transformed = processing.run(
      "native:reprojectlayer", 
      {
        "INPUT": road_raw,
        "TARGET_CRS": target_crs,
        "OUTPUT": "TEMPORARY_OUTPUT"
      },
      context = context,
      is_child_algorithm = True
    )["OUTPUT"]      
      
      
    road_dissolve = self.UTIL.dissolve(road_transformed)

    road_final = processing.run(
      "hrisk:initroad",{
        "INPUT": road_dissolve,
        "TARGET_CRS": target_crs,
        "OVERWRITE": True,
        "OUTPUT": "TEMPORARY_OUTPUT"
      },
      context = context,
      is_child_algorithm = True
    )["OUTPUT"]
      
    road_final_fts = context.getMapLayer(road_final)
    
    new_fields = self.UTIL.newFieldsWithHistory(road_final_fts.fields())
    # set sink and add features with values
    (sink, dest_id) = self.parameterAsSink(
      parameters, "OUTPUT", context,
      new_fields, road_final_fts.wkbType(), target_crs
    )
    self.UTIL.addFeaturesWithHistoryToSink(
      sink, road_final_fts, new_fields,
      current_process=self.CURRENT_PROCESS
    )
      
    return {"OUTPUT": dest_id}
  
  def name(self):
    return self.__class__.__name__
  
  def displayName(self):
    return self.tr("Road centerline (Ja)")

  def group(self):
    return self.tr('Fetch geometries (Ja)')

  def groupId(self):
    return 'fetchgeomja'

  def createInstance(self):
    return fetchroadja()

  # placing here is necessary, when employing pylupdate
  def tr(self, string):
    return QCoreApplication.translate(self.__class__.__name__, string)
