from qgis.PyQt.QtCore import (QCoreApplication, QT_TRANSLATE_NOOP, QVariant)
from qgis.core import (
  QgsProcessingParameterCrs,
  QgsProcessingAlgorithm,
  QgsVectorLayer,
  QgsProcessingParameterExtent,
  QgsProcessingParameterDistance,
  QgsFeature,
  QgsPoint,
  QgsFeature,
  QgsCoordinateReferenceSystem,
  QgsProcessingParameterRasterDestination,
  QgsProcessingParameterNumber,
  QgsReferencedRectangle,
  QgsRectangle
  )
from qgis import processing

import urllib
import json
import os
from ..algutil.hriskutil import HrUtil
from ..algutil.worldmeshtile import WorldMeshTile
from ..algutil.hriskpostprocessor import HrPostProcessor
from ..algutil.hriskvar import PostProcessors

class fetchpopja(QgsProcessingAlgorithm):
  
  PARAMETERS = {  
    "FETCH_EXTENT": {
      "ui_func": QgsProcessingParameterExtent,
      "ui_args":{
        "description": QT_TRANSLATE_NOOP("fetchpopja","Extent for fetching data")
      }
    },
    "TARGET_CRS": {
      "ui_func": QgsProcessingParameterCrs,
      "ui_args": {
        "description": QT_TRANSLATE_NOOP("fetchpopja","Target CRS (Cartesian coordinates)")
      }
    },
    "BUFFER": {
      "ui_func": QgsProcessingParameterDistance,
      "ui_args": {
        "description": QT_TRANSLATE_NOOP("fetchpopja","Buffer of the fetch area (using Target CRS)"),
        "defaultValue": 0.0,
        "parentParameterName": "TARGET_CRS"
      }
    },
    "MAX_DOWNLOAD": {
      "ui_func": QgsProcessingParameterNumber,
      "ui_args": {
        "description": QT_TRANSLATE_NOOP("fetchpopja","Maximum number of download"),
        "type": QgsProcessingParameterNumber.Integer,
        "defaultValue": 100
      }
    },
    "OUTPUT": {
      "ui_func": QgsProcessingParameterRasterDestination,
      "ui_args": {
        "description": QT_TRANSLATE_NOOP("fetchpopja","Population" )
      }
    }
  }  
  
  FETCH_BASE_URL = "https://api.e-stat.go.jp/rest/3.0/app/json/getStatsData"
  FETCH_CRS = QgsCoordinateReferenceSystem("EPSG:6668")
  FETCH_GEOM_TYPE = "Point"
  FETCH_MESH_LEVEL = "5"
  
  ESTAT_ID_MESH_FILE = os.path.join(os.path.dirname(__file__),"estatId_mesh_list.txt")
  ESTAT_ID_MESH_DICT = {}
  
  ESTAT_API_URL = "https://api.e-stat.go.jp/rest/3.0/app/json/getStatsData"   
  ESTAT_API_PARAMS = {
    "appId": "b877fd89560ce21475681dba1a6681dd6426cbc3",
    "statsDataId": "",
    "statsCode": "00200521",
    "cdCat01": "0010",
    "cdArea": "",
    "metaGetFlg": "N"
  }
  
  
  LAT_UNIT =  7.5 / 3600.0
  LNG_UNIT = 11.25 / 3600.0
    
  def __init__(self):
    super().__init__()
    self.UTIL = HrUtil(self)
    
  def initAlgorithm(self, config):    
    (extent, target_crs) = self.UTIL.getExtentAndCrsUsingCanvas()
    self.UTIL.setDefaultValue("FETCH_EXTENT", extent)
    self.UTIL.setDefaultValue("TARGET_CRS", target_crs.authid())
    self.UTIL.initParameters()
  
  
  def processAlgorithm(self, parameters, context, feedback):   
    import ptvsd
    ptvsd.debug_this_thread()
    with open(self.ESTAT_ID_MESH_FILE) as f:
      self.ESTAT_ID_MESH_DICT = {key: value for line in f for (key, value) in [line.strip().split(None, 1)]}

    self.UTIL.registerProcessingParameters(parameters, context, feedback)
    self.CURRENT_PROCESS = self.UTIL.parseCurrentProcess()
    # get target x-y CRS, to apply the buffer and determine the fetch area
    target_crs = self.parameterAsCrs(parameters, "TARGET_CRS", context)
    
    # check whether the target CRS is x-y coordinates
    self.UTIL.checkCrsAsCartesian(target_crs)
    
    # get the extent, using the target CRS
    fetch_extent = self.parameterAsExtent(
      parameters, "FETCH_EXTENT", context, 
      self.parameterAsCrs(parameters, "TARGET_CRS", context)
      )
    
    # get the buffer
    buffer = self.parameterAsDouble(parameters, "BUFFER",context)
    
    # get the fetch area, using the extent and buffer
    fetch_area = QgsReferencedRectangle(
      QgsRectangle(
        fetch_extent.xMinimum() - buffer,
        fetch_extent.yMinimum() - buffer,
        fetch_extent.xMaximum() + buffer,
        fetch_extent.yMaximum() + buffer
      ),
      target_crs
    )
    
    tile = WorldMeshTile(
      level = self.FETCH_MESH_LEVEL, 
      crs = QgsCoordinateReferenceSystem("EPSG:6668")
    )
        
    mesh5_in_fetch_area = tile.cellMeshCode(fetch_area)
    mesh1_in_fetch_area = set([mc5[2:6] for mc5 in mesh5_in_fetch_area])
    
    mesh1_dict = {k: [] for k in mesh1_in_fetch_area}
    for mc5 in mesh5_in_fetch_area:
      mesh1_dict[mc5[2:6]].append(mc5)
    
    query = []
    for mc1, mc5_list in mesh1_dict.items():
      params_estat = self.ESTAT_API_PARAMS.copy()
      params_estat["statsDataId"] = self.ESTAT_ID_MESH_DICT.get(mc1)
    
      for mc5_shortlist in [mc5_list[i:i+20] for i in range(0, len(mc5_list), 20)]:
        params_estat["cdArea"] = ",".join([mc5[2:] for mc5 in mc5_shortlist])
        query.append(urllib.parse.urlencode(params_estat))
      
    fetch_args = {}
    for i, q in enumerate(query): 
      fetch_args[f"{i+1}/{len(query)}"] = {
        "url": self.FETCH_BASE_URL,
        "query": "?" + q
      }
          
          
    if len(fetch_args) > self.parameterAsInt(parameters, "MAX_DOWNLOAD", context):
      feedback.reportError(self.tr("Too many downloads are required: ") + str(len(fetch_args)))
      raise Exception(self.tr("Too many downloads are required: ") + str(len(fetch_args)))
          
    fetch_results = self.UTIL.downloadFilesConcurrently(args = fetch_args)
        
    # then merge the downloaded files
    init_string = "{geom}?crs={crs}&index=yes&field=population:integer".format(
      geom = self.FETCH_GEOM_TYPE, crs = self.FETCH_CRS.authid()
    )

    vec_layer = QgsVectorLayer(init_string, "pop", "memory")
    vec_pr = vec_layer.dataProvider()    
    
    for file, query in zip(fetch_results.values(), query):
      with open(file, 'r') as f:
        body = json.load(f)
      
      try:
        results = body["GET_STATS_DATA"]["STATISTICAL_DATA"]["DATA_INF"]["VALUE"]
        if not isinstance(results, list):
          results = [results]
          
        for stat_dict in results:
          try:
            mc5_at_the_area = [
              mc5 for mc5 in mesh5_in_fetch_area
              if mc5[2:] == stat_dict["@area"]
            ][0]
            longlat = tile.cellXyCenter(mc5_at_the_area)
            
            ft = QgsFeature(vec_layer.fields())
            ft.setGeometry(QgsPoint(longlat[0], longlat[1]))
            ft["population"] = stat_dict["$"]
            vec_pr.addFeatures([ft])
          except:
            feedback.pushInfo(self.tr("No worldmesh code was found for " + stat_dict))
      except:
        feedback.pushInfo(self.tr("No population results were obtained from " + query))
        
    
    pop_result = processing.run(
      "gdal:rasterize",
      {
        "INPUT": vec_layer,
        "FIELD": "population",
        "UNITS": 1,
        "WIDTH": self.LNG_UNIT,
        "HEIGHT": self.LAT_UNIT,
        "DATA_TYPE": 2,
        "INIT": 0,
        "NODATA": 0,
        "OUTPUT": self.parameterAsOutputLayer(parameters, "OUTPUT", context)
      },
      context = context,
      is_child_algorithm = True
    )["OUTPUT"]
    
    PostProcessors[pop_result] = HrPostProcessor(
      history = [self.CURRENT_PROCESS], 
      color_args = {"coloring": "single_band_pseudo_color", "theme": "Purples", "opacity": 0.5},
      set_min_to_zero = True   
    )
    self.UTIL.registerPostProcessAlgorithm(context, PostProcessors)

    return {"OUTPUT":pop_result}
  
  def name(self):
    return 'fetchpopja'
  
  def displayName(self):
    return self.tr("Population (Ja)")

  def group(self):
    return self.tr('Fetch geometries (Ja)')

  def groupId(self):
    return 'fetchgeomja'

  def createInstance(self):
    return fetchpopja()
  
  # placing here is necessary, when employing pylupdate
  def tr(self, string):
    return QCoreApplication.translate(self.__class__.__name__, string)
