from qgis.PyQt.QtCore import (QT_TRANSLATE_NOOP, QVariant, QCoreApplication)
from qgis.core import (
  QgsProcessing,
  QgsProcessingAlgorithm,
  QgsProcessingParameterFeatureSource,
  QgsProcessingParameterFeatureSink,
  QgsProcessingParameterBoolean,
  QgsFeatureRequest,
  QgsProcessingParameterCrs,
  QgsVectorLayer,
  QgsField,
  QgsWkbTypes,
  QgsProcessingParameterNumber
  )


from qgis import processing

from ..algutil.hriskvar import PostProcessors
from ..algutil.hriskutil import HrUtil
from ..algutil.hriskfields import HrFields
from ..algutil.hriskpostprocessor import HrPostProcessor

class initsource(QgsProcessingAlgorithm):
  PARAMETERS = {                  
    "INPUT": {
      "ui_func": QgsProcessingParameterFeatureSource,
      "ui_args":{
        "description": QT_TRANSLATE_NOOP("initsource","Source layer"),
        "types": [QgsProcessing.TypeVectorLine, QgsProcessing.TypeVectorPoint],
        "optional": True
      }
    },
    "USE_ZVALUE": {
      "ui_func": QgsProcessingParameterBoolean,
      "ui_args":{
        "description" : QT_TRANSLATE_NOOP("initsource","Use z values of the geometry?"),
        "defaultValue": False
      }
    },
    "Z_VALUE": {
      "ui_func": QgsProcessingParameterNumber,
      "ui_args":{
        "description" : QT_TRANSLATE_NOOP("initsource","Default z value (if current z value is not used)"),
        "defaultValue": 1.0
      }
    },
    "TARGET_CRS": {
      "ui_func": QgsProcessingParameterCrs,
      "ui_args": {
        "description": QT_TRANSLATE_NOOP("initsource","Target CRS (Cartesian coordinates)")
      }
    },
    "OVERWRITE": {
      "ui_func": QgsProcessingParameterBoolean,
      "ui_args":{
        "description" : QT_TRANSLATE_NOOP("initsource","Overwrite existing fields?"),
        "defaultValue": True
      }
    },
    "OUTPUT": {
      "ui_func": QgsProcessingParameterFeatureSink,
      "ui_args": {
        "description": QT_TRANSLATE_NOOP("initsource","Source" )
      }
    }
  }
  
  FIELDS = {}
  FIELDS_WITH_VALUES = {}
  
  def __init__(self):
    super().__init__()
    self.UTIL = HrUtil(self)
    
    self.FIELDS["COMMON"] = HrFields.fromQgsFieldList([
      QgsField("PK", QVariant.Int, "int"),
      QgsField("HISTORY", QVariant.String, "string")      
    ]).setComments("note: common fields; ")
    
    self.FIELDS["REQUIRED"] = HrFields.fromQgsFieldList([
      QgsField("LWd63",   QVariant.Double, "double", comment = "default: -99.0; "), # Octave-band sound power level at 63 Hz (dB) (6-18h)
      QgsField("LWd125",  QVariant.Double, "double", comment = "default: -99.0; "), # Octave-band sound power level at 125 Hz (dB) (6-18h)
      QgsField("LWd250",  QVariant.Double, "double", comment = "default: -99.0; "), # Octave-band sound power level at 250 Hz (dB) (6-18h)
      QgsField("LWd500",  QVariant.Double, "double", comment = "default: -99.0; "), # Octave-band sound power level at 500 Hz (dB) (6-18h)
      QgsField("LWd1000", QVariant.Double, "double", comment = "default: -99.0; "), # Octave-band sound power level at 1000 Hz (dB) (6-18h)
      QgsField("LWd2000", QVariant.Double, "double", comment = "default: -99.0; "), # Octave-band sound power level at 2000 Hz (dB) (6-18h)
      QgsField("LWd4000", QVariant.Double, "double", comment = "default: -99.0; "), # Octave-band sound power level at 4000 Hz (dB) (6-18h)
      QgsField("LWd8000", QVariant.Double, "double", comment = "default: -99.0; "), # Octave-band sound power level at 8000 Hz (dB) (6-18h)
      QgsField("LWe63",   QVariant.Double, "double", comment = "default: -99.0; "), # Octave-band sound power level at 63 Hz (dB) (18-22h)
      QgsField("LWe125",  QVariant.Double, "double", comment = "default: -99.0; "), # Octave-band sound power level at 125 Hz (dB) (18-22h)
      QgsField("LWe250",  QVariant.Double, "double", comment = "default: -99.0; "), # Octave-band sound power level at 250 Hz (dB) (18-22h)
      QgsField("LWe500",  QVariant.Double, "double", comment = "default: -99.0; "), # Octave-band sound power level at 500 Hz (dB) (18-22h)
      QgsField("LWe1000", QVariant.Double, "double", comment = "default: -99.0; "), # Octave-band sound power level at 1000 Hz (dB) (18-22h)
      QgsField("LWe2000", QVariant.Double, "double", comment = "default: -99.0; "), # Octave-band sound power level at 2000 Hz (dB) (18-22h)
      QgsField("LWe4000", QVariant.Double, "double", comment = "default: -99.0; "), # Octave-band sound power level at 4000 Hz (dB) (18-22h)
      QgsField("LWe8000", QVariant.Double, "double", comment = "default: -99.0; "), # Octave-band sound power level at 8000 Hz (dB) (18-22h)
      QgsField("LWn63",   QVariant.Double, "double", comment = "default: -99.0; "), # Octave-band sound power level at 63 Hz (dB) (22-6h)
      QgsField("LWn125",  QVariant.Double, "double", comment = "default: -99.0; "), # Octave-band sound power level at 125 Hz (dB) (22-6h)
      QgsField("LWn250",  QVariant.Double, "double", comment = "default: -99.0; "), # Octave-band sound power level at 250 Hz (dB) (22-6h)
      QgsField("LWn500",  QVariant.Double, "double", comment = "default: -99.0; "), # Octave-band sound power level at 500 Hz (dB) (22-6h)
      QgsField("LWn1000", QVariant.Double, "double", comment = "default: -99.0; "), # Octave-band sound power level at 1000 Hz (dB) (22-6h)
      QgsField("LWn2000", QVariant.Double, "double", comment = "default: -99.0; "), # Octave-band sound power level at 2000 Hz (dB) (22-6h)
      QgsField("LWn4000", QVariant.Double, "double", comment = "default: -99.0; "), # Octave-band sound power level at 4000 Hz (dB) (22-6h)
      QgsField("LWn8000", QVariant.Double, "double", comment = "default: -99.0; "), # Octave-band sound power level at 8000 Hz (dB) (22-6h)  
      QgsField("Yaw", QVariant.Double, "double", comment = "default: 0.0; "), # Yaw angle (degrees)
      QgsField("Pitch", QVariant.Double, "double", comment = "default: 0.0; "), # Pitch angle (degrees)
      QgsField("Roll", QVariant.Double, "double", comment = "default: 0.0; "), # Roll angle (degrees)
      QgsField("Dir_ID", QVariant.Int, "integer", comment = "default: 0; "), # Direction ID (OMNIDIRECTIONAL(0), ROLLING(1), TRACTIONA(2), TRACTIONB(3), AERODYNAMICA(4), AERODYNAMICB(5), BRIDGE(6))
    ])
  
  def initAlgorithm(self, config):
    (_, target_crs) = self.UTIL.getExtentAndCrsUsingCanvas()
    self.UTIL.setDefaultValue("TARGET_CRS", target_crs.authid())
    self.UTIL.initParameters()
    
  def processAlgorithm(self, parameters, context, feedback):
    
    self.UTIL.registerProcessingParameters(parameters, context, feedback)
    self.CURRENT_PROCESS = self.UTIL.parseCurrentProcess()
    
    # Re-project the input layer to the target CRS
    target_crs = self.parameterAsCrs(parameters, "TARGET_CRS", context)
    self.UTIL.checkCrsAsCartesian(target_crs)

    try:
      input_lyr_proj = processing.run(
        "native:reprojectlayer", 
        {
          "INPUT": self.parameterAsSource(parameters, "INPUT", context).materialize(QgsFeatureRequest(), feedback),
          "TARGET_CRS": target_crs,
          "OUTPUT": "TEMPORARY_OUTPUT"
        },
        context = context,
        is_child_algorithm = True
      )["OUTPUT"]      
      input_lyr_proj = context.getMapLayer(input_lyr_proj)
            
    except:
      input_lyr_proj = QgsVectorLayer(f"Point?crs={target_crs.authid()}", "temp", "memory")
    
    # Set Z values
    if self.parameterAsBoolean(parameters, "USE_ZVALUE", context):
      if not QgsWkbTypes.hasZ(input_lyr_proj.wkbType()):
        feedback.reportError(self.tr("The input layer does not have z values. Please check the input layer."))
        raise Exception("The input layer does not have z values.")
      input_lyr_fixed = input_lyr_proj
    else:
      input_lyr_fixed = processing.run(
        "native:setzvalue",
        {
          "INPUT": input_lyr_proj,
          "Z_VALUE": self.parameterAsDouble(parameters, "Z_VALUE", context),
          "OUTPUT": "TEMPORARY_OUTPUT"
        },
        context = context
      )["OUTPUT"]
      
    try:
      self.FIELDS["EXISTING"] = HrFields.fromQgsFields(
        input_lyr_fixed.fields()
      ).setComments("note: existing fields; ", append = False)
    except:
      feedback.reportError(self.tr("The fields of input layer is invalid. The names may be duplicated (case-insensitive)."))
      raise Exception
    
    # Set the fields with default values
    overwrite = self.parameterAsBoolean(parameters, "OVERWRITE", context)
    n = input_lyr_fixed.featureCount()
    
    # the common fields are always overwritten
    self.FIELDS_WITH_VALUES = {
      "PK": {
        "type": QVariant.Int, 
        "value": list(range(1, n+1))
      },
      "HISTORY": {
        "type": QVariant.String, 
        "value": self.CURRENT_PROCESS,
        "append": True
      }
    }
    
    # the required fields are added, if not existing and overwrite flag is not set
    for fld, defval in zip(self.FIELDS["REQUIRED"], self.FIELDS["REQUIRED"].defaultValues()):
      if fld.name().lower() in self.FIELDS["EXISTING"].lowerNames() and not overwrite:
          feedback.pushWarning(self.tr("Existing field is used: ") + fld.name())
      else:
        self.FIELDS_WITH_VALUES[fld.name()] = {
          "type": fld.type(),
          "value": defval
        }
        
    # Display the results
    feedback.pushInfo(self.tr("The following fields are initialized: "))
    feedback.pushInfo("; ".join([str(key) for key in self.FIELDS_WITH_VALUES.keys()]))
        
    dest_id = self.UTIL.outputVectorLayer(
      vector_layer= input_lyr_fixed,
      param_sink = "OUTPUT",
      fields_with_values= self.FIELDS_WITH_VALUES
    )
    
    PostProcessors[dest_id] = HrPostProcessor(history = [self.CURRENT_PROCESS])
              
    return {"OUTPUT": dest_id}
  
  def name(self):
    return self.__class__.__name__

  def displayName(self):
    return self.tr("Sound source")

  def group(self):
    return self.tr("Initialize features")

  def groupId(self):
    return "initfeature"

  def createInstance(self):
    return initsource()

  # placing here is necessary, when employing pylupdate
  def tr(self, string):
    return QCoreApplication.translate(self.__class__.__name__, string)
