from qgis.PyQt.QtCore import (QT_TRANSLATE_NOOP, QVariant, QCoreApplication)
from qgis.core import (
  QgsProcessing,
  QgsProcessingAlgorithm,
  QgsProcessingParameterFeatureSource,
  QgsProcessingParameterFeatureSink,
  QgsProcessingParameterBoolean,
  QgsFeatureRequest,
  QgsProcessingParameterCrs,
  QgsVectorLayer,
  QgsField,
  QgsWkbTypes,
  QgsProcessingParameterNumber
  )


from qgis import processing

from ..algutil.hriskvar import PostProcessors
from ..algutil.hriskutil import HrUtil
from ..algutil.hriskfields import HrFields
from ..algutil.hriskpostprocessor import HrPostProcessor

class initroad(QgsProcessingAlgorithm):
  PARAMETERS = {                  
    "INPUT": {
      "ui_func": QgsProcessingParameterFeatureSource,
      "ui_args":{
        "description": QT_TRANSLATE_NOOP("initroad","Line layer"),
        "types": [QgsProcessing.TypeVectorLine],
        "optional": True
      }
    },
    "USE_ZVALUE": {
      "ui_func": QgsProcessingParameterBoolean,
      "ui_args":{
        "description" : QT_TRANSLATE_NOOP("initroad","Use z values of the geometry?"),
        "defaultValue": False
      }
    },
    "Z_VALUE": {
      "ui_func": QgsProcessingParameterNumber,
      "ui_args":{
        "description" : QT_TRANSLATE_NOOP("initroad","Default z value (if current z value is not used)"),
        "defaultValue": 0.05
      }
    },
    "TARGET_CRS": {
      "ui_func": QgsProcessingParameterCrs,
      "ui_args": {
        "description": QT_TRANSLATE_NOOP("initroad","Target CRS (Cartesian coordinates)")
      }
    },
    "OVERWRITE": {
      "ui_func": QgsProcessingParameterBoolean,
      "ui_args":{
        "description" : QT_TRANSLATE_NOOP("initroad","Overwrite existing fields?"),
        "defaultValue": True
      }
    },
    "OUTPUT": {
      "ui_func": QgsProcessingParameterFeatureSink,
      "ui_args": {
        "description": QT_TRANSLATE_NOOP("initroad","Road" )
      }
    }
  }
  
  FIELDS = {}
  FIELDS_WITH_VALUES = {}
  
  def __init__(self):
    super().__init__()
    self.UTIL = HrUtil(self)
    
    self.FIELDS["COMMON"] = HrFields.fromQgsFieldList([
      QgsField("PK", QVariant.Int, "int"),
      QgsField("HISTORY", QVariant.String, "string")      
    ]).setComments("note: common fields; ")
    
    self.FIELDS["REQUIRED"] = HrFields.fromQgsFieldList([
      QgsField("LV_d",      QVariant.Double, "double", comment = "default: 0.0; "), # Hourly average light vehicle count (6-18h)
      QgsField("LV_e",      QVariant.Double, "double", comment = "default: 0.0; "), # Hourly average light vehicle count (18-22h)
      QgsField("LV_n",      QVariant.Double, "double", comment = "default: 0.0; "), # Hourly average light vehicle count (22-6h)
      QgsField("MV_d",      QVariant.Double, "double", comment = "default: 0.0; "), # Hourly average medium heavy vehicle count (6-18h)
      QgsField("MV_e",      QVariant.Double, "double", comment = "default: 0.0; "), # Hourly average medium heavy vehicle count (18-22h)
      QgsField("MV_n",      QVariant.Double, "double", comment = "default: 0.0; "), # Hourly average medium heavy vehicle count (22-6h)
      QgsField("HGV_d",     QVariant.Double, "double", comment = "default: 0.0; "), # Hourly average heavy duty vehicles count (6-18h)
      QgsField("HGV_e",     QVariant.Double, "double", comment = "default: 0.0; "), # Hourly average heavy duty vehicles count (18-22h)
      QgsField("HGV_n",     QVariant.Double, "double", comment = "default: 0.0; "), # Hourly average heavy duty vehicles count (22-6h)
      QgsField("WAV_d",     QVariant.Double, "double", comment = "default: 0.0; "), # Hourly average mopeds < 50cc count (6-18h)
      QgsField("WAV_e",     QVariant.Double, "double", comment = "default: 0.0; "), # Hourly average mopeds < 50cc count (18-22h)
      QgsField("WAV_n",     QVariant.Double, "double", comment = "default: 0.0; "), # Hourly average mopeds < 50cc count (22-6h)
      QgsField("WBV_d",     QVariant.Double, "double", comment = "default: 0.0; "), # Hourly average motorcycle > 50cc count (6-18h)
      QgsField("WBV_e",     QVariant.Double, "double", comment = "default: 0.0; "), # Hourly average motorcycle > 50cc count (18-22h)
      QgsField("WBV_n",     QVariant.Double, "double", comment = "default: 0.0; "), # Hourly average motorcycle > 50cc count (22-6h)
      QgsField("LV_spd_d",  QVariant.Double, "double", comment = "default: 60.0; "), # Hourly average light vehicle speed (6-18h)
      QgsField("LV_spd_e",  QVariant.Double, "double", comment = "default: 60.0; "), # Hourly average light vehicle speed (18-22h)
      QgsField("LV_spd_n",  QVariant.Double, "double", comment = "default: 60.0; "), # Hourly average light vehicle speed (22-6h)
      QgsField("MV_spd_d",  QVariant.Double, "double", comment = "default: 60.0; "), # Hourly average medium heavy vehicle speed (6-18h)
      QgsField("MV_spd_e",  QVariant.Double, "double", comment = "default: 60.0; "), # Hourly average medium heavy vehicle speed (18-22h)
      QgsField("MV_spd_n",  QVariant.Double, "double", comment = "default: 60.0; "), # Hourly average medium heavy vehicle speed (22-6h)
      QgsField("HGV_spd_d", QVariant.Double, "double", comment = "default: 60.0; "), # Hourly average heavy duty vehicles speed (6-18h)
      QgsField("HGV_spd_e", QVariant.Double, "double", comment = "default: 60.0; "), # Hourly average heavy duty vehicles speed (18-22h)
      QgsField("HGV_spd_n", QVariant.Double, "double", comment = "default: 60.0; "), # Hourly average heavy duty vehicles speed (22-6h)
      QgsField("WAV_spd_d", QVariant.Double, "double", comment = "default: 60.0; "), # Hourly average mopeds < 50cc speed (6-18h)
      QgsField("WAV_spd_e", QVariant.Double, "double", comment = "default: 60.0; "), # Hourly average mopeds < 50cc speed (18-22h)
      QgsField("WAV_spd_n", QVariant.Double, "double", comment = "default: 60.0; "), # Hourly average mopeds < 50cc speed (22-6h)
      QgsField("WBV_spd_d", QVariant.Double, "double", comment = "default: 60.0; "), # Hourly average motorcycle > 50cc speed (6-18h)
      QgsField("WBV_spd_e", QVariant.Double, "double", comment = "default: 60.0; "), # Hourly average motorcycle > 50cc speed (18-22h)
      QgsField("WBV_spd_n", QVariant.Double, "double", comment = "default: 60.0; "), # Hourly average motorcycle > 50cc speed (22-6h)
      QgsField("LWd63",     QVariant.Double, "double", comment = "default: -99.0; "), # Octave-band sound power level at 63 Hz (dB) (6-18h)
      QgsField("LWd125",    QVariant.Double, "double", comment = "default: -99.0; "), # Octave-band sound power level at 125 Hz (dB) (6-18h)
      QgsField("LWd250",    QVariant.Double, "double", comment = "default: -99.0; "), # Octave-band sound power level at 250 Hz (dB) (6-18h)
      QgsField("LWd500",    QVariant.Double, "double", comment = "default: -99.0; "), # Octave-band sound power level at 500 Hz (dB) (6-18h)
      QgsField("LWd1000",   QVariant.Double, "double", comment = "default: -99.0; "), # Octave-band sound power level at 1000 Hz (dB) (6-18h)
      QgsField("LWd2000",   QVariant.Double, "double", comment = "default: -99.0; "), # Octave-band sound power level at 2000 Hz (dB) (6-18h)
      QgsField("LWd4000",   QVariant.Double, "double", comment = "default: -99.0; "), # Octave-band sound power level at 4000 Hz (dB) (6-18h)
      QgsField("LWd8000",   QVariant.Double, "double", comment = "default: -99.0; "), # Octave-band sound power level at 8000 Hz (dB) (6-18h)
      QgsField("LWe63",     QVariant.Double, "double", comment = "default: -99.0; "), # Octave-band sound power level at 63 Hz (dB) (18-22h)
      QgsField("LWe125",    QVariant.Double, "double", comment = "default: -99.0; "), # Octave-band sound power level at 125 Hz (dB) (18-22h)
      QgsField("LWe250",    QVariant.Double, "double", comment = "default: -99.0; "), # Octave-band sound power level at 250 Hz (dB) (18-22h)
      QgsField("LWe500",    QVariant.Double, "double", comment = "default: -99.0; "), # Octave-band sound power level at 500 Hz (dB) (18-22h)
      QgsField("LWe1000",   QVariant.Double, "double", comment = "default: -99.0; "), # Octave-band sound power level at 1000 Hz (dB) (18-22h)
      QgsField("LWe2000",   QVariant.Double, "double", comment = "default: -99.0; "), # Octave-band sound power level at 2000 Hz (dB) (18-22h)
      QgsField("LWe4000",   QVariant.Double, "double", comment = "default: -99.0; "), # Octave-band sound power level at 4000 Hz (dB) (18-22h)
      QgsField("LWe8000",   QVariant.Double, "double", comment = "default: -99.0; "), # Octave-band sound power level at 8000 Hz (dB) (18-22h)
      QgsField("LWn63",     QVariant.Double, "double", comment = "default: -99.0; "), # Octave-band sound power level at 63 Hz (dB) (22-6h)
      QgsField("LWn125",    QVariant.Double, "double", comment = "default: -99.0; "), # Octave-band sound power level at 125 Hz (dB) (22-6h)
      QgsField("LWn250",    QVariant.Double, "double", comment = "default: -99.0; "), # Octave-band sound power level at 250 Hz (dB) (22-6h)
      QgsField("LWn500",    QVariant.Double, "double", comment = "default: -99.0; "), # Octave-band sound power level at 500 Hz (dB) (22-6h)
      QgsField("LWn1000",   QVariant.Double, "double", comment = "default: -99.0; "), # Octave-band sound power level at 1000 Hz (dB) (22-6h)
      QgsField("LWn2000",   QVariant.Double, "double", comment = "default: -99.0; "), # Octave-band sound power level at 2000 Hz (dB) (22-6h)
      QgsField("LWn4000",   QVariant.Double, "double", comment = "default: -99.0; "), # Octave-band sound power level at 4000 Hz (dB) (22-6h)
      QgsField("LWn8000",   QVariant.Double, "double", comment = "default: -99.0; "), # Octave-band sound power level at 8000 Hz (dB) (22-6h)
      QgsField("pvmt",      QVariant.String, "string", comment = "default: DEF; "), # CNOSSOS road pavement identifier
      QgsField("temp_d",    QVariant.Double, "double", comment = "default: 20.0; "), # Average temperature (6-18h)
      QgsField("temp_e",    QVariant.Double, "double", comment = "default: 20.0; "), # Average temperature (18-22h)
      QgsField("temp_n",    QVariant.Double, "double", comment = "default: 20.0; "), # Average temperature (22-6h)
      QgsField("ts_stud",   QVariant.Double, "double", comment = "default: 0.0; "), # A limited period Ts (in months) over the year where a average proportion pm_stud of light vehicles are equipped with studded tyres (6-18h)
      QgsField("pm_stud",   QVariant.Double, "double"), # Average proportion of vehicles equipped with studded tyres during ts_stud period
      QgsField("junc_dist", QVariant.Double, "double", comment = "default: 999.0; "), # Distance to junction in meters
      QgsField("junc_type", QVariant.Int,    "int",    comment = "default: 0; "), # Type of junction (k=0 none, k = 1 for a crossing with traffic lights ; k = 2 for a roundabout)
      QgsField("slope",     QVariant.Double, "double", comment = "default: 0.0; "), # Slope (in %) of the road section. If the field is not filled in, the LINESTRING z-values will be used
      QgsField("way",       QVariant.Int   , "int",    comment = "default: 3; ") # the way of the road section. 1 = one way road section and the traffic goes in the same way that the slope definition you have used, 2 = one way road section and the traffic goes in the inverse way that the slope definition you have used, 3 = bi-directional traffic flow, the flow is split into two components and correct half for uphill and half for downhill
    ])
    
  # LD: light vehicle
  # MD: medium heavy vehicle, delivery vans > 3.5 tons,  buses, touring cars, etc. with two axles and twin tyre mounting on rear axle
  # HGV: heavy duty vehicles, touring cars, buses, with three or more axles
  # WAV: mopeds, tricycles or quads < 50 cc
  # WBV: motorcycles, tricycles or quads > 50 cc
  
  def initAlgorithm(self, config):
    (_, target_crs) = self.UTIL.getExtentAndCrsUsingCanvas()
    self.UTIL.setDefaultValue("TARGET_CRS", target_crs.authid())
    self.UTIL.initParameters()
    
  def processAlgorithm(self, parameters, context, feedback):
    self.UTIL.registerProcessingParameters(parameters, context, feedback)
    self.CURRENT_PROCESS = self.UTIL.parseCurrentProcess()
    
    # Re-project the input layer to the target CRS
    target_crs = self.parameterAsCrs(parameters, "TARGET_CRS", context)
    self.UTIL.checkCrsAsCartesian(target_crs)
    

    try:
      input_lyr = self.parameterAsSource(parameters, "INPUT", context).materialize(QgsFeatureRequest(), feedback)
      input_lyr_proj = processing.run(
        "native:reprojectlayer", 
        {
          "INPUT": input_lyr,
          "TARGET_CRS": target_crs,
          "OUTPUT": "TEMPORARY_OUTPUT"
        },
        context = context,
        is_child_algorithm = True
      )["OUTPUT"]      
      input_lyr_proj = context.getMapLayer(input_lyr_proj)
            
    except:
      input_lyr_proj = QgsVectorLayer(f"Polygon?crs={target_crs.authid()}", "temp", "memory")
      
    
    # Set Z values
    if self.parameterAsBoolean(parameters, "USE_ZVALUE", context):
      if not QgsWkbTypes.hasZ(input_lyr_proj.wkbType()):
        feedback.reportError(self.tr("The input layer does not have z values. Please check the input layer."))
        raise Exception("The input layer does not have z values.")
      input_lyr_fixed = input_lyr_proj
    else:
      input_lyr_fixed = processing.run(
        "native:setzvalue",
        {
          "INPUT": input_lyr_proj,
          "Z_VALUE": 0.05,
          "OUTPUT": "TEMPORARY_OUTPUT"
        },
        context = context
      )["OUTPUT"]
    
    # Import the fields of the input layer
    try:
      self.FIELDS["EXISTING"] = HrFields.fromQgsFields(
        input_lyr_fixed.fields()
      ).setComments("note: existing fields; ", append = False)
    except:
      feedback.reportError(self.tr("The fields of input layer is invalid. The names may be duplicated (case-insensitive)."))
      raise Exception
    
    # Set the fields with default values
    overwrite = self.parameterAsBoolean(parameters, "OVERWRITE", context)
    n = input_lyr_fixed.featureCount()
    
    # the common fields are always overwritten
    self.FIELDS_WITH_VALUES = {
      "PK": {
        "type": QVariant.Int, 
        "value": list(range(1, n+1))
      },
      "HISTORY": {
        "type": QVariant.String, 
        "value": self.CURRENT_PROCESS,
        "append": True
      }
    }
    
    # the required fields are added, if not existing and overwrite flag is not set
    for fld, defval in zip(self.FIELDS["REQUIRED"], self.FIELDS["REQUIRED"].defaultValues()):
      if fld.name().lower() in self.FIELDS["EXISTING"].lowerNames() and not overwrite:
          feedback.pushWarning(self.tr("Existing field is used: ") + fld.name())
      else:
        self.FIELDS_WITH_VALUES[fld.name()] = {
          "type": fld.type(),
          "value": defval
        }

    # Display the results
    feedback.pushInfo(self.tr("The following fields are initialized: "))
    feedback.pushInfo("; ".join([str(key) for key in self.FIELDS_WITH_VALUES.keys()]))
        
    dest_id = self.UTIL.outputVectorLayer(
      vector_layer= input_lyr_fixed,
      param_sink = "OUTPUT",
      fields_with_values= self.FIELDS_WITH_VALUES
    )
    
    PostProcessors[dest_id] = HrPostProcessor(history = [self.CURRENT_PROCESS])
              
    return {"OUTPUT": dest_id}
  
  def name(self):
    return self.__class__.__name__

  def displayName(self):
    return self.tr("Road")

  def group(self):
    return self.tr("Initialize features")

  def groupId(self):
    return "initfeature"

  def createInstance(self):
    return initroad()

  # placing here is necessary, when employing pylupdate
  def tr(self, string):
    return QCoreApplication.translate(self.__class__.__name__, string)
