/**
 * Utility class for Noise Modelling in Groovy
 * 
 * This class provides methods for managing noise modeling calculations,
 * including database operations and script execution.
 * 
 * Dependencies:
 * - H2GIS database for spatial data processing
 * - NoiseModelling framework for acoustic calculations
 * - SLF4J and Log4j for logging
 * 
 * @author Junta Tagusari
 * @version 1.0
 */

// Import required libraries for database connectivity, file handling, and logging
import org.h2gis.api.ProgressVisitor;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import java.sql.Connection;
import java.nio.file.Path;
import java.nio.file.Paths;
import org.apache.log4j.PropertyConfigurator;
import java.io.FileWriter;
import java.io.PrintWriter;


/**
 * NoiseModelling utility class for managing database operations and script execution
 * This class encapsulates all the functionality needed for noise modeling calculations
 */

class NoiseModelling {

  // Database connection for H2GIS spatial database
  Connection connection
  // Path to NoiseModelling installation (currently unused)
  Path noiseModellingPath
  
  /**
   * Constructor for NoiseModelling class
   * @param connection Database connection to H2GIS spatial database
   */
  NoiseModelling(Connection connection) {
    this.connection = connection
    initializeLogging()
  }

  /**
   * Initialize Log4j logging configuration
   * Creates a temporary configuration file and sets up both file and console logging
   * This ensures all processing steps are properly logged for debugging and monitoring
   */
  private void initializeLogging() {
    def tempConfigFile = File.createTempFile("log4j", ".properties")
    def logFilePath = new File("hrisk_application.log").absolutePath.replace("\\", "/")
    
    // Configure Log4j with both file and console appenders
    tempConfigFile.text = """
# File appender configuration
log4j.appender.file=org.apache.log4j.FileAppender
log4j.appender.file.File=${logFilePath}
log4j.appender.file.Append=true
log4j.appender.file.layout=org.apache.log4j.PatternLayout
log4j.appender.file.layout.ConversionPattern=[%t] %-5p %d{yyyy-MM-dd HH:mm:ss} - %m%n

# Console appender configuration
log4j.appender.stdout=org.apache.log4j.ConsoleAppender
log4j.appender.stdout.Target=System.out
log4j.appender.stdout.layout=org.apache.log4j.PatternLayout
log4j.appender.stdout.layout.ConversionPattern=[%t] %-5p %d{yyyy-MM-dd HH:mm:ss} - %m%n

# Root logger configuration - output to both file and console
log4j.rootLogger=INFO, file, stdout

# Specific logger configurations - ensure they write to file
log4j.logger.org.hrisk=INFO, file, stdout
log4j.logger.org.noise_planet=INFO, file, stdout
log4j.logger.org.noisemodelling=INFO, file, stdout  
log4j.logger.org.h2=INFO, file
log4j.logger.org.cts=ERROR, file

# Prevent duplicate logging
log4j.additivity.org.hrisk=false
log4j.additivity.org.noise_planet=false
log4j.additivity.org.noisemodelling=false
log4j.additivity.org.h2=false
log4j.additivity.org.cts=false
""".stripIndent()

    // Apply the logging configuration
    System.setProperty("log4j.configuration", "file:" + tempConfigFile.absolutePath)
    PropertyConfigurator.configure(tempConfigFile.absolutePath)

    // Test the logging configuration
    Logger testLogger = LoggerFactory.getLogger("org.hrisk")
    testLogger.info("Log configuration initialized - file path: " + logFilePath)
  }
    
  /**
   * Execute a Groovy script with the given arguments
   * @param scriptPath Relative path to the script from the current script directory
   * @param arguments Map of arguments to pass to the script
   * @return Result object returned by the executed script
   */
  def runScript(scriptPath, arguments) {
    Logger logger = LoggerFactory.getLogger("org.hrisk")
    
    try {        
      Path currentScriptDir = Paths.get(getClass().protectionDomain.codeSource.location.toURI()).parent
      String scriptWithExtension = scriptPath.endsWith(".groovy") ? scriptPath : scriptPath + ".groovy"
      
      Path scriptPathObj = currentScriptDir.resolve(scriptWithExtension)
      logger.info("Running script: ${scriptPathObj}\n")
      
      // Check if the script file exists
      if (!scriptPathObj.toFile().exists()) {
        def errorMsg = "Script file not found: ${scriptPathObj}"
        logger.error(errorMsg)
        throw new FileNotFoundException(errorMsg)
      }
        
      // Create a new Groovy shell and parse the script
      GroovyShell shell = new GroovyShell()
      Script scriptInstance = shell.parse(new File(scriptPathObj.toString()))
      Binding binding = new Binding();
      scriptInstance.setBinding(binding);
      scriptInstance.run();
      
      // Get the input parameters defined in the target script
      def scriptInputs = binding.getVariables().get("inputs");
      
      // Filter out unknown arguments to prevent errors
      def keysToRemove = []
      arguments.each{ k, v -> 
        if(scriptInputs[k] == null) {
          keysToRemove.add(k)
          logger.warn("Ignored unknown argument: " + k + " = " + v)
        } 
      }
      
      // Remove unknown arguments from the arguments map
      keysToRemove.each { key ->
        arguments.remove(key)
      }
      
      // Execute the script's main function with database connection and arguments
      Object result = scriptInstance.invokeMethod("exec", [connection, arguments])
      if(result != null) {
        def resultMsg = result.toString()
        logger.info(resultMsg)
      }
      return result
    } catch (Exception e) {
      logger.error("Error executing script ${scriptPath}: ${e.getMessage()}", e)
      throw e
    } finally {
      // Cleanup completed - no additional resources to close
    }
  }

  /**
   * Clear all tables from the database
   * This method executes the Clean_Database script to remove existing data
   */
  def clearDatabase() {
    Logger logger = LoggerFactory.getLogger("org.hrisk")
    try {
      runScript("Database_Manager/Clean_Database", ["areYouSure": true])
      logger.info("Database cleared successfully.")
    } catch (Exception e) {
      logger.error("Error clearing database: ${e.getMessage()}", e)
      throw e
    }
  }

  /**
   * Import a spatial data file into the database
   * @param pathFile Path to the input file (supports various formats: GeoJSON, Shapefile, CSV, etc.)
   * @param inputSRID Spatial Reference System Identifier for the input data
   * @return String The name of the created table in uppercase
   */
  def importTable(pathFile, inputSRID){
    String bname = null
    def logger = LoggerFactory.getLogger("org.hrisk")
    
    if (pathFile) {
      File f = new File(pathFile)
      // Check if the input file exists
      if (!f.exists()) {
        logger.error("Input file does not exist: ${pathFile}")
        throw new FileNotFoundException("Input file does not exist: ${pathFile}")
      }
        
      // Determine the file format based on extension
      String extension = f.getName().toLowerCase().substring(f.getName().lastIndexOf('.') + 1)
      List<String> vectorExtensions = ["csv", "dbf", "geojson", "gpx", "bz2", "gz", "osm", "shp", "tsv"]
      List<String> rasterExtensions = ["asc"]

      // Import vector data files
      if (vectorExtensions.contains(extension)) {
        runScript(
          "Import_and_Export/Import_File",
          ["pathFile":pathFile, "inputSRID": inputSRID]
        )
      // Import raster data files  
      } else if (rasterExtensions.contains(extension)) {
        runScript(
          "Import_and_Export/Import_Asc_File",
          ["pathFile":pathFile, "inputSRID": inputSRID]
        )
      } else {
        throw new IllegalArgumentException("Unsupported file format: " + extension)
      }

      // Extract table name from filename (without extension, in uppercase)
      bname = f.getName()
      bname = bname.substring(0,bname.lastIndexOf('.')).toUpperCase()
    }
    
    return bname
  }

  /**
   * Export a database table to a GeoJSON file
   * @param table Name of the table to export
   * @param exportDir Directory where the exported file will be saved
   * @return String Full path to the exported file
   */
  def exportTables(table, exportDir){
    Path p_result = Paths.get(exportDir).resolve(Paths.get(table + ".geojson"))
    runScript(
      "Import_and_Export/Export_Table",
      ["exportPath": p_result, "tableToExport":table]
    )

    return p_result.toString()
  }

  /**
   * Get a list of all table names in the current database
   * @return List<String> List of table names
   */
  def tableNames() {
    def table_names_str = runScript("Database_Manager/Display_Database",[])
    if (table_names_str == null || table_names_str.toString().trim().isEmpty()) {
      return []
    }

    // Parse the HTML-formatted table names string returned by the Display_Database script
    def table_names = table_names_str.split("</br></br>")
      .findAll { it != null && !it.trim().isEmpty() }
      .collect { it.trim() }

      return table_names
    }
    
  def parseBoolean(value) {
    if (value instanceof Boolean) {
      return value
    } else if (value instanceof String) {
      return value.equalsIgnoreCase("true") || value.equalsIgnoreCase("1")
    } else if (value instanceof Integer) {
      return value > 0
    }
    return false
  }
}