from qgis.PyQt.QtCore import (
  QCoreApplication,QVariant
  )

from qgis.core import (
  QgsVectorLayer,
  QgsRectangle,
  QgsProcessingParameterDefinition, 
  QgsProcessingUtils,
  QgsVectorFileWriter,
  QgsCoordinateTransformContext,
  QgsFeatureRequest,
  QgsCoordinateReferenceSystem,
  QgsProcessingFeedback,
  QgsProcessingContext,
  QgsReferencedRectangle,
  QgsCoordinateTransform,
  QgsProject,
  QgsFields,
  QgsField,
  QgsFeature,
  QgsProcessingParameterExtent,
  )

import os
import re
import shutil
import datetime
import zipfile
import json
import concurrent
import requests
import itertools

from qgis.utils import iface
from qgis import processing
from copy import deepcopy

class HrUtil(object):

  PARENT = None
  PARENT_PARAMETERS = None
  PARENT_CONTEXT = None
  PARENT_FEEDBACK = None
  
  def __init__(self, parent, parent_parameters:dict = {}, parent_context: QgsProcessingContext= None, parent_feedback: QgsProcessingFeedback = None) -> None:
    self.PARENT = parent
    self.PARENT_PARAMETERS = parent_parameters
    self.PARENT_CONTEXT = parent_context
    self.PARENT_FEEDBACK = parent_feedback
    self.SINK = None
    self.DEST_ID = None
  
  def registerProcessingParameters(self, parent_parameters:dict, parent_context: QgsProcessingContext, parent_feedback: QgsProcessingFeedback):
    self.PARENT_PARAMETERS = parent_parameters
    self.PARENT_CONTEXT = parent_context
    self.PARENT_FEEDBACK = parent_feedback
  
  def asQgsReferencedRectangle(self, extent_str: str, target_crs: QgsCoordinateReferenceSystem) -> QgsReferencedRectangle:
    """
    Parses a string in the format "{xmin},{xmax},{ymin},{ymax} [EPSG:{auth_id}]"
    and returns a QgsReferencedRectangle object.

    Args:
      extent_str (str): The extent string to parse.
      target_crs (QgsCoordinateReferenceSystem): The target CRS.

    Returns:
      QgsReferencedRectangle: The parsed rectangle with the specified CRS.
    """
    try:
      match = re.match(r"(-?\d+\.?\d*),(-?\d+\.?\d*),(-?\d+\.?\d*),(-?\d+\.?\d*) \[EPSG:(\d+)\]", extent_str)
      if not match:
        raise ValueError(f"Invalid extent string format: {extent_str}")

      xmin, xmax, ymin, ymax = map(float, match.groups()[:4])

      # CRS を作成
      crs = QgsCoordinateReferenceSystem(f"EPSG:{match.group(5)}")
      if not crs.isValid():
        raise ValueError(f"Invalid CRS: EPSG:{match.group(5)}")
        
      rect = QgsReferencedRectangle(QgsRectangle(xmin, ymin, xmax, ymax), crs)
      rect_tr = self.transformExtent(rect, crs, target_crs)

      return rect_tr
    except Exception as e:
      raise ValueError(f"Failed to parse extent string: {extent_str}. Error: {e}")
  
  def transformExtent(
    self,
    extent:QgsReferencedRectangle, 
    source_crs:QgsCoordinateReferenceSystem, 
    destination_crs:QgsCoordinateReferenceSystem
    ) -> QgsReferencedRectangle:
    """
    Transforms the extent of a QgsReferencedRectangle from a source CRS to a destination CRS.

    Args:
        extent (QgsReferencedRectangle): The rectangle to transform.
        source_crs (QgsCoordinateReferenceSystem, optional): The source coordinate reference system. 
            Defaults to the CRS of the extent if not provided.
        destination_crs (QgsCoordinateReferenceSystem, optional): The target coordinate reference system. 
            Defaults to the instance's CRS if not provided.

    Returns:
        QgsReferencedRectangle: The transformed rectangle in the destination CRS.
    """
    if source_crs == destination_crs:
      return extent
    
    proj = QgsProject.instance()
    transform_context = proj.transformContext()
    if transform_context.coordinateOperations():
      tr1 = QgsCoordinateTransform(source_crs, proj.crs(), transform_context)
      extent_tmp = QgsReferencedRectangle(tr1.transformBoundingBox(extent), proj.crs())    
      tr2 = QgsCoordinateTransform(proj.crs(), destination_crs, transform_context)
      extent_tr = QgsReferencedRectangle(tr2.transformBoundingBox(extent_tmp), destination_crs)
    else:
      self.PARENT_FEEDBACK.pushWarning(self.tr("Transform cannot be obtained from current project. Default transformation is applied."))
      transform_context = QgsCoordinateTransformContext()
      tr = QgsCoordinateTransform(source_crs, destination_crs, QgsCoordinateTransformContext())
      extent_tr = QgsReferencedRectangle(tr.transformBoundingBox(extent), destination_crs)
    return extent_tr
  
  
  @staticmethod
  def getPluginName(meta_path:str = os.path.join(os.path.dirname(os.path.dirname(__file__)), "metadata.txt")) -> str:
    try:
      with open(meta_path) as meta:
        version_match = re.search(r"name=(.*)", meta.read())
        return version_match.group(1)
    except:
      return "unknown"
  
  @staticmethod
  def getPluginVersion(meta_path:str = os.path.join(os.path.dirname(os.path.dirname(__file__)), "metadata.txt")) -> str:
    try:
      with open(meta_path) as meta:
        version_match = re.search(r"version=(\S+)", meta.read())
        return version_match.group(1)
    except:
      return "unknown"
  
  @staticmethod
  def getNoiseModellingVersion() -> str:
    try:
      nm_home = os.environ["NOISEMODELLING_HOME"]
    except:
      raise Exception("Environment variable of NOISEMODELLING_HOME is not set.")
    
    try:
      for jarfile in os.listdir(os.path.join(nm_home, "lib")):
        if jarfile.startswith("noisemodelling-jdbc-"):
          with zipfile.ZipFile(os.path.join(nm_home, "lib", jarfile), 'r') as jar:
            with jar.open('META-INF/MANIFEST.MF') as manifest:
              version_match = re.search(r"Bundle-Version: (\S+)", manifest.read().decode())
              return version_match.group(1)
    except:
      return "unknown"

  def parseCurrentProcess(self, with_nm:bool=False):
    
    caller = self.PARENT.__class__.__name__
    plugin_name = self.getPluginName()
    plugin_version = self.getPluginVersion()

    
    hr = caller + f" ({plugin_name} v {plugin_version})"
    dt = datetime.datetime.now().strftime("%Y-%m-%d %H:%M:%S")
    
    if with_nm:        
      nm = "NoiseModelling v " + self.getNoiseModellingVersion()
      return hr + " with " + nm + " at " + dt 
    else:
      return hr + " at " + dt

  def setDefaultValue(self, key, value) -> None:
    assert hasattr(self.PARENT, "PARAMETERS"), "PARAMETERS is not defined in the class."
    self.PARENT.PARAMETERS[key]["ui_args"]["defaultValue"] = value

  def initParameters(self) -> None:

    assert hasattr(self.PARENT, "PARAMETERS"), "PARAMETERS attribute is not defined in the class."
    
    for key, value in self.PARENT.PARAMETERS.items():
      args = value.get("ui_args")
      args["name"] = key
      args["description"] = self.PARENT.tr(args["description"])
              
      ui = value.get("ui_func")(**args)
      
      if value.get("advanced") is not None and value.get("advanced") == True:
        ui.setFlags(QgsProcessingParameterDefinition.FlagAdvanced)
        
      self.PARENT.addParameter(ui)  

  @classmethod
  def checkCrsAsCartesian(cls, crs: QgsCoordinateReferenceSystem) -> None:
    assert crs.isValid(), "The CRS is not valid."
    
    if crs.isGeographic() or crs.authid() == "EPSG:3857":
      raise Exception("The Target CRS is NOT a Cartesian Coordinate System.")

  @classmethod
  def getExtentAndCrsUsingCanvas(cls, get_xy_coords:bool=True) -> tuple[str, QgsCoordinateReferenceSystem]:

    try:
      map_rectangle = iface.mapCanvas().extent()
      proj_crs = iface.mapCanvas().mapSettings().destinationCrs()
      
      use_rect = QgsReferencedRectangle(map_rectangle, proj_crs)

      use_crs = proj_crs
      if get_xy_coords:
        # if the CRS is geographic, input the long/lat to the getUtmCrs
        if proj_crs.isGeographic():
          use_crs = cls.getUtmCrs(map_rectangle.center().x(), map_rectangle.center().y())
        # if the CRS is web-mercator, transform long/lat coordinates and input to the getUtmCrs
        elif proj_crs.authid() == "EPSG:3857":
          transform = QgsCoordinateTransform(proj_crs, QgsCoordinateReferenceSystem("EPSG:4326"), QgsProject.instance())
          map_rectangle = transform.transformBoundingBox(map_rectangle)
          use_crs = cls.getUtmCrs(map_rectangle.center().x(), map_rectangle.center().y())
      
      use_extent = ",".join(
        list(map(lambda x: str(x), [
        use_rect.xMinimum(), use_rect.xMaximum(), use_rect.yMinimum(), use_rect.yMaximum()
        ]))
      ) + f"[{proj_crs.authid()}]"
      
      return use_extent, use_crs
    except:
      return "", QgsCoordinateReferenceSystem()

  @staticmethod
  def getUtmCrs(lng: float, lat: float) -> QgsCoordinateReferenceSystem:
    """
    Gets the CRS of the UTM using longitude and latitude.

    Parameters:
    - lng (float): The longitude.
    - lat (float): The latitude.

    Returns:
    QgsCoordinateReferenceSystem: The UTM CRS.

    Raises:
    None
    """
    epsg_code = 32600 + 100 * (lat < 0) + int((lng + 180) / 6) + 1
    crs = QgsCoordinateReferenceSystem(f'EPSG:{epsg_code}')
    assert crs.isValid(), "The CRS is not valid."
    return crs
  
  def dissolve(self, vl:QgsVectorLayer, field_names_drop:list[str] = ["layer","path"]) -> QgsVectorLayer:
    
    lyr_valid = processing.run(
      "qgis:checkvalidity",
      {
        "INPUT_LAYER": vl,
        "VALID_OUTPUT": "TEMPORARY_OUTPUT"
      },
      context = self.PARENT_CONTEXT,
      is_child_algorithm = True
    )["VALID_OUTPUT"]
    
    field_names_dissolve = [
      fld.name() 
      for fld in self.PARENT_CONTEXT.getMapLayer(lyr_valid).fields() 
      if fld.name() not in field_names_drop
    ]
  
    # Dissolve
    lyr_dissolve = processing.run(
      "native:dissolve", 
      {
        "INPUT": lyr_valid,
        "FIELD": field_names_dissolve,
        "OUTPUT": "TEMPORARY_OUTPUT"
      },
      context = self.PARENT_CONTEXT,
      is_child_algorithm = True
    )["OUTPUT"]
    
    # Multipart to Single parts
    lyr_single = processing.run(
      "native:multiparttosingleparts", 
      {
        "INPUT": lyr_dissolve,
        "OUTPUT": "TEMPORARY_OUTPUT"
      },
      context = self.PARENT_CONTEXT,
      is_child_algorithm = True
    )["OUTPUT"]
    
    return lyr_single
  
  def downloadFile(self, session = None, feedback_progress: bool = False, method:str = "GET", url:str = "", params:dict = None, data:dict = None, json:dict = None, dest:str = None, stream:bool=True, **kwargs:dict) -> str:

    if session is None:
      session = requests.Session()
    try:
      response = session.request(method = method, url = url, params = params, data = data, json = json, stream = stream)
      # if query is not None:
      #   response = session.request(method = method, url = url + query, stream = stream)
      # else:
      #   response = session.request(method = method, url = url, stream = stream)
    except Exception as e:
      return str(e)
    # if download is successful
    if response.status_code == 200:
      file_size = int(response.headers.get('content-length', 0))
      if dest is None:
        try:
          dest = re.search(r'.*\.[^|]*', os.path.basename(url)).group(0)
        except:
          dest = "downloaded_file"
        
      temp_dir = self.createTempDir()
      file_path = os.path.join(temp_dir, dest)
      
      if not os.path.exists(os.path.dirname(file_path)):
        os.makedirs(os.path.dirname(file_path))

      # write downloaded file and show progress
      with open(file_path, 'wb') as file:
        for data in response.iter_content(chunk_size=1024):
          file.write(data)
          if feedback_progress:
            self.PARENT_FEEDBACK.setProgress(int((file.tell() / file_size) * 100))
      
      if feedback_progress:
        self.PARENT_FEEDBACK.setProgress(0)
      
      return file_path
      
    else:
      return Exception("The Link was missing or download was failed.")


  def downloadHeader(self, session = None, feedback_progress: bool = False, method:str = "GET", url:str = "", query:str = None, dest:str = None, stream:bool=True, **kwargs:dict) -> str:
    file_size = -1
    
    if session is None:
      session = requests.Session()
      
    try:
      if query is not None:
        response = session.request(method = method, url = url + query, stream = stream)
      else:
        response = session.request(method = method, url = url, stream = stream)
        
      if response.status_code == 200:
        file_size = int(response.headers.get('content-length', 0)) / 1e6
    except:
      pass
    
    # if download is successful
      
    return file_size


  def downloadHeadersConcurrently(self, session=None, args:dict={}, parallel = True, omit_null = True) -> dict:
    
    mx_wk = None if parallel else 1
    downloaded = {}
    
    if session is None:
      session = requests.Session()
    
    # fetch in parallel
    with concurrent.futures.ThreadPoolExecutor(max_workers=mx_wk) as executor:
      for key, data_size in zip(
        args.keys(),
        executor.map(
        lambda d_arg : self.downloadHeader(
          session, 
          feedback_progress = not parallel,
          **d_arg 
          ), 
        args.values()
        )
      ):
        if self.PARENT_FEEDBACK.isCanceled():
          self.PARENT_FEEDBACK.reportError(self.tr("Data download was canceled."))
          raise Exception(self.tr("Data download was canceled."))
        
        self.PARENT_FEEDBACK.pushInfo(f"({key}) Got info from: {args[key]['url']}")
        
        if omit_null:
          downloaded[key] = data_size
    
    return downloaded

  def downloadFilesConcurrently(self, session=None, args:dict={}, parallel = True, omit_null = True) -> dict:
    """Download files concurrently.

    Args:
        session (_type_, optional): download session. Defaults to None.
        args (dict, optional): download arguments passed to downloadFile. Defaults to {}.
        parallel (bool, optional): whether the download is parallely done. Defaults to True.
        omit_null (bool, optional): _description_. Defaults to True.

    Raises:
        Exception: Data download was canceled.

    Returns:
        dict: paths of the downloaded files
    """
    
    mx_wk = None if parallel else 1
    downloaded = {}
    
    if session is None:
      session = requests.Session()
    
    # fetch in parallel
    with concurrent.futures.ThreadPoolExecutor(max_workers=mx_wk) as executor:
      for key, data_path in zip(
        args.keys(),
        executor.map(
        lambda d_arg : self.downloadFile(
          session, 
          feedback_progress = not parallel,
          **d_arg 
          ), 
        args.values()
        )
      ):
        if self.PARENT_FEEDBACK.isCanceled():
          self.PARENT_FEEDBACK.reportError(self.tr("Data download was canceled."))
          raise Exception(self.tr("Data download was canceled."))
        
        self.PARENT_FEEDBACK.pushInfo(f"({key}) Downloaded from: {args[key]['url']}")
        
        if omit_null and isinstance(data_path, str):
          downloaded[key] = data_path
    
    return downloaded

  @classmethod
  def extractArchive(cls, archive_path: str, extension: str = "", extract_dir: str = None, pattern: str = None) -> list[str]:
    """Extract the archive file.

    Args:
        archive_path (str): path of the archive file
        extension (str, optional): explicit extension of the archive file. Defaults to "".
        extract_dir (str, optional): directory where the archive is extracted. Defaults to None.
        pattern (str, optional): pattern of the file path, only paths matches with which are returned. Defaults to None.

    Returns:
        list[str]: paths of the extracted files
    """

    if extract_dir is None:
      extract_dir = os.path.normpath(os.path.dirname(QgsProcessingUtils.generateTempFilename("")))
    
    if not os.path.exists(extract_dir):
      os.makedirs(extract_dir)      
    
    if len(extension) > 0 and not archive_path.endswith(extension):
      os.rename(archive_path, archive_path + extension)
      
    extracted_files = []
    extracted_dirs = []
    
    try:
      shutil.unpack_archive(archive_path, extract_dir)
      for root, dirs, files in os.walk(extract_dir):
        extracted_dirs.append(root)
        paths = [os.path.join(root, x) for x in files]
        if pattern is None:
          extracted_files += paths
        else:
          for path in paths:
            if re.search(pattern, path):
              extracted_files.append(path)
    except:
      pass # if the file is not an archive file, do nothing
    
    return extracted_dirs, extracted_files
  
  @staticmethod
  def createTempDir() -> str:
    """Create a temporary directory.

    Returns:
        str: path of the temporary directory
    """
    tmp_dir = os.path.normpath(os.path.dirname(QgsProcessingUtils.generateTempFilename("")))
    if not os.path.exists(tmp_dir):
      os.makedirs(tmp_dir)
    
    return tmp_dir
  
      
  @staticmethod
  def saveVectorLayer(vector_layer: QgsVectorLayer, path: str) -> None:
    """Save the vector layer.

    Args:
        vector_layer (QgsVectorLayer): vector layer
        path (str): path to save the vector layer
    """
    save_options = QgsVectorFileWriter.SaveVectorOptions()
    save_options.driverName = "GeoJSON"
    QgsVectorFileWriter.writeAsVectorFormatV3(
      vector_layer, path, QgsCoordinateTransformContext(), save_options
    )
  
  
  @staticmethod
  def newFieldsWithHistory(current_fields: QgsFields) -> QgsFields:
    """Creates new fields with history.

    Args:
        current_fields (QgsFields): current fields

    Returns:
        QgsFields: new fields with history
    """
    # note that one-by-one appending is necessary to remove unsupported field type
    new_fields = QgsFields()
    for fld in current_fields:
      if fld.type() in [QVariant.Int, QVariant.Double, QVariant.String]:
        new_fields.append(fld)
      else:
        new_fields.append(QgsField(fld.name(), QVariant.String))
        
    if "HISTORY" not in [fld.name() for fld in current_fields]:
      new_fields.append(QgsField("HISTORY", QVariant.String))
    return new_fields

  def pivotWider(
    self,
    vector_layer:QgsVectorLayer,
    unique_keys:list[str],
    group_keys:list[str],
    field_name_parser:callable = None
  ):
    
    input_flds = vector_layer.fields()
    
    assert len(unique_keys) > 0, "The field names to be used for the pivoting are not specified."
    assert len(group_keys) > 0, "The field names to be used for the pivoting are not specified."
    
    assert all([uk in input_flds.names() for uk in unique_keys]), "The field names to be used for the pivoting are not found in the input fields."
    
    assert all([gk in input_flds.names() for gk in group_keys]), "The field names to be used for the pivoting are not found in the input fields."
    
    if field_name_parser is None:
      def field_name_parser(group_vals, value_key):
        return "L" + "_".join([str(gv) for gv in group_vals]) + "_" + value_key

    other_flds = QgsFields()
    for fld in input_flds:
      if fld.name() not in unique_keys + group_keys:
        other_flds.append(fld)
        
    # procedures to create a sink
    # 1. create new fields
    # 2. create sink
    # 3. add features to the sink
    
    group_values = []
    for gk in group_keys:
      group_values.append(set([ft[gk] for ft in vector_layer.getFeatures()]))
      
    output_flds = QgsFields()    
    
    for uk in unique_keys:
      output_flds.append(QgsField(uk, input_flds[input_flds.indexOf(uk)].type()))
    
    for fld in other_flds:
      for gv in itertools.product(*group_values):
        output_flds.append(
          QgsField(field_name_parser(gv, fld.name()), fld.type())
        )
    
    (sink, dest_id) = QgsProcessingUtils.createFeatureSink(
      'memory:',
      self.PARENT_CONTEXT,
      output_flds,
      vector_layer.wkbType(),
      vector_layer.crs()
    )
        
    pivot_request = QgsFeatureRequest()
    for uk in unique_keys:
      pivot_request.addOrderBy(uk)
        
    unique_values_tmp = []
    ft_output = None
    for ft_input in vector_layer.getFeatures(pivot_request):
      unique_values_ft = [ft_input[uk] for uk in unique_keys]
      unique_values_ft.append(ft_input.geometry().asWkt())

      # initialize the ft_output by setting the geometry
      if unique_values_ft != unique_values_tmp:
        if ft_output is not None:
          sink.addFeature(ft_output)
        ft_output = QgsFeature(output_flds)
        ft_output.setGeometry(ft_input.geometry())
        unique_values_tmp = unique_values_ft.copy()
      
      # set values to the output feature
      for fld in ft_input.fields():
        if fld.name() in unique_keys:
          ft_output[fld.name()] = ft_input[fld.name()]
        elif fld.name() in other_flds.names():
          group_values = tuple([ft_input[gk] for gk in group_keys])
          new_fld_name = field_name_parser(group_values, fld.name())
          ft_output[new_fld_name] = ft_input[fld.name()]
    
    # output the last feature
    if ft_output is not None:
      sink.addFeature(ft_output)
      
    return dest_id
  
  def outputVectorLayer(
    self,
    vector_layer:str = None,
    param_sink:str = "OUTPUT",
    fields_with_values:dict = {}
  ) -> str:

    
    assert all(["type" in fld.keys() and "value" in fld.keys() for fld in fields_with_values.values()]), "The field type or value is not specified."
    
    n_fts = vector_layer.featureCount()
    output_fields = vector_layer.fields()
    fv = deepcopy(fields_with_values)
    for key, fld in fv.items():
      # broadcast the field value
      if not isinstance(fld["value"], list):
        fld["value"] = [fld["value"]]
      if n_fts > 0 and n_fts % len(fld["value"]) != 0:
        raise Exception("The length of the field value is not matched with the output.")
      if n_fts > 0:
        fv[key]["value"] = fld["value"] * (n_fts // len(fld["value"]))
        
      if not output_fields.append(QgsField(key, int(fld["type"]))):
        self.PARENT_FEEDBACK.pushWarning(f"Field {key} is overwritten.")
        
    # assert "HISTORY" in output_fields.names(), "The field name 'HISTORY' is not found in the output fields."
    
    (sink, dest_id) = self.PARENT.parameterAsSink(
      self.PARENT_PARAMETERS, 
      param_sink, 
      self.PARENT_CONTEXT, 
      output_fields, 
      vector_layer.wkbType(), 
      vector_layer.sourceCrs()
    )

    for i, ft in enumerate(vector_layer.getFeatures()):
      new_ft = QgsFeature(output_fields)
      new_ft.setGeometry(ft.geometry())
      
      for fld in output_fields:
        if fld.name() in fv.keys():
          if fld.name() in ft.fields().names() and ft[fld.name()] is not None and isinstance(ft[fld.name()], str) and fv[fld.name()].get("append", False):
            new_ft[fld.name()] = f"{ft[fld.name()]}; {fv[fld.name()]['value'][i]}"
          else:
            new_ft[fld.name()] = fv[fld.name()]["value"][i]
        else:
          new_ft[fld.name()] = ft[fld.name()]
            
      sink.addFeature(new_ft)
    
    if os.path.isfile(dest_id):
      dest_id = dest_id.replace(os.path.sep, "/")
    
    return dest_id
    
  
  def parseCrs(self, check_cartesian:bool = True) -> QgsCoordinateReferenceSystem:
    """Parses the CRS from the parameters of the PARENT instance.

    Args:
        check_cartesian (bool, optional): Check if the result is cartesian coordinates. Defaults to True.

    Returns:
        QgsCoordinateReferenceSystem: The CRS of the target.
    """
    target_crs = None
    if "TARGET_CRS" in self.PARENT_PARAMETERS.keys():
      target_crs = self.PARENT.parameterAsCrs(self.PARENT_PARAMETERS, "TARGET_CRS", self.PARENT_CONTEXT)
    else:
      for ui_key, ui_settings in self.PARENT.PARAMETERS.items():
        if ui_settings.get("crs_reference", False):
          target_crs = self.PARENT.parameterAsSource(self.PARENT_PARAMETERS, ui_key, self.PARENT_CONTEXT).sourceCrs()
    
    if target_crs is not None and check_cartesian:
      self.checkCrsAsCartesian(target_crs)
    return target_crs
  
  
  def tr(self, string:str):
    return QCoreApplication.translate(self.__class__.__name__, string)
      
  # Post processing; append layers
  @staticmethod
  def registerPostProcessAlgorithm(context: QgsProcessingContext, postprocessors: dict) -> dict:
    """Registers the post-processors to the layers.

    Args:
        context (QgsProcessingContext): context of the processing
        postprocessors (dict): maste dictionary of postprocessors

    Returns:
        dict: {}
    """
    postprocessors_norm = postprocessors | {os.path.normpath(k): v for k, v in postprocessors.items()}
    for lyr_id, _ in context.layersToLoadOnCompletion().items():
      if lyr_id in postprocessors_norm.keys():
        context.layerToLoadOnCompletionDetails(lyr_id).setPostProcessor(postprocessors_norm[lyr_id])
    return {}
  
  
  @staticmethod
  def getColorThemes(path:str = os.path.join(os.path.dirname(__file__), "color_themes.json")) -> dict:
    """Reads the color themes from the json file.

    Args:
        path (str, optional): Path of the json file of the colors. Defaults to os.path.join(os.path.dirname(__file__), "color_themes.json").

    Returns:
        dict: Dictionary of the color themes
    """
    with open(path, "r") as f:
      color_themes = json.load(f)
    
    return color_themes
