from qgis.PyQt.QtCore import (QT_TRANSLATE_NOOP, QCoreApplication, QVariant)
from qgis.core import (
  QgsProcessing,
  QgsProcessingAlgorithm,
  QgsProcessingParameterFeatureSource,
  QgsProcessingParameterNumber,
  QgsProcessingParameterFeatureSink,
  QgsProcessingParameterBoolean,
  QgsProcessingParameterExtent,
  QgsProcessingParameterFolderDestination,
  QgsVectorLayer,
  QgsProcessingParameterString,
  QgsProcessingParameterCrs
  )


from ..algutil.hriskutil import HrUtil
from ..algutil.hrisknoisemodelling import NoiseModelling
from ..algutil.hriskpostprocessor import HrPostProcessor
from ..algutil.hriskvar import PostProcessors, HriskHome

import os
import re
from qgis import processing

class receiverregulargrid(QgsProcessingAlgorithm):
  PARAMETERS = { 
    "FENCE_EXTENT": {
      "ui_func": QgsProcessingParameterExtent,
      "ui_args":{
        "description": QT_TRANSLATE_NOOP("receiverfacade","Calculation extent"),
        "defaultValue": None
      }
    },
    "TARGET_CRS": {
      "ui_func": QgsProcessingParameterCrs,
      "ui_args": {
        "description": QT_TRANSLATE_NOOP("receiverfacade","Target CRS (Cartesian coordinates)")
      }
    },
    "BUILDING": {
      "ui_func": QgsProcessingParameterFeatureSource,
      "ui_args":{
        "description": QT_TRANSLATE_NOOP("receiverregulargrid","Building layer"),
        "types": [QgsProcessing.TypeVectorPolygon],
        "optional": True
      },
      "nm_key": "buildingGeomPath"
    },    
    "SOURCE": {
      "ui_func": QgsProcessingParameterFeatureSource,
      "ui_args":{
        "description": QT_TRANSLATE_NOOP("receiverregulargrid","Source layer"),
        "types": [QgsProcessing.TypeVectorPoint,QgsProcessing.TypeVectorLine,QgsProcessing.TypeVectorPolygon],
        "optional": True,
      },
      "nm_key": "sourceGeomPath"
    },    
    
    "DELTA": {
      "ui_func": QgsProcessingParameterNumber,
      "ui_args": {
        "description": QT_TRANSLATE_NOOP("receiverregulargrid","Distance between receivers (m)"),
        "type": QgsProcessingParameterNumber.Double,
        "minValue": 1.0, "defaultValue": 10.0, "maxValue": 100.0
      },
      "nm_key": "delta"
    },
    "HEIGHT": {
      "ui_func": QgsProcessingParameterNumber,
      "ui_args": {
        "description": QT_TRANSLATE_NOOP("receiverregulargrid","Height of receivers (m)"),
        "type": QgsProcessingParameterNumber.Double,
        "minValue": 0.01, "defaultValue": 4.0, "maxValue": 100.0
      },
      "nm_key": "height"
    },
    "RUN_CALCULATION": {
      "advanced": True,
      "ui_func": QgsProcessingParameterBoolean,
      "ui_args": {
        "description": QT_TRANSLATE_NOOP("receiverregulargrid","Run the calculation (or just prepare the files)?"),
        "defaultValue": True
      }
    },
    "OVERWRITE": {
      "advanced": True,
      "ui_func": QgsProcessingParameterBoolean,
      "ui_args": {
        "description": QT_TRANSLATE_NOOP("receiverregulargrid","Overwrite results?"),
        "defaultValue": False
      }
    },
    "WORKING_DIRECTORY": {
      "advanced": True,
      "ui_func": QgsProcessingParameterFolderDestination,
      "ui_args": {
        "description": QT_TRANSLATE_NOOP("receiverregulargrid","Working directory"),
        "defaultValue": QgsProcessing.TEMPORARY_OUTPUT,
      }
    },
    "SCRIPT_RUNNER": {
      "advanced": True,
      "ui_func": QgsProcessingParameterString,
      "ui_args": {
        "description": QT_TRANSLATE_NOOP("receiverregulargrid","Script runner name"),
        "defaultValue": "set_receiver_regular",
      }
    },
    "OUTPUT": {
      "ui_func": QgsProcessingParameterFeatureSink,
      "ui_args": {
        "description": QT_TRANSLATE_NOOP("receiverregulargrid","Receivers at regular grid" ),
        "defaultValue": QgsProcessing.TEMPORARY_OUTPUT
      }     
    }
  }
  
  NM_OUTPUT = {
    "RECEIVER": ["RECEIVERS"]
  }
  
  GROOVY_SCRIPT = os.path.join(HriskHome, "groovy", "receiverregulargrid.groovy")
  
  def __init__(self) -> None:
    super().__init__()
    self.UTIL = HrUtil(self)

  def initAlgorithm(self, config):
    self.UTIL.initParameters()

  def processAlgorithm(self, parameters, context, feedback):    
    
    self.UTIL.registerProcessingParameters(parameters, context, feedback)
    self.CURRENT_PROCESS = self.UTIL.parseCurrentProcess(with_nm=True)
    
    target_crs = self.UTIL.parseCrs()
    nm_version = self.UTIL.getNoiseModellingVersion()

    self.NOISEMODELLING = NoiseModelling(
      crs = target_crs,
      work_dir= self.parameterAsString(parameters, "WORKING_DIRECTORY", context),
      feedback = feedback,
      overwrite= self.parameterAsBool(parameters, "OVERWRITE", context)
    )

    self.NOISEMODELLING.initArgs(groovy_script = self.GROOVY_SCRIPT)    
    
    rect = self.UTIL.asQgsReferencedRectangle(parameters["FENCE_EXTENT"], target_crs)
    
    fence_uri = processing.run(
      "native:extenttolayer",
      {
        "INPUT": rect,
        "OUTPUT": os.path.join(self.NOISEMODELLING.WORK_DIR, "tmp_fence.geojson")
      }
    )["OUTPUT"]
    self.NOISEMODELLING.vectorLayerToArg(fence_uri, "FENCE", {}, "fenceGeomPath")
    
    
    for ui_key, ui_settings in self.PARAMETERS.items():
      if parameters.get(ui_key) is not None:
        value = parameters[ui_key]
      else:
        value = ui_settings["ui_args"].get("defaultValue", None)
      if value is not None:
        self.NOISEMODELLING.uiParametersToArg(ui_key, ui_settings, value)
    
    if nm_version[0] == "4":
      nm_runner = "org.noisemodelling.runner.Main"
    elif nm_version[0] == "5":
      nm_runner = "org.noise_planet.noisemodelling.runner.Main"
      
    self.NOISEMODELLING.setJavaCommand(
      nm_runner = nm_runner,
      script_name = self.parameterAsString(parameters, "SCRIPT_RUNNER", context),
    )
    
    if not self.parameterAsBool(parameters, "RUN_CALCULATION", context):
      return {"OUTPUT": None}      
  
    results = self.NOISEMODELLING.run()
    feedback.setProgress(100)

    receiver_result = QgsVectorLayer(results.get(self.NM_OUTPUT["RECEIVER"][0]))

    fields_with_values = {
      "HISTORY": {
        "type": QVariant.String,
        "value": self.CURRENT_PROCESS,
        "append": False
      }
    }
    
    dest_id = self.UTIL.outputVectorLayer(
      vector_layer= receiver_result,
      param_sink = "OUTPUT",
      fields_with_values = fields_with_values
    )
    
    PostProcessors[dest_id] = HrPostProcessor(history = [self.CURRENT_PROCESS])
    
    return {"OUTPUT": dest_id}


  def name(self):
    return self.__class__.__name__

  def displayName(self):
    return self.tr("Regular grid")

  def group(self):
    return self.tr('Set receivers')

  def groupId(self):
    return 'receiver'

  def createInstance(self):
    return receiverregulargrid()

  # placing here is necessary, when employing pylupdate
  def tr(self, string):
    return QCoreApplication.translate(self.__class__.__name__, string)
