from qgis.PyQt.QtCore import (
  QCoreApplication,QVariant
  )
from qgis.PyQt.QtGui import QColor
from qgis.core import (
  QgsVectorLayer,
  QgsProcessingParameterDefinition, 
  QgsProcessingParameterNumber,
  QgsProcessingUtils,
  QgsVectorFileWriter,
  QgsCoordinateTransformContext,
  QgsFeatureRequest,
  QgsProcessingParameterString,
  QgsProcessingParameterBoolean,
  QgsCoordinateReferenceSystem,
  QgsProcessingFeedback,
  QgsProcessingContext,
  QgsReferencedRectangle,
  QgsCoordinateTransform,
  QgsProject,
  QgsFields,
  QgsField,
  QgsFeature,
  QgsProcessingParameterExtent,
  QgsRectangle,
  QgsSingleBandPseudoColorRenderer,
  QgsRasterShader,
  QgsColorRampShader,
  QgsRasterBandStats,
  QgsFillSymbol,
  QgsGraduatedSymbolRenderer,
  QgsRendererRange,
  QgsClassificationRange,
  QgsMarkerSymbol,
  QgsFillSymbol,
  QgsLineSymbol,
  QgsWkbTypes,
  QgsProcessingLayerPostProcessorInterface
  )

import os
import asyncio
import re
import shutil
import glob
import datetime
import zipfile
import copy
import json
import math
import itertools
import concurrent
import requests
import platform

from .hriskvar import PostProcessors, ColorThemes, EquatorLength, MeshUnit

from qgis.utils import iface
from qgis import processing

# note they are implicitly used in HrTile
from .worldmesh import (
  meshcode_to_latlong,
  cal_meshcode1,
  cal_meshcode2,
  cal_meshcode3,
  cal_meshcode4,
  cal_meshcode5,
  cal_meshcode6,
  cal_meshcode_ex100m_12,
  cal_meshcode_ex100m_13,
  cal_meshcode_ex50m_13,
  cal_meshcode_ex50m_14,
  cal_meshcode_ex10m_14,
  cal_meshcode_ex1m_16
)

class HrApi(object):

  PARENT = None
  PARENT_PARAMETERS = None
  PARENT_CONTEXT = None
  PARENT_FEEDBACK = None
  
  def __init__(self, parent, parent_parameters:dict = {}, parent_context: QgsProcessingContext= None, parent_feedback: QgsProcessingFeedback = None) -> None:
    self.PARENT = parent
    self.PARENT_PARAMETERS = parent_parameters
    self.PARENT_CONTEXT = parent_context
    self.PARENT_FEEDBACK = parent_feedback
    self.SINK = None
    self.DEST_ID = None
  
  def registerProcessingParameters(self, parent_parameters:dict, parent_context: QgsProcessingContext, parent_feedback: QgsProcessingFeedback):
    self.PARENT_PARAMETERS = parent_parameters
    self.PARENT_CONTEXT = parent_context
    self.PARENT_FEEDBACK = parent_feedback
  
  
  def transformExtentToProjCrs(self, extent:QgsReferencedRectangle):
    # Note that eather CRS argument of QgsCoordinateTransform must be the CRS of the QgsProject
    tr = QgsCoordinateTransform(extent.crs(), QgsProject.instance().crs(), QgsProject.instance())
    extent_tr = QgsReferencedRectangle(tr.transformBoundingBox(extent), QgsProject.instance().crs())
    return extent_tr
    
  
  @staticmethod
  def getHriskVersion() -> str:
    try:
      with open(os.path.join(os.path.dirname(__file__), "metadata.txt")) as hrisk_meta:
        version_match = re.search(r"version=(\S+)", hrisk_meta.read())
        return version_match.group(1)
    except:
      return "unknown"
  
  @staticmethod
  def getNoiseModellingVersion() -> str:
    try:
      nm_home = os.environ["NOISEMODELLING_HOME"]
    except:
      raise Exception("Environment variable of NOISEMODELLING_HOME is not set.")
    
    try:
      for jarfile in os.listdir(os.path.join(nm_home, "lib")):
        if jarfile.startswith("noisemodelling-jdbc-"):
          with zipfile.ZipFile(os.path.join(nm_home, "lib", jarfile), 'r') as jar:
            with jar.open('META-INF/MANIFEST.MF') as manifest:
              version_match = re.search(r"Bundle-Version: (\S+)", manifest.read().decode())
              return version_match.group(1)
    except:
      return "unknown"

  def parseCurrentProcess(self, with_nm=False):
    
    caller = self.PARENT.__class__.__name__
    hrisk_version = self.getHriskVersion()

    
    hr = caller + " (" + "H-RISK v " + hrisk_version +" )"
    dt = datetime.datetime.now().strftime("%Y-%m-%d %H:%M:%S")
    
    if with_nm:        
      nm = "NoiseModelling v " + self.getNoiseModellingVersion()
      return hr + " with " + nm + " at " + dt 
    else:
      return hr + " at " + dt

  def setDefaultValue(self, key, value) -> None:
    assert hasattr(self.PARENT, "PARAMETERS"), "PARAMETERS is not defined in the class."
    self.PARENT.PARAMETERS[key]["ui_args"]["defaultValue"] = value

  def initParameters(self) -> None:

    assert hasattr(self.PARENT, "PARAMETERS"), "PARAMETERS attribute is not defined in the class."
    
    for key, value in self.PARENT.PARAMETERS.items():
      args = value.get("ui_args")
      args["name"] = key
      args["description"] = self.PARENT.tr(args["description"])
              
      ui = value.get("ui_func")(**args)
      
      if value.get("advanced") is not None and value.get("advanced") == True:
        ui.setFlags(QgsProcessingParameterDefinition.FlagAdvanced)
        
      self.PARENT.addParameter(ui)  

  @classmethod
  def checkCrsAsCartesian(cls, crs: QgsCoordinateReferenceSystem) -> None:
    assert crs.isValid(), "The CRS is not valid."
    
    if crs.isGeographic() or crs.authid() == "EPSG:3857":
      raise Exception(cls.tr("The Target CRS is NOT a Cartesian Coordinate System."))

  @classmethod
  def getExtentAndCrsUsingCanvas(cls, get_xy_coords=True) -> tuple[str, QgsCoordinateReferenceSystem]:

    try:
      map_rectangle = iface.mapCanvas().extent()
      proj_crs = iface.mapCanvas().mapSettings().destinationCrs()
      
      use_rect = QgsReferencedRectangle(map_rectangle, proj_crs)

      use_crs = proj_crs
      if get_xy_coords:
        # if the CRS is geographic, input the long/lat to the getUtmCrs
        if proj_crs.isGeographic():
          use_crs = cls.getUtmCrs(map_rectangle.center().x(), map_rectangle.center().y())
        # if the CRS is web-mercator, transform long/lat coordinates and input to the getUtmCrs
        elif proj_crs.authid() == "EPSG:3857":
          transform = QgsCoordinateTransform(proj_crs, QgsCoordinateReferenceSystem("EPSG:4326"), QgsProject.instance())
          map_rectangle = transform.transformBoundingBox(map_rectangle)
          use_crs = cls.getUtmCrs(map_rectangle.center().x(), map_rectangle.center().y())
      
      use_extent = ",".join(
        list(map(lambda x: str(x), [
        use_rect.xMinimum(), use_rect.xMaximum(), use_rect.yMinimum(), use_rect.yMaximum()
        ]))
      ) + f"[{proj_crs.authid()}]"
      
      return use_extent, use_crs
    except:
      return "", QgsCoordinateReferenceSystem()

  @staticmethod
  def getUtmCrs(lng: float, lat: float) -> QgsCoordinateReferenceSystem:
    """
    Gets the CRS of the UTM using longitude and latitude.

    Parameters:
    - lng (float): The longitude.
    - lat (float): The latitude.

    Returns:
    QgsCoordinateReferenceSystem: The UTM CRS.

    Raises:
    None
    """
    epsg_code = 32600 + 100 * (lat < 0) + int((lng + 180) / 6) + 1
    crs = QgsCoordinateReferenceSystem(f'EPSG:{epsg_code}')
    assert crs.isValid(), "The CRS is not valid."
    return crs
  
  def dissolve(self, vl:QgsVectorLayer, field_names_drop:list[str] = ["layer","path"]) -> QgsVectorLayer:
    
    lyr_valid = processing.run(
      "qgis:checkvalidity",
      {
        "INPUT_LAYER": vl,
        "VALID_OUTPUT": "TEMPORARY_OUTPUT"
      },
      context = self.PARENT_CONTEXT,
      is_child_algorithm = True
    )["VALID_OUTPUT"]
    
    field_names_dissolve = [
      fld.name() 
      for fld in self.PARENT_CONTEXT.getMapLayer(lyr_valid).fields() 
      if fld.name() not in field_names_drop
    ]
  
    # Dissolve
    lyr_dissolve = processing.run(
      "native:dissolve", 
      {
        "INPUT": lyr_valid,
        "FIELD": field_names_dissolve,
        "OUTPUT": "TEMPORARY_OUTPUT"
      },
      context = self.PARENT_CONTEXT,
      is_child_algorithm = True
    )["OUTPUT"]
    
    # Multipart to Single parts
    lyr_single = processing.run(
      "native:multiparttosingleparts", 
      {
        "INPUT": lyr_dissolve,
        "OUTPUT": "TEMPORARY_OUTPUT"
      },
      context = self.PARENT_CONTEXT,
      is_child_algorithm = True
    )["OUTPUT"]
    
    return lyr_single
  
  def downloadFile(self, session = None, feedback_progress: bool = False, method:str = "GET", url:str = "", params:dict = None, data:dict = None, json:dict = None, dest:str = None, stream:bool=True, **kwargs:dict) -> str:

    if session is None:
      session = requests.Session()
    try:
      response = session.request(method = method, url = url, params = params, data = data, json = json, stream = stream)
      # if query is not None:
      #   response = session.request(method = method, url = url + query, stream = stream)
      # else:
      #   response = session.request(method = method, url = url, stream = stream)
    except Exception as e:
      return str(e)
    # if download is successful
    if response.status_code == 200:
      file_size = int(response.headers.get('content-length', 0))
      if dest is None:
        try:
          dest = re.search(r'.*\.[^|]*', os.path.basename(url)).group(0)
        except:
          dest = "downloaded_file"
        
      temp_dir = self.createTempDir()
      file_path = os.path.join(temp_dir, dest)
      
      if not os.path.exists(os.path.dirname(file_path)):
        os.makedirs(os.path.dirname(file_path))

      # write downloaded file and show progress
      with open(file_path, 'wb') as file:
        for data in response.iter_content(chunk_size=1024):
          file.write(data)
          if feedback_progress:
            self.PARENT_FEEDBACK.setProgress(int((file.tell() / file_size) * 100))
      
      if feedback_progress:
        self.PARENT_FEEDBACK.setProgress(0)
      
      return file_path
      
    else:
      return Exception("The Link was missing or download was failed.")

  def downloadFilesConcurrently(self, session=None, args:dict={}, parallel = True, omit_null = True) -> dict:
    
    mx_wk = None if parallel else 1
    downloaded = {}
    
    if session is None:
      session = requests.Session()
    
    # fetch in parallel
    with concurrent.futures.ThreadPoolExecutor(max_workers=mx_wk) as executor:
      for key, data_path in zip(
        args.keys(),
        executor.map(
        lambda d_arg : self.downloadFile(
          session, 
          feedback_progress = not parallel,
          **d_arg 
          ), 
        args.values()
        )
      ):
        if self.PARENT_FEEDBACK.isCanceled():
          self.PARENT_FEEDBACK.reportError(self.tr("Data download was canceled."))
          raise Exception(self.tr("Data download was canceled."))
        
        self.PARENT_FEEDBACK.pushInfo(f"({key}) Downloaded from: {args[key]['url']}")
        
        if omit_null and isinstance(data_path, str):
          downloaded[key] = data_path
    
    return downloaded

  @classmethod
  def extractArchive(cls, archive_path: str, extension: str = "", extract_dir: str = None, pattern: str = None) -> list[str]:   

    if extract_dir is None:
      extract_dir = os.path.normpath(os.path.dirname(QgsProcessingUtils.generateTempFilename("")))
    
    if not os.path.exists(extract_dir):
      os.makedirs(extract_dir)      
    
    if len(extension) > 0 and not archive_path.endswith(extension):
      os.rename(archive_path, archive_path + extension)
      
    extracted_files = []
    extracted_dirs = []
    
    try:
      shutil.unpack_archive(archive_path, extract_dir)
      for root, dirs, files in os.walk(extract_dir):
        extracted_dirs.append(root)
        paths = [os.path.join(root, x) for x in files]
        if pattern is None:
          extracted_files += paths
        else:
          for path in paths:
            if re.search(pattern, path):
              extracted_files.append(path)
    except:
      pass # if the file is not an archive file, do nothing
    
    return extracted_dirs, extracted_files
  
  @staticmethod
  def createTempDir() -> str:
    tmp_dir = os.path.normpath(os.path.dirname(QgsProcessingUtils.generateTempFilename("")))
    if not os.path.exists(tmp_dir):
      os.makedirs(tmp_dir)
    
    return tmp_dir
  
  def parseNoiseModellingArgs(self, args:dict, target_crs:QgsCoordinateReferenceSystem, wps_args:dict = {}, ui_parameters:dict = {}) -> dict:
    
    assert isinstance(args, dict), self.tr("The paths must be a dictionary.")
    assert "GROOVY_SCRIPT" in args, self.tr("Groovy script is not specified.")
    assert os.environ["NOISEMODELLING_HOME"], self.tr("Environment variable of NOISEMODELLING_HOME is not set.")
    
    tmp_dir = self.createTempDir()
    grv_dir = os.path.join(os.path.dirname(__file__), "noisemodelling","hriskscript")
    wps_path = os.path.join(os.path.dirname(__file__), "noisemodelling","bin","wps_scripts")
    
    nm = {}
    
    nm["HOME"] = os.environ["NOISEMODELLING_HOME"]
    nm["TEMP_DIR"] = tmp_dir
    
    for key, value in args.items():
      nm[key] = value.replace("%nmtmp%", tmp_dir).replace("%grvhome%", grv_dir)

    
    # note that the order of the dictionary is important!
    nm_wps = {
      "w": '"' + tmp_dir + '"', 
      "s": '"' + nm["GROOVY_SCRIPT"] + '"',
      "noiseModellingHome": '"' + os.path.normpath(nm["HOME"]) + '"',
      "exportDir": '"' + tmp_dir+ '"',
      "inputSRID": target_crs.authid().replace("EPSG:", ""),
    }
    
    # set other arguments
    if ui_parameters is not None:
      for key, value in ui_parameters.items():
        if value.get("n_mdl") is not None:
          if value.get("save_layer_get_path", False) is True:
            src = self.PARENT.parameterAsSource(self.PARENT_PARAMETERS, key, self.PARENT_CONTEXT)
            if src is None: continue
            if src.sourceCrs() != target_crs:
              self.PARENT_FEEDBACK.reportError(self.tr("CRS is not the same among input features."), fatalError=True)
              raise Exception(self.tr("CRS is not the same among input features."))
            vl = src.materialize(QgsFeatureRequest(), self.PARENT_FEEDBACK)
            vl_path = os.path.join(tmp_dir, key + ".geojson")
            self.saveVectorLayer(vl, vl_path)
            # register in the WPS_ARGS
            nm_wps[value.get("n_mdl")] = '"' + vl_path + '"'
          else:
            if value.get("ui_func") == QgsProcessingParameterString:
              value_input = '"' + self.PARENT.parameterAsString(self.PARENT_PARAMETERS, key, self.PARENT_CONTEXT) + '"'
            if value.get("ui_func") == QgsProcessingParameterBoolean:
              value_input = self.PARENT.parameterAsInt(self.PARENT_PARAMETERS, key, self.PARENT_CONTEXT)
            if value.get("ui_func") == QgsProcessingParameterNumber:
              if value.get("ui_args").get("type", QgsProcessingParameterNumber.Double) == QgsProcessingParameterNumber.Integer:
                value_input = self.PARENT.parameterAsInt(self.PARENT_PARAMETERS, key, self.PARENT_CONTEXT)
              else:
                value_input = self.PARENT.parameterAsDouble(self.PARENT_PARAMETERS, key, self.PARENT_CONTEXT)
              
            # register in the WPS_ARGS
            nm_wps[value.get("n_mdl")] = value_input
    nm_wps.update(wps_args)
    
    nm["WPS_ARGS"] = nm_wps
    
    pf = platform.system()
    if pf in ["Darwin","Linux"]:
      nm["CMD"] = '/bin/bash "' + wps_path + '"' + "".join([" -" + k + " " + str(v) for k, v in nm_wps.items()])
    elif pf == "Windows":    
      nm["CMD"] = '"' + wps_path + '"' + "".join([" -" + k + " " + str(v) for k, v in nm_wps.items()])
      
    return nm

    
  @staticmethod
  def saveVectorLayer(vector_layer: QgsVectorLayer, path: str) -> None:
    save_options = QgsVectorFileWriter.SaveVectorOptions()
    save_options.driverName = "GeoJSON"
    QgsVectorFileWriter.writeAsVectorFormatV3(
      vector_layer, path, QgsCoordinateTransformContext(), save_options
    )
  
  def execNoiseModellingCmd(self, cmd, temp_dir) -> None:
    loop = asyncio.new_event_loop()
    loop.run_until_complete(self.streamNoiseModellingCmd(cmd, temp_dir))
    loop.close()
    
  async def streamNoiseModellingCmd(self, cmd, temp_dir) -> None:
    proc = await asyncio.create_subprocess_shell(
      cmd,
      stdout = asyncio.subprocess.PIPE,
      stderr = asyncio.subprocess.PIPE,
      cwd    = temp_dir
    )

    while True:
      if proc.stdout.at_eof() or proc.stderr.at_eof():
        break

      stderr_raw = await proc.stderr.readline()  # for debugging
      try:
        stderr = stderr_raw.decode("utf-8","ignore")
      except:
        stderr = ""
        self.PARENT_FEEDBACK.reportError(self.tr("NoiseModelling script was not successfully executed."), fatalError=True)
        raise Exception(self.tr("NoiseModelling script was not successfully executed."))

      if stderr:
        self.PARENT_FEEDBACK.pushConsoleInfo(stderr.replace("\n", ""))

      prg_match = re.search(r".*[0-9]+\.[0-9]+.*%", stderr)
      if prg_match:
        self.PARENT_FEEDBACK.setProgress(
          int(float(re.search(r"[0-9]+\.[0-9]+", prg_match.group()).group()))
        )
  
  @staticmethod
  def newFieldsWithHistory(current_fields: QgsFields) -> QgsFields:
    # note that one-by-one appending is necessary to remove unsupported field type
    new_fields = QgsFields()
    for fld in current_fields:
      if fld.type() in [QVariant.Int, QVariant.Double, QVariant.String]:
        new_fields.append(fld)
      else:
        new_fields.append(QgsField(fld.name(), QVariant.String))
        
    if "HISTORY" not in [fld.name() for fld in current_fields]:
      new_fields.append(QgsField("HISTORY", QVariant.String))
    return new_fields
  
  @staticmethod
  def addFeaturesWithHistoryToSink(
    sink, vector_layer:QgsVectorLayer, fields:QgsFields, current_process:str,
    additional_attributes:dict = {}
    ) -> None:
    
    for i, ft in enumerate(vector_layer.getFeatures()):
      new_ft = QgsFeature(fields)
      new_ft.setGeometry(ft.geometry())
      
      for fld in fields:
        if fld.name() != "HISTORY":
          if fld.name() in additional_attributes.keys():
            try:
              new_ft[fld.name()] = additional_attributes[fld.name()][i]
            except:
              raise Exception(f"Length of the additional attribute {fld.name} is not matched with the output.")
          elif fld.name() in ft.fields().names():
            new_ft[fld.name()] = ft[fld.name()]
        else:
          if "HISTORY" in ft.fields().names() and len(ft["HISTORY"]) > 0:
            new_ft["HISTORY"] = ft["HISTORY"] + "; " + current_process
          else:
            new_ft["HISTORY"] = current_process
      
      sink.addFeature(new_ft)
  
  @classmethod
  def fenceExtentAsLayer(cls, context:QgsProcessingContext, fence_extent:QgsProcessingParameterExtent, target_crs:QgsCoordinateReferenceSystem) -> None:
    fence_layer = processing.run(
      "native:extenttolayer",
      {
        "INPUT": fence_extent,
        "OUTPUT": "TEMPORARY_OUTPUT"
      },
      context = context,
      is_child_algorithm = True
    )["OUTPUT"]
    
    fence_transformed = processing.run(
      "native:reprojectlayer",
      {
        "INPUT": fence_layer,
        "TARGET_CRS": target_crs,
        "OUTPUT": "TEMPORARY_OUTPUT"
      },
      context = context,
      is_child_algorithm = True
    )["OUTPUT"]
    
    return fence_transformed

  @classmethod
  def fenceExtentAsWkt(cls, context:QgsProcessingContext, fence_extent:QgsProcessingParameterExtent, target_crs:QgsCoordinateReferenceSystem) -> None:
    fence_layer = cls.fenceExtentAsLayer(context, fence_extent, QgsCoordinateReferenceSystem("EPSG:4326"))      
    fence_layer = context.getMapLayer(fence_layer)
    return fence_layer.getFeature(1).geometry().asWkt()

  def parseCrs(self, check_cartesian = True):
    if "TARGET_CRS" in self.PARENT_PARAMETERS.keys():
      target_crs = self.PARENT.parameterAsCrs(self.PARENT_PARAMETERS, "TARGET_CRS", self.PARENT_CONTEXT)
    else:
      crs_key = [key for key, value in self.PARENT.PARAMETERS.items() if value.get("crs_reference") is not None and value.get("crs_reference") == True]
      target_crs = self.PARENT.parameterAsSource(self.PARENT_PARAMETERS, crs_key[0], self.PARENT_CONTEXT).sourceCrs()
    
    if check_cartesian:
      self.checkCrsAsCartesian(target_crs)
    return target_crs
  
  
  def tr(self, string):
    return QCoreApplication.translate(self.__class__.__name__, string)
      
  # Post processing; append layers
  @staticmethod
  def registerPostProcessAlgorithm(context):
    for lyr_id, _ in context.layersToLoadOnCompletion().items():
      if lyr_id in PostProcessors.keys():
        context.layerToLoadOnCompletionDetails(lyr_id).setPostProcessor(PostProcessors[lyr_id])
    return {}
  
  @staticmethod
  def getColorThemes(path:str = None):
    json_path = os.path.join(os.path.dirname(__file__), "color_themes.json") if path is None else path
    with open(json_path, "r") as f:
      color_themes = json.load(f)

    try:
      color_themes["Default"] = copy.deepcopy(ColorThemes["Weninger color scheme"])
    except:
      color_themes["Default"] = {
        "theme_ref": "NULL theme",
        "range_map":{
          "ALL": {"lower": -999, "upper": 999, "color": [160, 186, 191]}
        }
      }

# case insensitive Fields
class HrFields(QgsFields):

  def append(self, field, *args, overwrite = False, **kwargs):
    new_field_name = field.name().lower()
    existing_field_names = map(lambda x: x.lower(), self.names())
    
    if new_field_name in existing_field_names and not overwrite:
      raise Exception(f"Field {field.name()} is already defined.")
    else:
      if new_field_name in existing_field_names:
        filt = list(filter(lambda x: x.lower() == new_field_name, self.names()))
        if len(filt) > 1:
          raise Exception(f"Current fields already contain duplicated name (case insensitive): {field.name().lower()}.")
        else:
          self.remove(filt[0])
      super().append(field, *args, **kwargs)
      
  @classmethod
  def fromQgsFieldList(cls, QgsFieldList:list[QgsField], overwrite = False, **kwargs):
    assert isinstance(QgsFieldList, list), "QgsFieldList must be a list."
    fields_tmp = cls()
    for fld in QgsFieldList:
      assert isinstance(fld, QgsField), "Each element of QgsFieldList must be a QgsField."
      fields_tmp.append(fld, overwrite=overwrite, **kwargs)
    return fields_tmp
  
  def setComments(self, comment, append = True):
    fields_tmp = HrFields()
    for fld in self:
      if append:
        fld.setComment(fld.comment() + " " + comment)
      else:
        fld.setComment(comment)
      fields_tmp.append(fld)
    return fields_tmp
  
  @staticmethod
  def checkCaseInsensitiveFields(fields):
    names = [fld.name().lower() for fld in fields]
    if len(names) != len(set(names)):
      raise Exception("Current fields contain duplicated names (case-insensitive).")
  
  @classmethod
  def concatenateHrFields(cls, QgsFieldsList:list[QgsFields],overwrite = False):
    fields_target = cls()
    for fields in QgsFieldsList:
      for fld in fields():
        fields_target.append(fld, overwrite = overwrite)
    return fields_target
  
  
  def concat(self, fields, overwrite = False, **kwargs):
    fields_tmp = HrFields()
    for fld in self:
      field_idx = fields.indexFromName(fld.name())
      if field_idx >= 0:
        if overwrite:
          fields_tmp.append(fields.field(field_idx), **kwargs)
        else:
          fields_tmp.append(fld, **kwargs)
        fields.remove(field_idx)
      else:        
        fields_tmp.append(fld, **kwargs)
    
    for fld in fields:
      fields_tmp.append(fld, **kwargs)
    return fields_tmp
  
  def toQgsFields(self, **kwargs):
    fields_tmp = QgsFields()
    for fld in self.fields:
      fields_tmp.append(fld, **kwargs)
    return fields_tmp
  
  @classmethod
  def fromQgsFields(cls, fields, **kwargs):
    fields_tmp = cls()
    for fld in fields:
      fields_tmp.append(fld, **kwargs)
    return fields_tmp
  
  
  @classmethod
  def fromDict(cls, fields_dict:dict):
    fields_tmp = cls()
    for key, value in fields_dict.items():
      if type(value) in [int, bool]:
        fields_tmp.append(QgsField(key, QVariant.Int))
      elif type(value) in [float]:
        fields_tmp.append(QgsField(key, QVariant.Double))
      elif type(value) in [str]:
        fields_tmp.append(QgsField(key, QVariant.String))
    return fields_tmp
  
  @staticmethod
  def getDefaultValues(fields):
    vals = []
    for fld in fields:
      try:
        val_str = re.search(r'default\s?:\s?([^;]+);', fld.comment()).group(1)
        if fld.type() == QVariant.Double:
          vals.append(float(val_str))
        elif fld.type() == QVariant.Int:
          vals.append(int(val_str))
        elif fld.type() == QVariant.String:
          vals.append(val_str)
      except:
        vals.append(None)
    return vals


  @staticmethod
  def getSpecifiedComments(fields, key):
    vals = []
    for fld in fields:
      try:
        val_str = re.search(key + r'\s?:\s?([^;]+);', fld.comment()).group(1)
        vals.append(val_str)
      except:
        vals.append(None)
    return vals

  def __repr__(self):
    return str([
      f"({fld.displayName()} (type: {fld.displayType()}, {fld.comment()})"
      for fld in self.toList()
    ])


class HrNoiseColorRenderer (object):
  COLOR_ARGS = {
      "coloring": "none",
      "theme_ref": None,
      "attribute": "LAEQ",
      "band_idx" : 1,
      "fill_color": QColor(0,0,0,0),
      "border_color": QColor(0,0,0,255),
      "border_width": 0.5,
      "value_map" : {"min": {"value": None, "color": QColor("#f7f7f7")}},
      "range_map" : {"default": {"lower": -999, "upper": 999, "color": QColor(0,0,0,255)}},
      "set_clip" : True,
      "opacity" : 0.5
    }
  
  # initialize instance variables
  def __init__(
    self, 
    color_args:dict = {"coloring": "none"}, 
    visibility:bool = True,
    set_min_to_zero:bool = False,
    **kwargs
    ):
    super().__init__()  
    
    self.COLOR_ARGS = copy.deepcopy(self.COLOR_ARGS)
    self.GROUP = None
    self.VISIBILITY = True
    
    color_args.update(kwargs)
    self.setColorArgs(**color_args)
    self.setVisivility(visibility)
    if set_min_to_zero:
      self.setMinValToZero()
    
  
  # visibility setter
  def setVisivility(self, visibility):
    self.VISIBILITY = visibility
    
  def getVisibility(self):
    return self.VISIBILITY
  
  # color arguments setter
  def setColorArgs(self, **kwargs):    
    # if theme is given, look for the theme in the color themes
    if "theme" in kwargs.keys():
      if kwargs.get("theme_json_path") is None:
        try:
          color_theme = copy.deepcopy(ColorThemes[kwargs["theme"]])
        except:
          raise Exception("Unknown theme: " + kwargs["theme"])
      else:
        try:
          with open(kwargs["theme_json_path"], "r") as f:
            color_theme = json.load(f)
        except:
          raise Exception("Error in reading the external theme: " + kwargs["theme_json_path"])
          
      
      for k, v in kwargs.items():
        if k != "theme":
          color_theme[k] = v
      
      self.setColorArgs(**color_theme)
      return None
    
    # if each argument is given, set the argument
    # parsing the colors and setting the opacity is necessary
    for k, v in kwargs.items():
      if isinstance(v, dict):
        for vk, vv in v.items():
          if "color" in vv.keys():
            if isinstance(vv["color"], str):
              v[vk]["color"] = QColor(vv["color"])
            elif isinstance(vv["color"], list):
              v[vk]["color"] = QColor(*vv["color"])
            elif not isinstance(vv["color"], QColor):
              raise Exception("Unknown color format: " + str(vv["color"]))
          if "opacity" in kwargs.keys() and kwargs["opacity"] is not None:
            try:
              v[vk]["color"].setAlphaF(kwargs["opacity"])
            except:
              raise Exception("Error in setting the opacity: " + str(kwargs["opacity"]))
          
      self.COLOR_ARGS[k] = v
  
  def getColorArgs(self):
    return self.COLOR_ARGS
  
  def getColoring(self):
    return self.COLOR_ARGS["coloring"]
  
  # the minimum value of the color range is set to zero
  def setMinValToZero(self):
    self.COLOR_ARGS["value_map"]["min"]["value"] = 0.0
    
  # the minimum and maximum values of the color range are set to the minimum and maximum values of the layer
  def setValRangeUsingStats(self, layer, only_if_unset = True):
    stats = layer.dataProvider().bandStatistics(self.COLOR_ARGS["band_idx"], QgsRasterBandStats.All) 
    if only_if_unset:
      if self.COLOR_ARGS["value_map"]["min"]["value"] is None:
        self.COLOR_ARGS["value_map"]["min"]["value"] = stats.minimumValue
      if self.COLOR_ARGS["value_map"]["max"]["value"] is None:
        self.COLOR_ARGS["value_map"]["max"]["value"] = stats.maximumValue
    else:
      self.COLOR_ARGS["value_map"]["min"]["value"] = stats.minimumValue
      self.COLOR_ARGS["value_map"]["max"]["value"] = stats.maximumValue

  # Single band pseudo color renderer for a raster layer
  def setSingleBandPseudoColorRenderer(self, layer):    
    if layer.__class__.__name__ != "QgsRasterLayer":
      raise Exception("Single band pseudo color renderer is only applicable to a raster layer")
    if "min" not in self.COLOR_ARGS["value_map"].keys() or "max" not in self.COLOR_ARGS["value_map"].keys():
      self.setColorRangeUsingTheme("Greys")
      
    if self.COLOR_ARGS["value_map"]["min"]["value"] is None or self.COLOR_ARGS["value_map"]["max"]["value"] is None:
      self.setValRangeUsingStats(layer)
    
    cr_fun = QgsColorRampShader(
      minimumValue = self.COLOR_ARGS["value_map"]["min"]["value"],
      maximumValue = self.COLOR_ARGS["value_map"]["max"]["value"]
    )
    cr_fun.setColorRampItemList(
      [
        QgsColorRampShader.ColorRampItem(v["value"], v["color"]) for v in self.COLOR_ARGS["value_map"].values()
      ]
    )
    cr_fun.setClip(self.COLOR_ARGS["set_clip"])
    shader = QgsRasterShader()
    shader.setRasterShaderFunction(cr_fun)
    lyr_renderer = QgsSingleBandPseudoColorRenderer(layer.dataProvider(), self.COLOR_ARGS["band_idx"], shader)
    layer.setRenderer(lyr_renderer)
    
  
  # graduated symbol renderer for a vector layer
  def setGraduatedSymbolRenderer(self, layer):
    if layer.__class__.__name__ != "QgsVectorLayer":
      raise Exception("Graduated symbol renderer is only applicable to a vector layer")
    
    # change symboller according to the wkb-type
    symb_fun = None
    if layer.wkbType() in [QgsWkbTypes.Point, QgsWkbTypes.PointZ]:
      symb_fun = QgsMarkerSymbol.createSimple
    elif layer.wkbType() == QgsWkbTypes.Polygon:
      symb_fun = QgsFillSymbol.createSimple
    elif layer.wkbType() == QgsWkbTypes.LineString:
      symb_fun = QgsLineSymbol.createSimple
    else:
      raise Exception("Unsupported geometry type: " + str(layer.wkbType())) 
    
    # set renderer and symbols
    try:
      renderer = QgsGraduatedSymbolRenderer(self.COLOR_ARGS["attribute"])
      for key, value in self.COLOR_ARGS["range_map"].items():
        renderer.addClassRange(
          QgsRendererRange(
            QgsClassificationRange(key, value.get("lower"), value.get("upper")), 
            symb_fun({"color":value.get("color")})
          )
        )
      # apply renderer
      layer.setRenderer(renderer)
    except:
      raise Exception("Error in setting the graduated symbol renderer")
      
      
  # use a single symbol renderer for a vector layer
  def setFillSymbolRenderer(self, layer):
    if layer.__class__.__name__ != "QgsVectorLayer":
      raise Exception("Single symbol renderer is only applicable to a vector layer")
    symbol = QgsFillSymbol.createSimple({"color":self.COLOR_ARGS["fill_color"], "color_border":self.COLOR_ARGS["border_color"], "width_border":self.COLOR_ARGS["border_width"]})
    layer.renderer().setSymbol(symbol)
    layer.setOpacity(self.COLOR_ARGS["opacity"])
  
  
  def applyRenderer(self, layer):
    # set renderer
    if self.COLOR_ARGS["coloring"] != "none":
      if self.COLOR_ARGS["coloring"] == "single_band_pseudo_color":
        try:
          self.setSingleBandPseudoColorRenderer(layer)
        except Exception as e:
          raise Exception("Error in setting the single band pseudo color renderer: " + str(e))
      elif self.COLOR_ARGS["coloring"] == "simple_bbox":
        try:
          self.setFillSymbolRenderer(layer)      
        except Exception as e:
          raise Exception("Error in setting the fill symbol renderer: " + str(e))
      elif self.COLOR_ARGS["coloring"] == "graduated_symbol":
        try:
          self.setGraduatedSymbolRenderer(layer)
        except Exception as e:
          raise Exception("Error in setting the graduated symbol renderer: " + str(e))
          
      layer.triggerRepaint()

class HrPostProcessor (QgsProcessingLayerPostProcessorInterface):
  
  NCR = None
  GROUP = None
  HISTORY = None
  
  # initialize instance variables
  def __init__(self, history:list = [], group = None, **kwargs):
    super().__init__()
    self.HISTORY = []
    self.GROUP = None
    self.NCR = HrNoiseColorRenderer(**kwargs)
      
    self.setHistory(history)
    self.setGroup(group)
  
  def setColorArgs(self, **kwargs):
    self.NCR.setColorArgs(**kwargs)
  
  def setVisibility(self, **kwargs):
    self.NCR.setVisivility(**kwargs)
  
  def setMinValToZero(self):
    self.NCR.setMinValToZero()
  
  # clone the instance
  
  def clone(self):
    kwargs = {
      "history": self.HISTORY,
      "color_args": self.NCR.getColorArgs(),
      "group": self.GROUP,
      "visibility": self.NCR.getVisibility()
    }
    return HrPostProcessor(**kwargs)
  
  # group setter
  def setGroup(self, group):
    self.GROUP = group

  # history setter
  def setHistory(self, history:list):
    self.HISTORY = history

  # history getter
  def getHistory(self):
    return self.HISTORY
  
  # history extender
  def extendHistory(self, history:list):
    self.HISTORY.extend(history)
  
  # Post processing MAIN
  def postProcessLayer(self, layer, context, feedback):
    
    # set history items
    layer_meta = layer.metadata()
    if len(self.HISTORY) > 0:
      for history in self.HISTORY:
        layer_meta.addHistoryItem(history)
    layer.setMetadata(layer_meta)
    
    # set renderer
    try:
      self.NCR.applyRenderer(layer)
    except Exception as e:
      feedback.pushWarning(str(e))
    
    # set group
    if self.GROUP is not None:      
      root = QgsProject.instance().layerTreeRoot()
      
      grp = root.findGroup(self.GROUP)
      if grp is None:
        root.insertGroup(0, self.GROUP)
        grp = root.findGroup(self.GROUP)
        
      vl = root.findLayer(layer.id())
      vl.setItemVisibilityChecked(self.NCR.getVisibility())
      vl_clone = vl.clone()
      parent = vl.parent()
      grp.insertChildNode(0, vl_clone)
      parent.removeChildNode(vl)
      

class HrTile(object):
  FAMILY = None
  MESH_LEVEL = None
  CRS = None
  X_ORIG = None
  Y_ORIG = None
  X_UNIT = None
  Y_UNIT = None
  
  def __init__(
    self, family:str = "web_mercator",  
    zoom:int = None, 
    mesh_level = None,
    crs:QgsCoordinateReferenceSystem = QgsCoordinateReferenceSystem("EPSG:4326"),
    xunit:float = None, yunit:float = None, 
    xorig:float = None, yorig:float = None
    ):
    
    self.FAMILY = family
    
    if family == "web_mercator":
      assert zoom is not None, "The zoom level must be given."
      assert zoom in range(0, 21), "The zoom level must be in the range of 0 to 20."
      
      self.CRS = QgsCoordinateReferenceSystem("EPSG:3857")
      # note the bounds are different from epsg definition,
      # because it is modified to give square tiles
      self.X_UNIT = EquatorLength / 2 ** zoom
      self.Y_UNIT = -EquatorLength / 2 ** zoom
      self.X_ORIG = -EquatorLength / 2
      self.Y_ORIG = EquatorLength / 2
    
    elif family == "worldmesh":
      assert mesh_level is not None, "The worldmesh level must be given."
      assert mesh_level in ["1", "2", "3", "4", "5", "6", "_ex100m_12", "_ex50m_13", "_ex50m_14", "_ex10m_14", "_ex1m_16"], "The worldmesh level is not valid. It must be one of the following: 1, 2, 3, 4, 5, 6, _ex100m_12, _ex50m_13, _ex50m_14, _ex10m_14, _ex1m_16"
      assert crs.isGeographic(), "The CRS must be geographic."
      
      self.CRS = crs
      self.MESH_LEVEL = mesh_level
      self.X_UNIT = MeshUnit[mesh_level]["long"]
      self.Y_UNIT = -MeshUnit[mesh_level]["lat"]
      self.X_ORIG = -180.0
      self.Y_ORIG = 90.0
    

      
    elif family == "long_lat":
      assert xunit is not None, "The xunit must be given."
      assert yunit is not None, "The yunit must be given."
      assert crs.isGeographic(), "The CRS must be geographic."
      
      self.CRS = crs
      self.X_UNIT = xunit
      self.Y_UNIT = yunit
      self.X_ORIG = xorig if xorig is not None else -180.0
      self.Y_ORIG = yorig if yorig is not None else 90.0
    
    
    elif family == "meter_mesh":
      assert xunit is not None, "The xunit must be given."
      assert yunit is not None, "The yunit must be given."
      assert xorig is not None, "The xorig must be given."
      assert yorig is not None, "The yorig must be given."
      assert not crs.isGeographic(), "The CRS must not be geographic."
      
      self.CRS = crs
      self.X_UNIT = xunit
      self.Y_UNIT = yunit
      self.X_ORIG = xorig
      self.Y_ORIG = yorig
    
    else:
      raise Exception("Unknown tile family. Family must be one of the following: web_mercator, worldmesh, long_lat, meter_mesh.")

  def family(self):
    return self.FAMILY
  
  def cellXyIdx(self, x, y):
    pixel_x = int((x - self.X_ORIG) / self.X_UNIT)
    pixel_y = int((y - self.Y_ORIG) / self.Y_UNIT)
    return pixel_x, pixel_y
  
  def cellMeshCode(self, x, y):
    if self.FAMILY == "worldmesh":
      return eval(f"cal_meshcode{self.MESH_LEVEL}(y,x)")
    else:
      x, y = self.cellXyIdx(x, y)
      return f"{x}_{y}"
  
  def cellXyCenter(self, xidx, yidx):
    return (self.X_ORIG + self.X_UNIT * xidx + self.X_UNIT / 2, self.Y_ORIG + self.Y_UNIT * yidx + self.Y_UNIT / 2)
  
  def cellXyRectangle(self, xidx, yidx):
    return QgsRectangle(self.X_ORIG + self.X_UNIT * xidx, self.Y_ORIG + self.Y_UNIT * yidx, self.X_ORIG + self.X_UNIT * (xidx + 1), self.Y_ORIG + self.Y_UNIT * (yidx + 1))
  
  
  def unitLength(self, x_or_y:str=None):
    if self.family() == "web_mercator":
      return self.X_UNIT
    else:
      assert x_or_y is not None and x_or_y in ["x", "y"], "The argument must be either 'x' or 'y'."
      if x_or_y == "x":
        return self.X_UNIT
      else:
        return self.Y_UNIT
  
  def origin(self):
    return self.X_ORIG, self.Y_ORIG
  
  def xyIdxExtentEdge(self, extent:QgsReferencedRectangle):
    extent_tr = self.transformExtent(extent)
    
    if self.FAMILY == "worldmesh":
      return None
    else:
      xy_ul = (extent_tr.xMinimum(), extent_tr.yMaximum())
      xy_lr = (extent_tr.xMaximum(), extent_tr.yMinimum())
      x_ul, y_ul = self.cellXyIdx(*xy_ul)
      x_lr, y_lr = self.cellXyIdx(*xy_lr)
      return min(x_ul, x_lr), max(x_ul, x_lr), min(y_lr, y_ul), max(y_lr, y_ul)
     
  def codeToXYCenter(self, code):
    if self.FAMILY == "worldmesh":
      latlong_dict = meshcode_to_latlong(code)
      x_edge = latlong_dict["long"]
      y_edge = latlong_dict["lat"]
    else:
      idxs = code.split("_")
      x_edge = self.X_ORIG + self.X_UNIT * int(idxs[0])
      y_edge = self.Y_ORIG + self.Y_UNIT * int(idxs[1])
    
    return (x_edge + self.X_UNIT / 2, y_edge + self.Y_UNIT / 2)
  
  def codeToRectangle(self, code):
    xc, yc = self.codeToXYCenter(code)
    
    xmin = xc - math.abs(self.X_UNIT/2)
    xmax = xc + math.abs(self.X_UNIT/2)
    ymin = yc - math.abs(self.Y_UNIT/2)
    ymax = yc + math.abs(self.Y_UNIT/2)
    
    rect = QgsReferencedRectangle(QgsRectangle(xmin, ymin, xmax, ymax), self.CRS)
    
    return rect
  
  def transformExtent(self, extent:QgsReferencedRectangle):
    # Note that eather CRS argument of QgsCoordinateTransform must be the CRS of the QgsProject
    tr1 = QgsCoordinateTransform(extent.crs(), QgsProject.instance().crs(), QgsProject.instance())
    extent_tr1 = QgsReferencedRectangle(tr1.transformBoundingBox(extent), QgsProject.instance().crs())    
    tr2 = QgsCoordinateTransform(QgsProject.instance().crs(), self.CRS, QgsProject.instance())
    extent_tr2 = QgsReferencedRectangle(tr2.transformBoundingBox(extent_tr1), self.CRS)
    return extent_tr2
    
  def xyIdxExtentAll(self, extent:QgsReferencedRectangle):
    extent_tr = self.transformExtent(extent)
    xmin, xmax, ymin, ymax = self.xyIdxExtentEdge(extent_tr)
    return(list(itertools.product(range(xmin, xmax+1), range(ymin, ymax+1))))
            
  
  def codeExtentAll(self, extent:QgsReferencedRectangle):
    
    extent_tr = self.transformExtent(extent)
    
    if self.FAMILY == "worldmesh":
      xmin = extent_tr.xMinimum()
      xmax = extent_tr.xMaximum()
      ymin = extent_tr.yMinimum()
      ymax = extent_tr.yMaximum()
      
      x_grid = [
        xmin + self.X_UNIT * i 
        for i 
        in range(0, 1 + math.floor((xmax - xmin) / self.X_UNIT))
      ] + [xmax]
      y_grid = [
        ymin + self.Y_UNIT * i 
        for i 
        in range(0, 1 + math.floor((ymax - ymin) / self.Y_UNIT))
      ] + [ymax]
      
      codes = []
      for x, y in itertools.product(x_grid, y_grid):
        mcode = self.cellMeshCode(x, y)
        if not mcode in codes:
          codes.append(mcode)
    
    else:
      xy_idx = self.xyIdxExtentAll(extent_tr)
      codes = []
      for tx, ty in xy_idx:
        codes.append(f"{tx}_{ty}")
        
    return codes
            
      