from qgis.PyQt.QtCore import (QT_TRANSLATE_NOOP, QVariant, QCoreApplication)
from qgis.core import (
  QgsProcessing,
  QgsProcessingAlgorithm,
  QgsProcessingParameterFeatureSource,
  QgsProcessingParameterFeatureSink,
  QgsProcessingParameterBoolean,
  QgsFeatureRequest,
  QgsVectorLayer,
  QgsField
  )


from ..algutil.hriskvar import PostProcessors
from ..algutil.hriskutil import HrUtil
from ..algutil.hriskfields import HrFields
from ..algutil.hriskpostprocessor import HrPostProcessor

class initroadtraffictable(QgsProcessingAlgorithm):
  PARAMETERS = {                  
    "INPUT": {
      "ui_func": QgsProcessingParameterFeatureSource,
      "ui_args":{
        "description": QT_TRANSLATE_NOOP("initroadtraffictable","Road Traffic table"),
        "types": [QgsProcessing.TypeVectorLine, QgsProcessing.TypeVectorPoint, QgsProcessing.TypeVectorPolygon],
        "optional": True
      }
    },
    "OVERWRITE": {
      "ui_func": QgsProcessingParameterBoolean,
      "ui_args":{
        "description" : QT_TRANSLATE_NOOP("initroadtraffictable","Overwrite existing fields?"),
        "defaultValue": True
      }
    },
    "OUTPUT": {
      "ui_func": QgsProcessingParameterFeatureSink,
      "ui_args": {
        "description": QT_TRANSLATE_NOOP("initroadtraffictable","Road traffic table" )
      }
    }
  }
  
  FIELDS = {}
  FIELDS_WITH_VALUES = {}
  
  def __init__(self):
    super().__init__()
    self.UTIL = HrUtil(self)
    
    self.FIELDS["COMMON"] = HrFields.fromQgsFieldList([
      QgsField("PK", QVariant.Int, "int"),
      QgsField("HISTORY", QVariant.String, "string")      
    ]).setComments("note: common fields; ")
    
    self.FIELDS["REQUIRED"] = HrFields.fromQgsFieldList([
      QgsField("IdSource", QVariant.Int, "integer", comment = "default: 0; "), # Traffic ID
      QgsField("Period",   QVariant.String, "string", comment = "default: D; "), # time period
      QgsField("LV",  QVariant.Double, "double", comment = "default: 0.0; "), # light vehicle
      QgsField("MV",  QVariant.Double, "double", comment = "default: 0.0; "), # medium vehicle
      QgsField("HGV", QVariant.Double, "double", comment = "default: 0.0; "), # heavy duty vehicle
      QgsField("WAV", QVariant.Double, "double", comment = "default: 0.0; "), # mopeds
      QgsField("WBV", QVariant.Double, "double", comment = "default: 0.0; "), # motorcycles
      QgsField("LV_SPD",  QVariant.Double, "double", comment = "default: 0.0; "), # light vehicle
      QgsField("MV_SPD",  QVariant.Double, "double", comment = "default: 0.0; "), # medium vehicle
      QgsField("HGV_SPD", QVariant.Double, "double", comment = "default: 0.0; "), # heavy duty vehicle
      QgsField("WAV_SPD", QVariant.Double, "double", comment = "default: 0.0; "), # mopeds
      QgsField("WBV_SPD", QVariant.Double, "double", comment = "default: 0.0; "), # motorcycles
      QgsField("pvmt",      QVariant.String, "string", comment = "default: DEF; "), # CNOSSOS road pavement identifier
      QgsField("ts_stud",   QVariant.Double, "double", comment = "default: 0.0; "), # A limited period Ts (in months) over the year where a average proportion pm_stud of light vehicles are equipped with studded tyres (6-18h)
      QgsField("junc_dist", QVariant.Double, "double", comment = "default: 999.0; "), # Distance to junction in meters
      QgsField("junc_type", QVariant.Int,    "int",    comment = "default: 0; "), # Type of junction (k=0 none, k = 1 for a crossing with traffic lights ; k = 2 for a roundabout)
      QgsField("slope",     QVariant.Double, "double", comment = "default: 0.0; "), # Slope (in %) of the road section. If the field is not filled in, the LINESTRING z-values will be used
      QgsField("way",       QVariant.Int   , "int",    comment = "default: 3; ") # the way of the road section. 1 = one way road section and the traffic goes in the same way 
    ])
  
  def initAlgorithm(self, config):
    _, _ = self.UTIL.getExtentAndCrsUsingCanvas()
    self.UTIL.initParameters()
    
  def processAlgorithm(self, parameters, context, feedback):
    
    self.UTIL.registerProcessingParameters(parameters, context, feedback)
    self.CURRENT_PROCESS = self.UTIL.parseCurrentProcess()
    
    try:
      input_lyr = self.parameterAsSource(parameters, "INPUT", context).materialize(QgsFeatureRequest(), feedback)
    except:
      input_lyr = QgsVectorLayer(f"Point", "temp", "memory")
      
    try:
      self.FIELDS["EXISTING"] = HrFields.fromQgsFields(
        input_lyr.fields()
      ).setComments("note: existing fields; ", append = False)
    except:
      feedback.reportError(self.tr("The fields of input layer is invalid. The names may be duplicated (case-insensitive)."))
      raise Exception
    
    # Set the fields with default values
    overwrite = self.parameterAsBoolean(parameters, "OVERWRITE", context)
    n = input_lyr.featureCount()
    
    # the common fields are always overwritten
    self.FIELDS_WITH_VALUES = {
      "PK": {
        "type": QVariant.Int, 
        "value": list(range(1, n+1))
      },
      "HISTORY": {
        "type": QVariant.String, 
        "value": self.CURRENT_PROCESS,
        "append": True
      }
    }
    
    # the required fields are added, if not existing and overwrite flag is not set
    for fld, defval in zip(self.FIELDS["REQUIRED"], self.FIELDS["REQUIRED"].defaultValues()):
      if fld.name().lower() in self.FIELDS["EXISTING"].lowerNames() and not overwrite:
          feedback.pushWarning(self.tr("Existing field is used: ") + fld.name())
      else:
        self.FIELDS_WITH_VALUES[fld.name()] = {
          "type": fld.type(),
          "value": defval
        }
        
    # Display the results
    feedback.pushInfo(self.tr("The following fields are initialized: "))
    feedback.pushInfo("; ".join([str(key) for key in self.FIELDS_WITH_VALUES.keys()]))
        
    dest_id = self.UTIL.outputVectorLayer(
      vector_layer= input_lyr,
      param_sink = "OUTPUT",
      fields_with_values= self.FIELDS_WITH_VALUES
    )
    
    PostProcessors[dest_id] = HrPostProcessor(history = [self.CURRENT_PROCESS])
              
    return {"OUTPUT": dest_id}
  
  def name(self):
    return self.__class__.__name__

  def displayName(self):
    return self.tr("Road traffic table")

  def group(self):
    return self.tr("Initialize features")

  def groupId(self):
    return "initfeature"

  def createInstance(self):
    return initroadtraffictable()

  # placing here is necessary, when employing pylupdate
  def tr(self, string):
    return QCoreApplication.translate(self.__class__.__name__, string)
