from qgis.PyQt.QtCore import (QT_TRANSLATE_NOOP, QVariant, QCoreApplication)
from qgis.core import (
  QgsProcessing,
  QgsProcessingAlgorithm,
  QgsProcessingParameterFeatureSource,
  QgsProcessingParameterFeatureSink,
  QgsProcessingParameterBoolean,
  QgsFeatureRequest,
  QgsVectorLayer,
  QgsField
  )


from ..algutil.hriskvar import PostProcessors
from ..algutil.hriskutil import HrUtil
from ..algutil.hriskfields import HrFields
from ..algutil.hriskpostprocessor import HrPostProcessor

class initemissiontable(QgsProcessingAlgorithm):
  PARAMETERS = {                  
    "INPUT": {
      "ui_func": QgsProcessingParameterFeatureSource,
      "ui_args":{
        "description": QT_TRANSLATE_NOOP("initemissiontable","Emission table"),
        "types": [QgsProcessing.TypeVectorLine, QgsProcessing.TypeVectorPoint, QgsProcessing.TypeVectorPolygon],
        "optional": True
      }
    },
    "OVERWRITE": {
      "ui_func": QgsProcessingParameterBoolean,
      "ui_args":{
        "description" : QT_TRANSLATE_NOOP("initemissiontable","Overwrite existing fields?"),
        "defaultValue": True
      }
    },
    "OUTPUT": {
      "ui_func": QgsProcessingParameterFeatureSink,
      "ui_args": {
        "description": QT_TRANSLATE_NOOP("initemissiontable","Emission table" )
      }
    }
  }
  
  FIELDS = {}
  FIELDS_WITH_VALUES = {}
  
  def __init__(self):
    super().__init__()
    self.UTIL = HrUtil(self)
    
    self.FIELDS["COMMON"] = HrFields.fromQgsFieldList([
      QgsField("PK", QVariant.Int, "int"),
      QgsField("HISTORY", QVariant.String, "string")      
    ]).setComments("note: common fields; ")
    
    self.FIELDS["REQUIRED"] = HrFields.fromQgsFieldList([
      QgsField("IdSource", QVariant.Int, "integer", comment = "default: 0; "), # Source ID, which shall be linked to the PK of the source layer
      QgsField("LW63",   QVariant.Double, "double", comment = "default: -99.0; "), # Octave-band sound power level at 63 Hz (dB) 
      QgsField("LW125",  QVariant.Double, "double", comment = "default: -99.0; "), # Octave-band sound power level at 125 Hz (dB)
      QgsField("LW250",  QVariant.Double, "double", comment = "default: -99.0; "), # Octave-band sound power level at 250 Hz (dB)
      QgsField("LW500",  QVariant.Double, "double", comment = "default: -99.0; "), # Octave-band sound power level at 500 Hz (dB)
      QgsField("LW1000", QVariant.Double, "double", comment = "default: -99.0; "), # Octave-band sound power level at 1000 Hz (dB)
      QgsField("LW2000", QVariant.Double, "double", comment = "default: -99.0; "), # Octave-band sound power level at 2000 Hz (dB)
      QgsField("LW4000", QVariant.Double, "double", comment = "default: -99.0; "), # Octave-band sound power level at 4000 Hz (dB)
      QgsField("LW8000", QVariant.Double, "double", comment = "default: -99.0; "), # Octave-band sound power level at 8000 Hz (dB)
    ])
  
  def initAlgorithm(self, config):
    (_, target_crs) = self.UTIL.getExtentAndCrsUsingCanvas()
    self.UTIL.initParameters()
    
  def processAlgorithm(self, parameters, context, feedback):
    
    self.UTIL.registerProcessingParameters(parameters, context, feedback)
    self.CURRENT_PROCESS = self.UTIL.parseCurrentProcess()
    

    try:
      input_lyr = self.parameterAsSource(parameters, "INPUT", context).materialize(QgsFeatureRequest(), feedback)
    except:
      input_lyr = QgsVectorLayer(f"Point", "temp", "memory")
      
    try:
      self.FIELDS["EXISTING"] = HrFields.fromQgsFields(
        input_lyr.fields()
      ).setComments("note: existing fields; ", append = False)
    except:
      feedback.reportError(self.tr("The fields of input layer is invalid. The names may be duplicated (case-insensitive)."))
      raise Exception
    
    # Set the fields with default values
    overwrite = self.parameterAsBoolean(parameters, "OVERWRITE", context)
    n = input_lyr.featureCount()
    
    # the common fields are always overwritten
    self.FIELDS_WITH_VALUES = {
      "PK": {
        "type": QVariant.Int, 
        "value": list(range(1, n+1))
      },
      "HISTORY": {
        "type": QVariant.String, 
        "value": self.CURRENT_PROCESS,
        "append": True
      }
    }
    
    # the required fields are added, if not existing and overwrite flag is not set
    for fld, defval in zip(self.FIELDS["REQUIRED"], self.FIELDS["REQUIRED"].defaultValues()):
      if fld.name().lower() in self.FIELDS["EXISTING"].lowerNames() and not overwrite:
          feedback.pushWarning(self.tr("Existing field is used: ") + fld.name())
      else:
        self.FIELDS_WITH_VALUES[fld.name()] = {
          "type": fld.type(),
          "value": defval
        }
        
    # Display the results
    feedback.pushInfo(self.tr("The following fields are initialized: "))
    feedback.pushInfo("; ".join([str(key) for key in self.FIELDS_WITH_VALUES.keys()]))
        
    dest_id = self.UTIL.outputVectorLayer(
      vector_layer= input_lyr,
      param_sink = "OUTPUT",
      fields_with_values= self.FIELDS_WITH_VALUES
    )
    
    PostProcessors[dest_id] = HrPostProcessor(history = [self.CURRENT_PROCESS])
              
    return {"OUTPUT": dest_id}
  
  def name(self):
    return self.__class__.__name__

  def displayName(self):
    return self.tr("Emission table")

  def group(self):
    return self.tr("Initialize features")

  def groupId(self):
    return "initfeature"

  def createInstance(self):
    return initemissiontable()

  # placing here is necessary, when employing pylupdate
  def tr(self, string):
    return QCoreApplication.translate(self.__class__.__name__, string)
