from qgis.PyQt.QtCore import (QT_TRANSLATE_NOOP, QVariant, QCoreApplication)
from qgis.core import (
  QgsProcessing,
  QgsProcessingAlgorithm,
  QgsProcessingParameterFeatureSource,
  QgsProcessingParameterFeatureSink,
  QgsFeatureRequest,
  QgsVectorLayer,
  QgsProcessingParameterCrs,
  QgsField,
  QgsProperty,
  QgsProcessingParameterField
  )

from qgis import processing


from ..algutil.hriskvar import PostProcessors
from ..algutil.hriskutil import HrUtil
from ..algutil.hriskfields import HrFields
from ..algutil.hriskpostprocessor import HrPostProcessor

class initelevationpoint(QgsProcessingAlgorithm):
  PARAMETERS = {                  
    "INPUT": {
      "ui_func": QgsProcessingParameterFeatureSource,
      "ui_args":{
        "description": QT_TRANSLATE_NOOP("initelevationpoint","Point layer"),
        "types": [QgsProcessing.TypeVectorPoint],
        "optional": True
      }
    },    
    "FIELD_USE_AS_HEIGHT": {
      "ui_func": QgsProcessingParameterField,
      "ui_args": {
        "optional": True,
        "description": QT_TRANSLATE_NOOP("initelevationpoint","Field indicating the height (if not specified, Z values of the points are used)"),
        "parentLayerParameterName": "INPUT",
        "defaultValue": None
      }
    },
    "TARGET_CRS": {
      "ui_func": QgsProcessingParameterCrs,
      "ui_args": {
        "description": QT_TRANSLATE_NOOP("initelevationpoint","Target CRS (Cartesian coordinates)")
      }
    },
    "OUTPUT": {
      "ui_func": QgsProcessingParameterFeatureSink,
      "ui_args": {
        "description": QT_TRANSLATE_NOOP("initelevationpoint","Elevation points" )
      }
    }
  }
  
  FIELDS = {}
  
  def __init__(self):
    super().__init__()
    self.UTIL = HrUtil(self)
    
    self.FIELDS["COMMON"] = HrFields.fromQgsFieldList([
      QgsField("PK", QVariant.Int, "int"),
      QgsField("HISTORY", QVariant.String, "string")      
    ]).setComments("note: common fields; ")
    
    self.FIELDS["REQUIRED"] = HrFields.fromQgsFieldList([
      QgsField("alti", QVariant.Double, "double", comment = "default: $z;"),
    ]).setComments("note: required fields; ")
    
      
  def initAlgorithm(self, config):
    (_, target_crs) = self.UTIL.getExtentAndCrsUsingCanvas()
    self.UTIL.setDefaultValue("TARGET_CRS", target_crs.authid())
    self.UTIL.initParameters()
    
  def processAlgorithm(self, parameters, context, feedback):   
    
    self.UTIL.registerProcessingParameters(parameters, context, feedback)
    self.CURRENT_PROCESS = self.UTIL.parseCurrentProcess()

    # Re-project the input layer to the target CRS
    target_crs = self.parameterAsCrs(parameters, "TARGET_CRS", context)
    self.UTIL.checkCrsAsCartesian(target_crs)
    
    try:
      input_lyr = self.parameterAsSource(parameters, "INPUT", context).materialize(QgsFeatureRequest(), feedback)
    except:
      input_lyr = QgsVectorLayer(f"Polygon?crs={target_crs.authid()}", "temp", "memory")
      
    input_lyr_proj = processing.run(
      "native:reprojectlayer", 
      {
        "INPUT": input_lyr,
        "TARGET_CRS": target_crs,
        "OUTPUT": "TEMPORARY_OUTPUT"
      },
      context = context,
      is_child_algorithm = True
    )["OUTPUT"]                  
    
    # Set Z values
    fld_z = self.parameterAsString(parameters, "FIELD_USE_AS_HEIGHT", context)
    if len(fld_z) == 0:
      input_lyr_fixed = input_lyr_proj
    else:
      input_lyr_fixed = processing.run(
        "native:setzvalue",
        {
          "INPUT": input_lyr_proj,
          "Z_VALUE": QgsProperty.fromExpression(f'"{fld_z}"'),
          "OUTPUT": "memory:dem"
        },
        context = context,
        is_child_algorithm = True
      )["OUTPUT"]
    
    input_lyr_fixed = context.getMapLayer(input_lyr_fixed)
    
    # Import the fields of the input layer
    try:
      self.FIELDS["EXISTING"] = HrFields.fromQgsFields(
        input_lyr_fixed.fields()
      ).setComments("note: existing fields; ", append = False)
    except:
      feedback.reportError(self.tr("The fields of input layer is invalid. The names may be duplicated (case-insensitive)."))
      raise Exception
    
    # Set the fields with default values
    n = input_lyr_fixed.featureCount()
    
    # the common fields are always overwritten
    self.FIELDS_WITH_VALUES = {
      "PK": {
        "type": QVariant.Int, 
        "value": list(range(1, n+1))
      },
      "HISTORY": {
        "type": QVariant.String, 
        "value": self.CURRENT_PROCESS,
        "append": True
      }
    }
    
    # Set the required fields properties
    self.FIELDS_WITH_VALUES |= {
      "alti": {
        "type": QVariant.Double,
        "value": [ft.geometry().get().z() for ft in input_lyr_fixed.getFeatures()]
      }
    }
        
    # Display the results
    feedback.pushInfo(self.tr("The following fields are initialized: "))
    feedback.pushInfo("; ".join([str(key) for key in self.FIELDS_WITH_VALUES.keys()]))
    
    dest_id = self.UTIL.outputVectorLayer(
      vector_layer= input_lyr_fixed,
      param_sink = "OUTPUT",
      fields_with_values= self.FIELDS_WITH_VALUES
    )
    
    PostProcessors[dest_id] = HrPostProcessor(history = [self.CURRENT_PROCESS])
    
    return {"OUTPUT": dest_id}
  
  def name(self):
    return self.__class__.__name__
  
  def displayName(self):
    return self.tr("Elevation point")

  def group(self):
    return self.tr("Initialize features")

  def groupId(self):
    return "initfeature"

  def createInstance(self):
    return initelevationpoint()

  # placing here is necessary, when employing pylupdate
  def tr(self, string):
    return QCoreApplication.translate(self.__class__.__name__, string)
