from qgis.PyQt.QtCore import (QT_TRANSLATE_NOOP, QVariant, QCoreApplication)
from qgis.core import (
  QgsProcessing,
  QgsProcessingAlgorithm,
  QgsProcessingParameterFeatureSource,
  QgsProcessingParameterFeatureSink,
  QgsProcessingParameterBoolean,
  QgsFeatureRequest,
  QgsVectorLayer,
  QgsProcessingParameterCrs,
  QgsField,
  QgsProcessingParameterNumber
  )

from qgis import processing

from ..algutil.hriskvar import PostProcessors
from ..algutil.hriskutil import HrUtil
from ..algutil.hriskfields import HrFields
from ..algutil.hriskpostprocessor import HrPostProcessor
class initbuilding(QgsProcessingAlgorithm):
  PARAMETERS = {                  
    "INPUT": {
      "ui_func": QgsProcessingParameterFeatureSource,
      "ui_args":{
        "description": QT_TRANSLATE_NOOP("initbuilding","Polygon layer"),
        "types": [QgsProcessing.TypeVectorPolygon],
        "optional": True
      }
    },
    "TARGET_CRS": {
      "ui_func": QgsProcessingParameterCrs,
      "ui_args": {
        "description": QT_TRANSLATE_NOOP("initbuilding","Target CRS (Cartesian coordinates)")
      }
    },
    "HEIGHT": {
      "ui_func": QgsProcessingParameterNumber,
      "ui_args":{
        "optional": True,
        "description" : QT_TRANSLATE_NOOP("initbuilding","Default height of buildings"),
        "defaultValue": 8.0
      }
    },
    "OVERWRITE": {
      "ui_func": QgsProcessingParameterBoolean,
      "ui_args":{
        "description" : QT_TRANSLATE_NOOP("initbuilding","Overwrite existing fields?"),
        "defaultValue": True
      }
    },
    "OUTPUT": {
      "ui_func": QgsProcessingParameterFeatureSink,
      "ui_args": {
        "description": QT_TRANSLATE_NOOP("initbuilding","Building" )
      }
    }
  }
  
  FIELDS = {}
  FIELDS_WITH_VALUES = {}
  
  def __init__(self):
    super().__init__()
    self.UTIL = HrUtil(self)
    
    self.FIELDS["COMMON"] = HrFields.fromQgsFieldList([
      QgsField("PK", QVariant.Int, "int"),
      QgsField("HISTORY", QVariant.String, "string")      
    ]).setComments("note: common fields; ")
    
      
  def initAlgorithm(self, config):
    (_, target_crs) = self.UTIL.getExtentAndCrsUsingCanvas()
    self.UTIL.setDefaultValue("TARGET_CRS", target_crs.authid())
    self.UTIL.initParameters()
    
  def processAlgorithm(self, parameters, context, feedback):   
    
    self.UTIL.registerProcessingParameters(parameters, context, feedback)
    self.CURRENT_PROCESS = self.UTIL.parseCurrentProcess()
    
    # Re-project the input layer to the target CRS
    target_crs = self.parameterAsCrs(parameters, "TARGET_CRS", context)
    self.UTIL.checkCrsAsCartesian(target_crs)

    try:
      input_lyr_proj = processing.run(
        "native:reprojectlayer", 
        {
          "INPUT": self.parameterAsSource(parameters, "INPUT", context).materialize(QgsFeatureRequest(), feedback),
          "TARGET_CRS": target_crs,
          "OUTPUT": "TEMPORARY_OUTPUT"
        },
        context = context,
        is_child_algorithm = True
      )["OUTPUT"]      
      input_lyr_fixed = context.getMapLayer(input_lyr_proj)
            
    except:
      input_lyr_proj = QgsVectorLayer(f"Polygon?crs={target_crs.authid()}", "temp", "memory")
      input_lyr_fixed = input_lyr_proj
    
    
    # Import the fields of the input layer
    try:
      self.FIELDS["EXISTING"] = HrFields.fromQgsFields(
        input_lyr_fixed.fields()
      ).setComments("note: existing fields; ", append = False)
    except:
      feedback.reportError(self.tr("The fields of input layer is invalid. The names may be duplicated (case-insensitive)."))
      raise Exception
    
    # Set the fields with default values
    overwrite = self.parameterAsBoolean(parameters, "OVERWRITE", context)
    n = input_lyr_fixed.featureCount()
    
    # the common fields are always overwritten
    self.FIELDS_WITH_VALUES = {
      "PK": {
        "type": QVariant.Int, 
        "value": list(range(1, n+1))
      },
      "HISTORY": {
        "type": QVariant.String, 
        "value": self.CURRENT_PROCESS,
        "append": True
      }
    }
    
    # Set the required fields properties
    height_default = self.parameterAsDouble(parameters, "HEIGHT", context)
    self.FIELDS["REQUIRED"] = HrFields.fromQgsFieldList([
      QgsField("height", QVariant.Double, "double", comment = f"default: {height_default};"),
      QgsField("pop",    QVariant.Double, "double", comment = "default: 0.0;")
    ]).setComments("note: required fields; ")
    
    # the required fields are added, if not existing and overwrite flag is not set
    for fld, defval in zip(self.FIELDS["REQUIRED"], self.FIELDS["REQUIRED"].defaultValues()):
      if fld.name().lower() in self.FIELDS["EXISTING"].lowerNames() and not overwrite:
          feedback.pushWarning(self.tr("Existing field is used: ") + fld.name())
      else:
        self.FIELDS_WITH_VALUES[fld.name()] = {
          "type": fld.type(),
          "value": defval
        }
    
    # Display the results
    feedback.pushInfo(self.tr("The following fields are initialized: "))
    feedback.pushInfo("; ".join([str(key) for key in self.FIELDS_WITH_VALUES.keys()]))
    
    dest_id = self.UTIL.outputVectorLayer(
      vector_layer= input_lyr_fixed,
      param_sink = "OUTPUT",
      fields_with_values= self.FIELDS_WITH_VALUES
    )
    
    PostProcessors[dest_id] = HrPostProcessor(history = [self.CURRENT_PROCESS])
    
    return {"OUTPUT": dest_id}
  
  def name(self):
    return self.__class__.__name__
  
  def displayName(self):
    return self.tr("Building")

  def group(self):
    return self.tr("Initialize features")

  def groupId(self):
    return "initfeature"

  def createInstance(self):
    return initbuilding()

  # placing here is necessary, when employing pylupdate
  def tr(self, string):
    return QCoreApplication.translate(self.__class__.__name__, string)
