/**
 * Receiver Points at Regular Grid Generation Script
 * 
 * This Groovy script generates receiver points at regular grid positions.
 * It integrates with the NoiseModelling framework to process building and source files
 * and generate receivers at regular grid points for acoustic modeling purposes.
 * 
 * Dependencies:
 * - H2GIS database for spatial data processing
 * - NoiseModelling framework for acoustic calculations
 * - SLF4J and Log4j for logging
 * 
 * @author Junta Tagusari
 * @version 1.0
 */
 
import org.h2gis.api.ProgressVisitor;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import java.sql.Connection;
import java.nio.file.Path;
import java.nio.file.Paths;
import org.apache.log4j.PropertyConfigurator;
import java.io.FileWriter;
import java.io.PrintWriter;

title = 'Receiver at regular grid'
description = 'Geometry of receivers at grid points are obtained'

inputs = [
  buildingGeomPath:[
    name : "Path of the building file",
    title : "Path of the building file",
    description : "Path of the building file",
    min        : 0, max: 1,
    type : String.class
  ],
  sourceGeomPath:[
    name : "Path of the source file",
    title : "Path of the source file",
    description : "Path of the source file",
    min        : 0, max: 1,
    type : String.class
  ],
  fenceGeomPath:[
    name : "Path of the fence file",
    title : "Path of the fence file",
    description : "Path of the fence file",
    min        : 0, max: 1,
    type : String.class
  ],
  inputSRID: [
    name: 'Projection identifier',
    title: 'Projection identifier',
    description: 'Original projection identifier (also called SRID) of your table. It should be an EPSG code, a integer with 4 or 5 digits (ex: 3857 is Web Mercator projection). ' +
            '</br>  All coordinates will be projected from the specified EPSG to WGS84 coordinates. ' +
            '</br> This entry is optional because many formats already include the projection and you can also import files without geometry attributes.</br> ' +
            '</br> <b> Default value : 4326 </b> ',
    type: Integer.class,
    min: 0, max: 1
  ],
  delta: [
    name       : 'Receivers minimal distance',
    title      : 'Distance between receivers',
    description: 'Distance between receivers in the Cartesian plane in meters',
    min        : 0, max: 100,
    type       : Double.class
  ],
  height: [
    name       : 'height',
    title      : 'height',
    description: 'Height of receivers in meters (FLOAT)' +
            '</br> </br> <b> Default value : 4 </b> ',
    min        : 0, max: 10,
    type       : Double.class
  ],
  noiseModellingHome : [
    name: "Path of NOISEMODELLING_HOME",
    title: "Path of NOISEMODELLING_HOME",
    description: "Path of NOISEMODELLING_HOME",
    type : String.class
  ],
  exportDir : [
    name: "Path of export directory",
    title: "Path of export directory",
    description: "Path of export directory",
    min        : 0, max: 1,
    type : String.class
  ]
]

outputs = [
  result: [
    name: 'Result output string', 
    title: 'Result output string', 
    description: 'This type of result does not allow the blocks to be linked together.', 
    type: String.class
  ]
]


/**
 * Main execution function for regular grid receiver generation
 * This function orchestrates the regular grid receiver creation workflow:
 * 1. Clear existing database tables
 * 2. Import building, source, and fence geometry data
 * 3. Generate receiver points at regular grid positions
 * 4. Export results to files
 * 
 * @param connection H2GIS database connection
 * @param input Map containing input parameters (buildingGeomPath, sourceGeomPath, etc.)
 * @return List of result objects containing table names and export paths
 */
def exec(Connection connection, input) {
  Logger logger = LoggerFactory.getLogger("org.hrisk")
  
  try {
    logger.info("Starting noise calculation process")
    
    // Initialize the NoiseModelling utility class
    def noiseModelling = new NoiseModelling(connection)
    
    // Step 1: Clear any existing data in the database
    noiseModelling.clearDatabase()
    
    // Step 2: Import the geometry data into the database
    String tableBuilding = noiseModelling.importTable(input["buildingGeomPath"], input["inputSRID"])
    String sourcesTableName = noiseModelling.importTable(input["sourceGeomPath"], input["inputSRID"])
    String fenceTableName = noiseModelling.importTable(input["fenceGeomPath"], input["inputSRID"])

    // Get list of tables before running the noise calculation
    def tables_before_alg = noiseModelling.tableNames()
    logger.info("Current tables: " + tables_before_alg)

    // run calculation
    Map args = [
        "buildingTableName": tableBuilding, 
        "sourcesTableName": sourcesTableName, 
        "fenceTableName": fenceTableName, 
        "delta": input["delta"],
        "height": input["height"]
      ].findAll{ it.value!=null }

    noiseModelling.runScript(
      "Receivers/Regular_Grid",
      args
    )

    // Get list of tables after running the noise calculation
    def tables_after_alg = noiseModelling.tableNames()
    logger.info("Current tables: " + tables_after_alg)

    // Identify the new tables created by the calculation
    def result_tables = tables_after_alg - tables_before_alg
    logger.info("Result tables created by noise calculation: " + result_tables)

    // Step 4: Export all result tables to GeoJSON files
    results = []
    for (tbl in result_tables){
      geom_export = noiseModelling.exportTables(tbl, input["exportDir"])
      results.add(
        [
          "tableName": tbl,
          "exportPath": geom_export
        ]
      )
    }

    logger.info("Noise calculation process completed successfully")
    
    return results
  
  } finally {
    // Cleanup completed - no additional resources to close
    // No logWriter to close in this script
  }
}

