from qgis.PyQt.QtCore import (QT_TRANSLATE_NOOP, QCoreApplication, QVariant)
from qgis.core import (
  QgsProcessingParameterFolderDestination,
  QgsProcessingParameterFeatureSource,
  QgsProcessingParameterNumber,
  QgsProcessingParameterBoolean,
  QgsProcessingParameterString,
  QgsProcessingParameterField,
  QgsCoordinateReferenceSystem,
  QgsFeatureRequest,
  QgsReferencedRectangle,
  QgsProject,
  QgsVectorLayer,
  QgsVectorFileWriter,
  QgsProcessingFeedback,
  QgsFeatureSink,
  QgsProcessingUtils,
  QgsRasterLayer,
  QgsProcessingParameterExtent,
  QgsProcessingContext,  
  QgsProcessingParameterRasterLayer
)
from qgis import processing
import os
import shutil
import platform
import asyncio
import re
import stat

class NoiseModelling(object):
  CRS = None
  WORK_DIR = None
  SCRIPT_PATH = None
  NM_HOME = None
  JAVA_HOME = None
  ARGS = None
  FT_REQUEST = None
  FEEDBACK = None
  CMD_DIR = "bin"
  GROOVY_DIR = "groovy"
  GROOVY_UTIL = os.path.join(os.path.dirname(os.path.dirname(__file__)), "groovy", "NoiseModelling.groovy")
  OUT_DIR = "out"
  GEO_DIR = "geo"
  JAVA_CMD = None
  JAVA_RUNNER = None
  # OUTPUT_TARGET = None
  OVERWRITE = False
  NM_VERSION = None
  NM_GROOVY = None
  
  def __init__(
    self, 
    crs:QgsCoordinateReferenceSystem, 
    work_dir:str = None, 
    feedback: QgsProcessingFeedback = None,
    overwrite: bool = False
    ):
    self.CRS = crs
    self.FEEDBACK = feedback if feedback is not None else QgsProcessingFeedback()
    self.FT_REQUEST = QgsFeatureRequest()
    self.OVERWRITE = overwrite
    self.setWorkDir(work_dir)
    
    try:
      self.NM_HOME = os.environ["NOISEMODELLING_HOME"]
      self.NM_GROOVY_DIR = os.path.join(self.NM_HOME, "noisemodelling", "wps")
    except KeyError as e:
      raise Exception(self.tr(f"Environment variable of NOISEMODELLING_HOME is not set."))
    try:
      self.JAVA_HOME = os.environ["JAVA_FOR_NOISEMODELLING"]
    except KeyError as e:
      raise Exception(self.tr(f"Environment variable of JAVA_FOR_NOISEMODELLING is not set."))
        

  def setWorkDir(self, work_dir:str = None) -> None:
    self.WORK_DIR = work_dir if work_dir is not None else os.path.dirname(QgsProcessingUtils.generateTempFilename(""))
        
    self.CMD_DIR = os.path.join(self.WORK_DIR, self.CMD_DIR)
    
    self.GROOVY_DIR = os.path.join(
      self.WORK_DIR, 
      os.path.basename(self.GROOVY_DIR)
    )
    self.OUT_DIR = os.path.join(
      self.WORK_DIR, 
      os.path.basename(self.OUT_DIR)
    )
    self.GEO_DIR = os.path.join(
      self.WORK_DIR, 
      os.path.basename(self.GEO_DIR)
    )
    
    for dir in [self.CMD_DIR, self.GROOVY_DIR, self.OUT_DIR, self.GEO_DIR]:
      if not os.path.exists(dir):
        os.makedirs(dir)
    return
    
    
  def initArgs(self, groovy_script:str) -> None:
    
    assert os.path.exists(groovy_script), self.tr("Groovy script does not exist.")
    
    script_path = os.path.join(self.GROOVY_DIR, os.path.basename(groovy_script))
    
    if not self.OVERWRITE and os.path.exists(script_path):
      raise FileExistsError(self.tr("Groovy script already exists."))
    
    with open(script_path, "wb") as groovy_output:
      with open(self.GROOVY_UTIL, "rb") as groovy_util:
        shutil.copyfileobj(groovy_util, groovy_output)
      with open(groovy_script, "rb") as groovy_main:
        shutil.copyfileobj(groovy_main, groovy_output)
        
    shutil.copytree(self.NM_GROOVY_DIR, self.GROOVY_DIR, dirs_exist_ok=True)
    
    script_path_rel = f'"{os.path.relpath(script_path, self.WORK_DIR)}"'
    
    out_dir_rel = f'"{os.path.relpath(self.OUT_DIR, self.WORK_DIR)}"'
    
    self.ARGS = {
      "w": ".", 
      "s": script_path_rel,
      "d": "hriskdb",
      "noiseModellingHome": f'"{self.NM_HOME}"',
      "exportDir": out_dir_rel,
      "inputSRID": self.CRS.authid().replace("EPSG:", "")
    }
    
    # self.OUTPUT_TARGET = output_target
    
    self.FEEDBACK.pushCommandInfo("Working directory: " + "\n" + self.WORK_DIR + "\n")
    
    return
  
  def addArgs(self, **kwargs) -> None:
    for key, value in kwargs.items():
      if key in self.ARGS.keys():
        self.FEEDBACK.pushWarning(
          self.tr(f"The argument is already set. It will be overwritten.: ") + key
        )
      self.ARGS.update({key: value})
    return
    
  def setFeatureRequest(self, feature_request:QgsFeatureRequest=None, rect:QgsReferencedRectangle=None) -> None:
    assert rect is None or rect.crs() == self.CRS, self.tr("CRS of the rectangle does not match with the CRS of the NoiseModelling project.")
    if feature_request is not None:
      self.FT_REQUEST = feature_request
    elif rect is not None:
      ft = QgsFeatureRequest()
      self.FT_REQUEST = ft.setFilterRect(rect)
    else:
      raise Exception(self.tr("Feature request is not set."))
    return
  
  
  @classmethod
  def extentAsLayer(cls, context:QgsProcessingContext, extent:QgsProcessingParameterExtent, target_crs:QgsCoordinateReferenceSystem) -> QgsVectorLayer:
    """Converts the extent to a layer.

    Args:
        context (QgsProcessingContext): context of the processing
        fence_extent (QgsProcessingParameterExtent): extent of the fence
        target_crs (QgsCoordinateReferenceSystem): coordinate reference system

    Returns:
        QgsVectorLayer: Layer of the extent
    """
    extent_layer = processing.run(
      "native:extenttolayer",
      {
        "INPUT": extent,
        "OUTPUT": "TEMPORARY_OUTPUT"
      },
      context = context,
      is_child_algorithm = True
    )["OUTPUT"]
    
    extent_transformed = processing.run(
      "native:reprojectlayer",
      {
        "INPUT": extent_layer,
        "TARGET_CRS": target_crs,
        "OUTPUT": QgsProcessingUtils.generateTempFilename("") + "fence.geojson"
      },
      context = context,
      is_child_algorithm = True
    )["OUTPUT"]
    
    return extent_transformed
    
  @classmethod
  def extentAsWkt(cls, context:QgsProcessingContext, extent:QgsProcessingParameterExtent, target_crs:QgsCoordinateReferenceSystem) -> str:
    """Converts the extent to WKT.

    Args:
        context (QgsProcessingContext): context of the processing
        fence_extent (QgsProcessingParameterExtent): extent of the fence
        target_crs (QgsCoordinateReferenceSystem): coordinate reference system

    Returns:
        str: WKT of the extent
    """
    extent_layer = cls.extentAsLayer(context, extent, target_crs)      
    extent_layer = context.getMapLayer(extent_layer)
    return extent_layer.getFeature(1).geometry().asWkt()
  
  def uiFuncToType(self, ui_func) -> str:
    """Converts the UI function to the type.

    Args:
        ui_func (Union[QgsProcessingParameterString, QgsProcessingParameterField, QgsProcessingParameterBoolean, QgsProcessingParameterNumber]): UI function

    Returns:
        str: type of the UI function
    """
    if ui_func == QgsProcessingParameterFolderDestination:
      return "dir_path"
    elif ui_func == QgsProcessingParameterFeatureSource:
      return "vector_layer"
    elif ui_func == QgsProcessingParameterRasterLayer:
      return "raster_layer"
    elif ui_func in [QgsProcessingParameterString, QgsProcessingParameterField]:
      return "string"
    elif ui_func == QgsProcessingParameterBoolean:
      return "boolean"
    elif ui_func == QgsProcessingParameterNumber:
      return "number"
    else:
      return "unknown"

  def uiParametersToArg(self, ui_key, ui_settings:dict, value) -> None:
    if "nm_key" not in ui_settings.keys():
      return
    
    nm_key = ui_settings["nm_key"]
    assert "ui_func" in ui_settings.keys(), self.tr("UI function is not set.")
    
    type = self.uiFuncToType(ui_settings["ui_func"])
    
    if type == "vector_layer":
      self.vectorLayerToArg(value, ui_key, ui_settings, nm_key)
      return
    elif type == "raster_layer":
      self.rasterLayerToArg(value, ui_key, ui_settings, nm_key)
      return
    else:      
      value = int(value) if type == "boolean" else value
      value = f'"{value}"' if type == "string" else value
      self.addArgs(**{nm_key: value})
      return
    

  def vectorLayerToArg(self, layer_identifier:str, ui_key:str, ui_settings:dict, nm_key:str) -> None:
    layers = QgsProject.instance().mapLayers()
    if layer_identifier in layers.keys():
      src = layers[layer_identifier]
    else:
      assert os.path.exists(layer_identifier), self.tr("Layer file does not exist.")
      src = QgsVectorLayer(layer_identifier)
    
    if not ui_settings.get("ignore_crs", False) and src.crs().authid() != self.CRS.authid():
      raise Exception(self.tr("CRS is different: ") + ui_key)
    
    dst = os.path.join(self.GEO_DIR, ui_key + ".geojson")
    
    if os.path.exists(dst) and not self.OVERWRITE:
      raise FileExistsError(self.tr("File already exists."))
    
    save_options = QgsVectorFileWriter.SaveVectorOptions()
    save_options.driverName = "GeoJSON"
    writer = QgsVectorFileWriter.create(
      fileName = dst,
      fields = src.fields(),
      geometryType= src.wkbType(),
      srs = src.crs(),
      transformContext = QgsProject.instance().transformContext(),
      options = save_options,
      sinkFlags = QgsFeatureSink.SinkFlags()
    )
    
    for ft in src.getFeatures(self.FT_REQUEST):
      writer.addFeature(ft)
    
    dst = f'"{os.path.relpath(dst, self.WORK_DIR)}"'
    self.FEEDBACK.pushCommandInfo(f"Geometry was saved at: {dst}")
    
    self.addArgs(**{nm_key: dst})
    return
    
    
  def rasterLayerToArg(self, layer_identifier:str, ui_key:str, ui_settings:dict, nm_key:str) -> None:
    layers = QgsProject.instance().mapLayers()
    if layer_identifier in layers.keys():
      raster_layer = layers[layer_identifier]
    else:
      assert os.path.exists(layer_identifier), self.tr("Layer file does not exist.")
      raster_layer = QgsRasterLayer(layer_identifier)
    
    if not ui_settings.get("ignore_crs", False) and raster_layer.crs().authid() != self.CRS.authid():
      raise Exception(self.tr("CRS is different: ") + ui_key)
    
    if abs(raster_layer.rasterUnitsPerPixelX() - raster_layer.rasterUnitsPerPixelY()) > 1e-3:
      raise Exception(self.tr("Pixel size is not square: ") + ui_key)
    
    # note that only the DEM.asc is supported
    dst = os.path.join(self.GEO_DIR, "DEM.asc")

    if os.path.exists(dst) and not self.OVERWRITE:
      raise FileExistsError(self.tr("File already exists."))
    
    processing.run(
      "gdal:translate",
      {
        "INPUT": raster_layer,
        "DATA_TYPE": 0,
        "OUTPUT": dst,
        "TARGET_CRS": self.CRS,
        "NODATA": -9999,
        'EXTRA' : '-co "DECIMAL_PRECISION=3" -co "FORCE_CELLSIZE=YES"'
      },
      is_child_algorithm = True
    )
    
    dst = f'"{os.path.relpath(dst, self.WORK_DIR)}"'
    self.FEEDBACK.pushCommandInfo(f"Geometry was saved at: {dst}")
    
    self.addArgs(**{nm_key: dst})
    return
    
  def setJavaCommand(
    self, 
    nm_runner:str = "org.noisemodelling.runner.Main", # for NM 4.0.0
    # nm_runner:str = "org.noise_planet.noisemodelling.runner.Main", # for NM 5.0.0
    default_JVM_opts:str = "", 
    java_opts:str = "", 
    wps_scripts_opts:str = "",
    nm_args:dict = None,
    script_name = "nm_runner"
    ):
    
    lib_dir = os.path.join(self.NM_HOME,"lib")
    
    if not os.path.isdir(lib_dir):
      raise Exception(self.tr("Java libraries do not exist in NOISEMODELLING_HOME/lib"))
    
    cp = os.path.join(lib_dir, "*")
    
    
    pf = platform.system()
    if pf in ["Darwin","Linux"]:
      java_path = os.path.join(self.JAVA_HOME, "bin", "java")
      if not os.path.exists(java_path):
        java_path = os.path.join(self.JAVA_HOME, "jre", "bin", "java")
    elif pf == "Windows":     
      java_path = os.path.join(self.JAVA_HOME, "bin", "java.exe")
    
    if not os.path.exists(java_path):
      raise Exception(self.tr("Java executable does not exist in JAVA_FOR_NOISEMODELLING"))
        
    if pf == "Linux":
      try:
        import resource
        max_fd_limit = resource.getrlimit(resource.RLIMIT_NOFILE)[1]  # ハードリミットを取得
        max_fd = max_fd_limit  # Initialize max_fd with the hard limit
        resource.setrlimit(resource.RLIMIT_NOFILE, (int(max_fd), max_fd_limit))
      except ValueError:
        self.FEEDBACK.pushWarn(f"Could not set maximum file descriptor limit: {max_fd}")
      except Exception as e:
        self.FEEDBACK.pushWarn(f"Could not query maximum file descriptor limit: {e}")
    
    
    if pf == "Darwin":
      gradle_opts = os.environ.get("GRADLE_OPTS", "")
      gradle_opts += f' "-Xdock:name=NoiseModelling" "-Xdock:icon={os.environ["NOISEMODELLING_HOME"]}/media/gradle.icns"'
      os.environ["GRADLE_OPTS"] = gradle_opts
      print(f"Configured Dock options: {gradle_opts}")
    
    nm_args = nm_args if nm_args is not None else self.ARGS
    
    args_str = "".join([f" -{k} {v}" for k, v in nm_args.items()])
    
    self.JAVA_CMD = f'"{java_path}" {default_JVM_opts} {java_opts} {wps_scripts_opts} -classpath "{cp}" {nm_runner}' + args_str
    
    
    self.FEEDBACK.pushCommandInfo("Command for Run the NoiseModelling: " +"\n"+ self.JAVA_CMD + "\n")
    
    cmd_str = self.JAVA_CMD
    pf = platform.system()
    if pf in ["Darwin","Linux"]:
      self.JAVA_RUNNER = os.path.join(self.CMD_DIR, script_name)
      cmd_list = [
        '#!/bin/sh\n',
        'original_dir="$(pwd)"\n',
        'cd "$(dirname "$(dirname "$0")")"\n',
        f'{cmd_str}\n',
        'cd "$original_dir"'
      ]
    elif pf == "Windows":
      self.JAVA_RUNNER = os.path.join(self.CMD_DIR, script_name + ".bat") 
      cmd_list = [
        "@echo off\n",
        r'set "original_dir=%cd%"' + "\n",
        r'cd /d "%~dp0.."' + "\n",
        f'{cmd_str}\n',
        r'cd /d "%original_dir%"'
      ]
    
    with open(self.JAVA_RUNNER, "w") as f:
      f.writelines(cmd_list)
    
    if pf in ["Darwin", "Linux"]:
      os.chmod(self.JAVA_RUNNER, stat.S_IRUSR | stat.S_IWUSR | stat.S_IXUSR)  # ユーザーに実行権限を付与
    
    
    return
    
  def run(self):    
    loop = asyncio.new_event_loop()
    loop.run_until_complete(self.stream())
    loop.close()
    
    results = {
      os.path.splitext(file_name)[0]: os.path.join(self.OUT_DIR, file_name)
      for file_name in os.listdir(self.OUT_DIR)
    }
    return results
    
  async def stream(self) -> None:
    """Stream the noise modelling command.

    Args:
        cmd (str): NoiseModelling command
        temp_dir (str): Temporary directory

    Raises:
        Exception: NoiseModelling script was not successfully executed.
    """     
    
    proc = await asyncio.create_subprocess_shell(
      os.path.relpath(self.JAVA_RUNNER, self.WORK_DIR),
      stdout = asyncio.subprocess.PIPE,
      stderr = asyncio.subprocess.PIPE,
      cwd    = self.WORK_DIR
    )
    
    async def read_stream(stream, feedback_method, read_progress = False):
      while not stream.at_eof():
        line = await stream.readline()
        if line:
          try:
            proc_text = line.decode("utf-8", "ignore").replace("\n", "")
            proc_text = re.sub(r"<[^>]+>", " ", proc_text)
          except Exception:
            self.FEEDBACK.reportError(
                self.tr("NoiseModelling script was not successfully executed."),
                fatalError=True
            )
            raise Exception(self.tr("NoiseModelling script was not successfully executed."))

          if proc_text:
            feedback_method(proc_text)
              
          if read_progress:
            prg_match = re.search(r".*[0-9]+\.[0-9]+.*%", proc_text)
            if prg_match:
              self.FEEDBACK.setProgress(
                int(float(re.search(r"[0-9]+\.[0-9]+", prg_match.group()).group()))
              )
                  
    stdout_task = asyncio.create_task(read_stream(proc.stdout, self.FEEDBACK.pushConsoleInfo, True))
    stderr_task = asyncio.create_task(read_stream(proc.stderr, self.FEEDBACK.pushConsoleInfo, True))


    await asyncio.gather(stdout_task, stderr_task)
    
    await proc.wait()

  def tr(self, string):
    return QCoreApplication.translate(self.__class__.__name__, string)
  