# -*- coding: utf-8 -*-
import os, shutil
from PyQt5.QtGui import QColor, QIcon
from PyQt5.QtWidgets import QAction, QMessageBox
from PyQt5.QtCore import QVariant
from qgis.core import (
    QgsProject, QgsVectorLayer, QgsField, QgsFields, QgsFeature,
    QgsGeometry, QgsPointXY, QgsVectorFileWriter, QgsWkbTypes,
    QgsMarkerSymbol, QgsRasterMarkerSymbolLayer,
    QgsSimpleLineSymbolLayer, QgsCoordinateReferenceSystem,
    QgsPalLayerSettings, QgsVectorLayerSimpleLabeling, QgsTextFormat
)
from .hms_extractor_dialog import HMSExtractorDialog


class HMSExtractor:
    def __init__(self, iface):
        self.iface = iface
        self.plugin_dir = os.path.dirname(__file__)
        self.action = None
        self.dlg = None

    # ---------------------------------------------------------------------
    def initGui(self):
        icon_path = os.path.join(self.plugin_dir, "icon.png")

        self.action = QAction(QIcon(icon_path), "HMS Extractor", self.iface.mainWindow())
        self.action.setIconText("HMS")
        self.action.setToolTip("HMS Extractor")

        self.action.setPriority(QAction.HighPriority)
        self.action.triggered.connect(self.run)

        self.iface.addPluginToMenu("&HMS Extractor", self.action)
        self.iface.addToolBarIcon(self.action)

    # ---------------------------------------------------------------------
    def unload(self):
        self.iface.removePluginMenu("&HMS Extractor", self.action)
        self.iface.removeToolBarIcon(self.action)

    # ---------------------------------------------------------------------
    def run(self):
        self.dlg = HMSExtractorDialog(self.iface.mainWindow())
        self.dlg.lineEditOut.setText("HMS_Extracted_Results")

        self.dlg.lineEditFolder.textChanged.connect(self._refresh_basins)

        try:
            self.dlg.btnRun.clicked.disconnect()
        except:
            pass

        self.dlg.btnRun.clicked.connect(self._on_accept)
        self.dlg.show()

    # ---------------------------------------------------------------------
    def _refresh_basins(self):
        folder = self.dlg.lineEditFolder.text().strip()
        self.dlg.comboBasin.clear()
        if os.path.isdir(folder):
            basins = [f for f in os.listdir(folder) if f.lower().endswith(".basin")]
            basins.sort()
            self.dlg.comboBasin.addItems(basins)

    # ---------------------------------------------------------------------
    def _on_accept(self):
        vals = self.dlg.get_values()
        folder = vals["folder"]
        basin_name = vals["basin"]
        out_rel = vals["outdir"] or "HMS_Extracted_Results"

        load_elements = vals["checkLoadElements"]
        load_background = vals["checkLoadBackground"]

        if not os.path.isdir(folder):
            QMessageBox.warning(self.iface.mainWindow(), "HMS Extractor",
                                "Selecciona una carpeta válida del proyecto HMS.")
            return

        if not basin_name:
            QMessageBox.warning(self.iface.mainWindow(), "HMS Extractor",
                                "Selecciona un archivo .basin.")
            return

        basin_path = os.path.join(folder, basin_name)
        outdir = os.path.join(folder, out_rel)
        out_maps = os.path.join(outdir, "maps")
        os.makedirs(out_maps, exist_ok=True)

        elements = self.parse_basin(basin_path)
        background_maps = self.parse_background_maps(basin_path, folder) if load_background else []

        # ----------------------------------------------------
        # Copiar shapefiles del background
        # ----------------------------------------------------
        for bg in background_maps:
            src = bg.get("src_path")
            if not src or not src.lower().endswith(".shp"):
                continue
            if not os.path.exists(src):
                continue

            dst = os.path.join(out_maps, os.path.basename(src))
            shutil.copy2(src, dst)

            # copiar extensiones asociadas
            src_base = os.path.splitext(src)[0]
            dst_base = os.path.splitext(dst)[0]
            for ext in [".dbf", ".shx", ".prj", ".cpg"]:
                extra_src = src_base + ext
                if os.path.exists(extra_src):
                    shutil.copy2(extra_src, dst_base + ext)

            bg["copied_path"] = dst

        root = QgsProject.instance().layerTreeRoot()
        group_hms = root.addGroup("HMS Extracted Results")
        group_bg = root.addGroup("Background Layers")

        # ----------------------------------------------------
        # Cargar background
        # ----------------------------------------------------
        if load_background:
            for bg in reversed(background_maps):
                if not bg.get("shown"):
                    continue
                shp = bg.get("copied_path")
                if not shp:
                    continue

                name = os.path.splitext(os.path.basename(shp))[0]
                vl = QgsVectorLayer(shp, name, "ogr")
                if vl.isValid():
                    QgsProject.instance().addMapLayer(vl, False)
                    group_bg.addLayer(vl)

        crs = QgsCoordinateReferenceSystem("EPSG:32718")

        # =====================================================================
        #                    CREAR ELEMENTOS HMS EN QGIS
        # =====================================================================
        if load_elements:

            # ------------------- SUBBASINS -------------------
            etype = "subbasin"
            fields = QgsFields()
            for f in [
                ("name", QVariant.String),
                ("type", QVariant.String),
                ("down", QVariant.String),
                ("area", QVariant.Double),
                ("loss_meth", QVariant.String),
                ("cn", QVariant.Double),
                ("transform", QVariant.String),
                ("tc", QVariant.Double),
                ("stor_coeff", QVariant.Double)
            ]:
                fields.append(QgsField(f[0], f[1]))

            shp_path = os.path.join(outdir, f"{etype}.shp")
            writer = QgsVectorFileWriter(shp_path, "UTF-8", fields,
                                         QgsWkbTypes.Point, crs, "ESRI Shapefile")

            for name, e in elements.items():
                if e["type"] != etype:
                    continue
                if "x" in e:
                    feat = QgsFeature()
                    feat.setGeometry(QgsGeometry.fromPointXY(QgsPointXY(e["x"], e["y"])))
                    feat.setAttributes([
                        name, etype, e.get("downstream", ""),
                        e.get("area"), e.get("loss_meth"),
                        e.get("cn"), e.get("transform"),
                        e.get("tc"), e.get("stor_coeff")
                    ])
                    writer.addFeature(feat)

            del writer

            if os.path.exists(shp_path):
                layer = QgsVectorLayer(shp_path, etype, "ogr")
                QgsProject.instance().addMapLayer(layer, False)
                group_hms.addLayer(layer)
                self.apply_icon_symbol(layer, etype)
                self.enable_labels(layer)

            # ------------------- OTROS ELEMENTOS -------------------
            for etype in ["junction", "reservoir", "source", "sink"]:
                fields = QgsFields()
                fields.append(QgsField("name", QVariant.String))
                fields.append(QgsField("type", QVariant.String))
                fields.append(QgsField("down", QVariant.String))

                shp_path = os.path.join(outdir, f"{etype}.shp")
                writer = QgsVectorFileWriter(shp_path, "UTF-8", fields,
                                             QgsWkbTypes.Point, crs, "ESRI Shapefile")

                for name, e in elements.items():
                    if e["type"] != etype:
                        continue
                    if "x" in e:
                        feat = QgsFeature()
                        feat.setGeometry(QgsGeometry.fromPointXY(QgsPointXY(e["x"], e["y"])))
                        feat.setAttributes([name, etype, e.get("downstream", "")])
                        writer.addFeature(feat)

                del writer

                if os.path.exists(shp_path):
                    layer = QgsVectorLayer(shp_path, etype, "ogr")
                    QgsProject.instance().addMapLayer(layer, False)
                    group_hms.addLayer(layer)
                    self.apply_icon_symbol(layer, etype)
                    self.enable_labels(layer)

            # ------------------- REACH -------------------
            reach_path = os.path.join(outdir, "reach.shp")
            fields = QgsFields()
            fields.append(QgsField("from", QVariant.String))
            fields.append(QgsField("to", QVariant.String))

            writer = QgsVectorFileWriter(reach_path, "UTF-8", fields,
                                         QgsWkbTypes.LineString, crs, "ESRI Shapefile")

            for name, e in elements.items():
                dn = e.get("downstream")
                if not dn or dn not in elements:
                    continue

                if elements[dn]["type"] == "reach":
                    feat = QgsFeature()
                    feat.setGeometry(QgsGeometry.fromPolylineXY([
                        QgsPointXY(e["x"], e["y"]),
                        QgsPointXY(elements[dn]["x"], elements[dn]["y"])
                    ]))
                    feat.setAttributes([name, dn])
                    writer.addFeature(feat)

            del writer

            if os.path.exists(reach_path):
                layer = QgsVectorLayer(reach_path, "reach", "ogr")
                QgsProject.instance().addMapLayer(layer, False)
                group_hms.addLayer(layer)
                self.apply_reach_style(layer)
                self.enable_labels_custom(layer, "to")

            # ------------------- FLOW CONNECTIONS -------------------
            flow_path = os.path.join(outdir, "flow_connections.shp")
            fields = QgsFields()
            fields.append(QgsField("from", QVariant.String))
            fields.append(QgsField("to", QVariant.String))

            writer = QgsVectorFileWriter(flow_path, "UTF-8", fields,
                                         QgsWkbTypes.LineString, crs, "ESRI Shapefile")

            for name, e in elements.items():
                dn = e.get("downstream")
                if not dn or dn not in elements:
                    continue

                if elements[dn]["type"] != "reach":
                    feat = QgsFeature()
                    feat.setGeometry(QgsGeometry.fromPolylineXY([
                        QgsPointXY(e["x"], e["y"]),
                        QgsPointXY(elements[dn]["x"], elements[dn]["y"])
                    ]))
                    feat.setAttributes([name, dn])
                    writer.addFeature(feat)

            del writer

            if os.path.exists(flow_path):
                layer = QgsVectorLayer(flow_path, "flow_connections", "ogr")
                QgsProject.instance().addMapLayer(layer, False)
                group_hms.addLayer(layer)
                self.apply_flow_style(layer)

        QMessageBox.information(
            self.iface.mainWindow(),
            "HMS Extractor",
            f"Extracción completada.\n\nShapefiles guardados en:\n{outdir}"
        )

        self.dlg.accept()

    # =====================================================================
    #                        PARSERS DEL ARCHIVO BASIN
    # =====================================================================
    def parse_basin(self, basin_path):
        elements = {}

        if not os.path.exists(basin_path):
            return elements

        with open(basin_path, "r", encoding="utf-8", errors="ignore") as f:
            lines = f.readlines()

        current = None

        for raw in lines:
            line = raw.strip()
            if not line:
                continue
            low = line.lower()

            # Inicio de elemento
            if low.startswith(("subbasin:", "junction:", "reservoir:",
                               "reach:", "source:", "sink:")):

                tag = line.split(":", 1)[0].lower()
                name = line.split(":", 1)[1].strip()

                current = {"type": tag, "name": name}
                elements[name] = current
                continue

            if low.startswith("end:"):
                current = None
                continue

            if not current:
                continue

            # Campos comunes
            if low.startswith("canvas x"):
                current["x"] = float(line.split(":", 1)[1].strip())
            elif low.startswith("canvas y"):
                current["y"] = float(line.split(":", 1)[1].strip())
            elif low.startswith("downstream"):
                current["downstream"] = line.split(":", 1)[1].strip()

            # Solo para Subbasins
            if current["type"] == "subbasin":
                if low.startswith("area"):
                    current["area"] = float(line.split(":", 1)[1].strip())
                elif low.startswith("lossrate"):
                    current["loss_meth"] = line.split(":", 1)[1].strip()
                elif low.startswith("curve number"):
                    current["cn"] = float(line.split(":", 1)[1].strip())
                elif low.startswith("transform"):
                    current["transform"] = line.split(":", 1)[1].strip()
                elif low.startswith("time of concentration"):
                    current["tc"] = float(line.split(":", 1)[1].strip())
                elif low.startswith("storage coefficient"):
                    current["stor_coeff"] = float(line.split(":", 1)[1].strip())

        return elements

    # =====================================================================
    #                           PARSE BACKGROUND MAPS
    # =====================================================================
    def parse_background_maps(self, basin_path, project_folder):
        bg = []

        if not os.path.exists(basin_path):
            return bg

        with open(basin_path, "r", encoding="utf-8", errors="ignore") as f:
            lines = f.readlines()

        current = None

        def commit():
            nonlocal current
            if not current:
                return
            if current.get("src_path", "").lower().endswith(".shp"):
                bg.append(current)
            current = None

        for raw in lines:
            s = raw.strip()
            low = s.lower()

            if low.startswith("end:"):
                commit()
                continue

            if low.startswith("map shown"):
                shown = s.split(":", 1)[1].strip().lower() in ("yes", "true", "1")
                if current is None:
                    current = {"relpath": None, "src_path": None, "shown": shown}
                else:
                    current["shown"] = shown
                continue

            if low.startswith("map:"):
                if current:
                    commit()
                current = {"relpath": None, "src_path": None, "shown": None}
                continue

            if low.startswith("map file name"):
                rel = s.split(":", 1)[1].strip()
                src_path = rel if os.path.isabs(rel) else \
                    os.path.normpath(os.path.join(project_folder, rel))
                current["relpath"] = rel
                current["src_path"] = src_path
                continue

        commit()
        return bg

    # =====================================================================
    #                              ESTILOS
    # =====================================================================
    def apply_icon_symbol(self, vlayer, etype):
        icon_map = {
            "subbasin": "subbasin.png",
            "junction": "junction.png",
            "reservoir": "reservoir.png",
            "source": "source.png",
            "sink": "sink.png"
        }
        icon_file = icon_map.get(etype)
        if not icon_file:
            return

        icon_path = os.path.join(self.plugin_dir, "icons", icon_file)
        if not os.path.exists(icon_path):
            return

        symbol = QgsMarkerSymbol.createSimple({})
        rlayer = QgsRasterMarkerSymbolLayer(icon_path)
        rlayer.setSize(5.0)
        symbol.changeSymbolLayer(0, rlayer)

        vlayer.renderer().setSymbol(symbol)
        vlayer.triggerRepaint()

    def apply_reach_style(self, vlayer):
        sym = vlayer.renderer().symbol()
        if sym:
            lyr = QgsSimpleLineSymbolLayer()
            lyr.setColor(QColor(0, 102, 255))
            lyr.setWidth(0.5)
            sym.changeSymbolLayer(0, lyr)
            vlayer.triggerRepaint()

    def apply_flow_style(self, vlayer):
        sym = vlayer.renderer().symbol()
        if sym:
            lyr = QgsSimpleLineSymbolLayer()
            lyr.setColor(QColor(0, 0, 0))
            lyr.setWidth(0.3)
            sym.changeSymbolLayer(0, lyr)
            vlayer.triggerRepaint()

    # =====================================================================
    #                          ETIQUETAS
    # =====================================================================
    def enable_labels(self, vlayer):
        s = QgsPalLayerSettings()
        f = QgsTextFormat()
        f.setSize(8)
        f.setColor(QColor(0, 0, 0))
        s.setFormat(f)
        s.fieldName = "name"
        s.enabled = True
        vlayer.setLabeling(QgsVectorLayerSimpleLabeling(s))
        vlayer.setLabelsEnabled(True)
        vlayer.triggerRepaint()

    def enable_labels_custom(self, vlayer, field):
        if field not in vlayer.fields().names():
            return
        s = QgsPalLayerSettings()
        f = QgsTextFormat()
        f.setSize(8)
        f.setColor(QColor(0, 0, 0))
        s.setFormat(f)
        s.fieldName = field
        s.enabled = True
        vlayer.setLabeling(QgsVectorLayerSimpleLabeling(s))
        vlayer.setLabelsEnabled(True)
        vlayer.triggerRepaint()
