# -*- coding: utf-8 -*-

"""
/***************************************************************************
 HistoriqueParcelle
                                 A QGIS plugin
 Historique des parcelles (cadastre français)
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2024-09-04
        copyright            : (C) 2024 by HistoriqueParcelle.fr
        email                : contact@historiqueparcelle.fr
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

__author__ = 'HistoriqueParcelle.fr'
__date__ = '2024-09-04'
__copyright__ = '(C) 2024 by HistoriqueParcelle.fr'

# This will get replaced with a git SHA1 when you do a git archive

__revision__ = '$Format:%H$'

import json
from shapely.geometry import MultiPolygon, shape
from qgis.PyQt.QtCore import QCoreApplication, QSettings, QVariant, QUrl, QEventLoop
from qgis.core import (QgsProcessing,
                       QgsFeature,
                       QgsGeometry,
                       QgsJsonUtils,
                       QgsCoordinateReferenceSystem,
                       QgsFields, 
                       QgsWkbTypes,
                       QgsNetworkAccessManager,
                       QgsProcessingException,
                       QgsFeatureSink,
                       QgsProcessingAlgorithm,
                       QgsProcessingParameterField,
                       QgsProcessingParameterString,
                       QgsProcessingParameterVectorLayer,
                       QgsProcessingParameterFeatureSource,
                       QgsProcessingParameterFeatureSink)
from qgis.PyQt.QtNetwork import QNetworkRequest


class HistoriqueParcelleAlgorithm(QgsProcessingAlgorithm):
    """
    Géolocaliser les parcelles disparues via historiqueparcelle.fr 

    """

    # Constants used to refer to parameters and outputs. They will be
    # used when calling the algorithm from another algorithm, or when
    # calling from the QGIS console.

    OUTPUT = 'OUTPUT'
    ERROR_OUTPUT = 'OUTPUT_ERROR'
    INPUT = 'INPUT'
    API_KEY = 'API_KEY'
    API_URL = 'https://europe-west3-historiqueparcelle.cloudfunctions.net/API'
    APICREDIT_URL = 'https://europe-west3-historiqueparcelle.cloudfunctions.net/API_credits'
    ID_FIELD = 'ID_FIELD'

    def initAlgorithm(self, config):
        """
        Here we define the inputs and output of the algorithm, along
        with some other properties.
        """
        # Chargement de la clé API depuis les paramètres (si elle existe)
        settings = QSettings()
        saved_api_key = settings.value('historiqueparcelle/api_key', '')

        # We add the input database table source.
        self.addParameter(
            QgsProcessingParameterVectorLayer(
                self.INPUT,
                self.tr('Table contenant les identifiants des parcelles'),
                [QgsProcessing.TypeVector], 
                defaultValue=None,
                optional=False
            )
        )

        # We add a field name parameter for the ID field
        self.addParameter(
            QgsProcessingParameterField(
                self.ID_FIELD,
                self.tr('Champ contenant l\'identifiant des parcelles (ex:97419000AB0015)'),
                parentLayerParameterName=self.INPUT,
                type=QgsProcessingParameterField.String 
            )
        )

        # We add a string parameter for the API key
        self.addParameter(
            QgsProcessingParameterString(
                self.API_KEY,
                self.tr('Clé API historiqueparcelle.fr'),
                defaultValue=saved_api_key  # Utilise la clé stockée
            )
        )

        # We add a feature sink in which to store our processed features (this
        # usually takes the form of a newly created vector layer when the
        # algorithm is run in QGIS).
        self.addParameter(
            QgsProcessingParameterFeatureSink(
                self.OUTPUT,
                self.tr('Parcelles géoréférencées')
            )
        )

        self.addParameter(
            QgsProcessingParameterFeatureSink(
                self.ERROR_OUTPUT,
                self.tr('Parcelles non retrouvées')
            )
        )

    def queryAPI(self, url):
        # Envoyer la requête GET avec ou sans proxy selon la configuration
        qurl = QUrl(url)
        request = QNetworkRequest(qurl)
        network_manager = QgsNetworkAccessManager.instance()
        reply = network_manager.get(request)
        event_loop = QEventLoop()
        reply.finished.connect(event_loop.quit)
        event_loop.exec_()
        if reply.error() == reply.NoError:
            response_data = reply.readAll().data()  # Convertir QByteArray en bytes
            try:
                json_data = json.loads(response_data)  # Convertir en JSON
                return json_data
            except json.JSONDecodeError:
                return {"error":True, "msg":f"Erreur décodage JSON"}
        else:
            return {"error":True, "msg":f"Erreur Requête: {reply.errorString()}"}
   

    def processAlgorithm(self, parameters, context, feedback):
        """
        Here is where the processing itself takes place.
        """

        credits = 0
        # Récupérer la clé API fournie par l'utilisateur
        api_key = self.parameterAsString(parameters, self.API_KEY, context)
        
        # Sauvegarder la clé API dans les paramètres pour une utilisation future
        settings = QSettings()
        settings.setValue('historiqueparcelle/api_key', api_key)

        feedback.pushInfo(f"Clé API utilisée : {api_key}")

        #Verifier nombre de credits
        url = f'{self.APICREDIT_URL}?key={api_key}'
        credits_request = self.queryAPI(url)
        if 'error' in credits_request:
            if credits_request['error']==True:
                feedback.reportError(f"Erreur lors de la requête GET: {url}")
                return {}
        if 'credits' in credits_request:
            credits=credits_request['credits']
        feedback.pushInfo(f"Nombre de crédits disponibles : {credits}")
        
        source = self.parameterAsSource(parameters, self.INPUT, context)

        num_rows = source.featureCount()
        feedback.pushInfo(f"Nombre de crédits nécéssaires (au maximum) : {num_rows}")
        if num_rows > credits:
            feedback.reportError(f"Nombre de crédits insuffisant.")
            return {}
        
        # Récupérer le champ sélectionné par l'utilisateur
        field_name = self.parameterAsString(parameters, self.ID_FIELD, context)

        crs = QgsCoordinateReferenceSystem('EPSG:4326')
        #store output data
        (sink, dest_id) = self.parameterAsSink(parameters, self.OUTPUT,
        context, source.fields(), QgsWkbTypes.MultiPolygon, crs)
        #store error data
        (sink2, dest_id2) = self.parameterAsSink(parameters, self.ERROR_OUTPUT,
        context, source.fields(), QgsWkbTypes.NoGeometry, crs)

        # Itération sur chaque ligne (feature) de la source
        total = 100.0 / source.featureCount() if source.featureCount() else 0
        for current,feature in enumerate(source.getFeatures()):
            # Stop the algorithm if cancel button has been clicked
            if feedback.isCanceled():
                break
            # Accéder aux attributs de chaque ligne
            attributes = feature.attributes()
            # Récupérer la valeur du champ spécifié
            parcelle_id = feature.attribute(field_name)
            if len(parcelle_id)!=14:
                feedback.pushWarning(f"ID parcelle non valide : {parcelle_id}")
                sink2.addFeature(feature)
                continue
            feedback.pushInfo(f"Traitement parcelle : {parcelle_id}")
            commune = parcelle_id[0:5]
            prefixe = parcelle_id[5:8]
            section = parcelle_id[8:10]
            numero = parcelle_id[10:14]
            url = f"https://apicarto.ign.fr/api/cadastre/parcelle?code_insee={commune}&section={section}&numero={numero}&com_abs={prefixe}"
            res = self.queryAPI(url)
            if 'error' in res:
                if res['error']==True:
                    feedback.reportError(f"Erreur lors de la requête GET: {url}")
                    sink2.addFeature(feature)
                    continue
            if 'features' in res:
                if len(res['features'])>0:
                    geom = shape(res['features'][0]['geometry'])
                    geom = QgsGeometry.fromWkt(geom.wkt)
                    feat = QgsFeature()
                    feat.setGeometry(geom)
                    feat.setAttributes(attributes)
                    sink.addFeature(feat)
                    continue
            #HISTORIQUE PARCELLE REQUEST
            url=f'{self.API_URL}?key={api_key}&type=enfants&id={parcelle_id}'
            res2 = self.queryAPI(url)
            if 'error' in res2:
                if res2['error']==True:
                    feedback.reportError(f"Erreur lors de la requête GET: {url}")
                    sink2.addFeature(feature)
                    continue
            if 'valid' in res2:
                if res2['valid']==False:
                    feedback.reportError(f"Historique non retrouvé - {parcelle_id}")
                    sink2.addFeature(feature)   
                    continue                     
                else:
                    ids = res2['ids']
                    geoms=[]
                    for id in ids:
                        commune = id[0:5]
                        prefixe = id[5:8]
                        section = id[8:10]
                        numero = id[10:14]
                        url = f"https://apicarto.ign.fr/api/cadastre/parcelle?code_insee={commune}&section={section}&numero={numero}&com_abs={prefixe}"
                        res3 = self.queryAPI(url)
                        if 'error' in res3:
                            if res3['error']==True:
                                feedback.reportError(f"Erreur lors de la requête GET: {url}")
                                sink2.addFeature(feature)
                                continue
                        if 'features' in res3:
                            if len(res3['features'])>0:
                                geom = shape(res3['features'][0]['geometry'])
                                geom = QgsGeometry.fromWkt(geom.wkt)
                                geoms.append(geom)
                    if len(geoms)==0:
                        feedback.pushWarning(f"Aucune géométrie trouvée: [parent:{parcelle_id}, fille: {id}]")
                        sink2.addFeature(feature)
                        continue
                    combined_geom = QgsGeometry.unaryUnion(geoms)
                    feature.setGeometry(combined_geom)
                    sink.addFeature(feature)
            #update progress bar
            feedback.setProgress(int(current * total))

        # Return the results of the algorithm. In this case our only result is
        # the feature sink which contains the processed features, but some
        # algorithms may return multiple feature sinks, calculated numeric
        # statistics, etc. These should all be included in the returned
        # dictionary, with keys matching the feature corresponding parameter
        # or output names.
        return {self.OUTPUT: dest_id, self.ERROR_OUTPUT: dest_id2}

    def name(self):
        """
        Returns the algorithm name, used for identifying the algorithm. This
        string should be fixed for the algorithm, and must not be localised.
        The name should be unique within each provider. Names should contain
        lowercase alphanumeric characters only and no spaces or other
        formatting characters.
        """
        return 'Retrouver et géolocaliser les parcelles disparues'

    def displayName(self):
        """
        Returns the translated algorithm name, which should be used for any
        user-visible display of the algorithm name.
        """
        return self.tr(self.name())

    def group(self):
        """
        Returns the name of the group this algorithm belongs to. This string
        should be localised.
        """
        return self.tr(self.groupId())

    def groupId(self):
        """
        Returns the unique ID of the group this algorithm belongs to. This
        string should be fixed for the algorithm, and must not be localised.
        The group id should be unique within each provider. Group id should
        contain lowercase alphanumeric characters only and no spaces or other
        formatting characters.
        """
        return None #'historiqueparcelle'

    def tr(self, string):
        return QCoreApplication.translate('Processing', string)

    def createInstance(self):
        return HistoriqueParcelleAlgorithm()
