# -*- coding: utf-8 -*-

"""
/***************************************************************************
 Topographic position
                                 A QGIS plugin
 Compute hedge sposition in the watershed.
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2022-01-22
        copyright            : (C) 2022 by Gabriel Marquès
        email                : gabriel.marques@toulouse-inp.fr
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

__author__ = 'Gabriel Marquès'
__date__ = '2022-01-22'
__copyright__ = '(C) 2022 by Gabriel Marquès'

# This will get replaced with a git SHA1 when you do a git archive

__revision__ = '$Format:%H$'

from qgis.PyQt.QtCore import QCoreApplication
from qgis.core import (QgsProcessing,
                       QgsFeatureSink,
                       QgsProcessingAlgorithm,
                       QgsProcessingParameterFeatureSource,
                       QgsProcessingParameterFeatureSink,
                       QgsProcessingParameterNumber,
                       QgsProcessingParameterRasterLayer)
from qgis.PyQt.QtGui import QIcon
from hedge_tools import resources # Only need in hedge_tools.py normaly but just to keep track of import

from hedge_tools.tools.classes import class_hedge as h

class TopoPositionAlgorithm(QgsProcessingAlgorithm):
    """
    Extract the topographic position of the hedges (9 class)
    Default values are determined for a 5m resolution dem

    Parameters
    ---
    INPUT_POLY (QgisObject : QgsVectorLayer) : Polygon layer path. Contains hedges
    INPUT_ARC (QgisObject : QgsVectorLayer) : Linestring layer path. Contains arc (polyline) hedges
    INPUT_NODE (QgisObject : QgsVectorLayer) : Node layer path. Contains nodes to delimit the hedges
    INPUT_MNT (QgisObject : QgsRasterLayer) : MNT path.
    INPUT_SEARCH (int) : Search radius for determining topographic shape
    INPUT_SKIP (int) : Skip radius for which the topograhic shape will be ignored. Allow to ignore small topographic changes
    INPUT_KERNEL (int) : Must be odd. Kernel radius for majority filtering of topographic shape

    Return
    ---
    OUTPUT_POLY (QgisObject : QgsVectorLayer) : Polygon : Polygon layer with a topographic position attribute.
    """

    # Constants used to refer to parameters and outputs. They will be
    # used when calling the algorithm from another algorithm, or when
    # calling from the QGIS console.
    INPUT_POLY = "INPUT_POLY"
    INPUT_ARC = "INPUT_ARC"
    INPUT_NODE = "INPUT_NODE"
    INPUT_MNT = "INPUT_MNT"
    INPUT_SEARCH = "INPUT_SEARCH"
    INPUT_SKIP = "INPUT_SKIP"
    INPUT_KERNEL = "INPUT_KERNEL"
    OUTPUT_POLY = "OUTPUT_POLY"

    def initAlgorithm(self, config):
        """
        Here we define the inputs and output of the algorithm, along
        with some other properties.
        """

        # We add the input vector polygons features source.
        self.addParameter(
            QgsProcessingParameterFeatureSource(
                self.INPUT_POLY,
                self.tr("Polygons vector layer"),
                [QgsProcessing.TypeVectorPolygon]
            )
        )

        # We add the input vector lines features source.
        self.addParameter(
            QgsProcessingParameterFeatureSource(
                self.INPUT_ARC,
                self.tr("Arcs vector layer"),
                [QgsProcessing.TypeVectorLine]
            )
        )

        # We add the input vector nodes features source.
        self.addParameter(
            QgsProcessingParameterFeatureSource(
                self.INPUT_NODE,
                self.tr("Nodes vector layer"),
                [QgsProcessing.TypeVectorPoint]
            )
        )

        # We add the input vector mnt
        self.addParameter(
            QgsProcessingParameterRasterLayer(
                self.INPUT_MNT,
                self.tr("Elevation raster (DEM)"),
                [QgsProcessing.TypeRaster]
            )
        )

        self.addParameter(
            QgsProcessingParameterNumber(
                name=self.INPUT_SEARCH,
                description=self.tr("Search distance for terrain shapes (meters)"),
                type=QgsProcessingParameterNumber.Integer,
                defaultValue=100,
                optional=False,
                minValue=0,
                maxValue=200
            )
        )

        self.addParameter(
            QgsProcessingParameterNumber(
                name=self.INPUT_SKIP,
                description=self.tr("Skip distance for terrain shapes (meters)"),
                type=QgsProcessingParameterNumber.Integer,
                defaultValue=50,
                optional=False,
                minValue=0,
                maxValue=200
            )
        )

        self.addParameter(
            QgsProcessingParameterNumber(
                name=self.INPUT_KERNEL,
                description=self.tr("Kernel radius for median filter (pixels) - odd values"),
                type=QgsProcessingParameterNumber.Integer,
                defaultValue=7,
                optional=False,
                minValue=0,
                maxValue=11
            )
        )

    def processAlgorithm(self, parameters, context, feedback):
        """
        Here is where the processing itself takes place.
        """

        poly_layer = self.parameterAsVectorLayer(parameters, self.INPUT_POLY, context)
        arc_layer = self.parameterAsVectorLayer(parameters, self.INPUT_ARC, context)
        node_layer = self.parameterAsVectorLayer(parameters, self.INPUT_NODE, context)
        mnt = self.parameterAsRasterLayer(parameters, self.INPUT_MNT, context)

        feedback.pushInfo("Starting processing")

        # Check for cancellation
        if feedback.isCanceled():
            return {}

        # Init hedge class and compute length
        hedges = h.Hedges(poly_layer, arc_layer, node_layer)
        hedges.topographic_position(context, feedback, mnt)

        return {"OUTPUT_POLY": parameters[self.INPUT_POLY]}
    
    def icon(self):
        """
        Should return a QIcon which is used for your provider inside
        the Processing toolbox.
        """
        return QIcon(":/plugins/hedge_tools/images/hedge_tools.png")
    
    def shortHelpString(self):
        """
        Returns a localised short help string for the algorithm.
        """
        return self.tr("From an elevation raster (DEM), this algorithm gives hedges' topographic position. \
                        The results are stored in the 'topo_pos' field of the polygon layer. \n\
                        The default parameter values work well for a 5 meters resolution DEM, \
                        and the DEM extent should be slightly bigger than the vector layers' extent, \
                        as the search and skip parameters need contextual information. \
                        If it is not possible, hedges in the border of the study area might be given \
                        the wrong position or no data as a result. \n\
                        The search parameter represents the radius in meters for searching terrain shapes. \
                        A higher value will give more global terrain shapes. \n\
                        The skip parameter is the radius in meters where the terrain shapes will be ignored. \
                        A higher value will ignore bigger changes in topography. \n\
                        The kernel radius parameter is used to further remove local changes. \
                        A higher resolution DEM should use a bigger kernel (5 or 7).")

    def name(self):
        """
        Returns the algorithm name, used for identifying the algorithm. This
        string should be fixed for the algorithm, and must not be localised.
        The name should be unique within each provider. Names should contain
        lowercase alphanumeric characters only and no spaces or other
        formatting characters.
        """
        return "topographicposition"

    def displayName(self):
        """
        Returns the translated algorithm name, which should be used for any
        user-visible display of the algorithm name.
        """
        return self.tr("Topographic position")

    def group(self):
        """
        Returns the name of the group this algorithm belongs to. This string
        should be localised.
        """
        return self.tr("5 - Context level: geographic")

    def groupId(self):
        """
        Returns the unique ID of the group this algorithm belongs to. This
        string should be fixed for the algorithm, and must not be localised.
        The group id should be unique within each provider. Group id should
        contain lowercase alphanumeric characters only and no spaces or other
        formatting characters.
        """
        return 'geographiccontext'

    def tr(self, string):
        return QCoreApplication.translate('Processing', string)

    def createInstance(self):
        return TopoPositionAlgorithm()
