# -*- coding: utf-8 -*-

"""
/***************************************************************************
 Interfaee Type
                                 A QGIS plugin
 Retrieve intersecting interface (crop field, river, road, railway, building)
 and store their id in fields inside the polygon layer
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2022-01-22
        copyright            : (C) 2022 by Gabriel Marquès
        email                : gabriel.marques@toulouse-inp.fr
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

__author__ = 'Gabriel Marquès'
__date__ = '2022-01-22'
__copyright__ = '(C) 2022 by Gabriel Marquès'

# This will get replaced with a git SHA1 when you do a git archive

__revision__ = '$Format:%H$'

from qgis.PyQt.QtCore import QCoreApplication
from qgis.core import (QgsProcessing,
                       QgsFeatureSink,
                       QgsProcessingAlgorithm,
                       QgsProcessingParameterFeatureSource,
                       QgsProcessingParameterFeatureSink,
                       QgsProcessingParameterNumber,
                       QgsProcessingParameterField)
from qgis.PyQt.QtGui import QIcon
from hedge_tools import resources # Only need in hedge_tools.py normaly but just to keep track of import

from hedge_tools.tools.classes import class_hedge as h

class InterfaceTypeAlgorithm(QgsProcessingAlgorithm):
    """
    Retrieve intersecting interface (crop field, river, road, railway, building)
    and store their id in fields inside the polygon layer

    Parameters
    ---
    INPUT_POLY (QgisObject : QgsVectorLayer) : Polygon layer path. Contains hedges
    INPUT_ARC (QgisObject : QgsVectorLayer) : Linestring layer path. Contains arc (polyline) hedges
    INPUT_NODE (QgisObject : QgsVectorLayer) : Node layer path. Contains nodes to delimit the hedges
    INPUT_CROP (QgisObject : QgsVectorLayer) : Crop layer path.
    INPUT_CROP_ID (QgsField) : Crop id field to retrieve.
    INPUT_RIVER (QgisObject : QgsVectorLayer) : River layer path.
    INPUT_RIVER_ID (QgsField) : River id field to retrieve.
    INPUT_ROAD (QgisObject : QgsVectorLayer) : Road layer path.
    INPUT_ROAD_ID (QgsField) : Road id field to retrieve.
    INPUT_RAILWAY (QgisObject : QgsVectorLayer) : Railway layer path.
    INPUT_RAILWAY_ID (QgsField) : Railway id field to retrieve.
    INPUT_BUILDING (QgisObject : QgsVectorLayer) : Building layer path.
    INPUT_BUILDING_ID (QgsField) : Building id field to retrieve.

    Return
    ---
    OUTPUT_POLY (QgisObject : QgsVectorLayer) : Polygon : Polygon layer with new fields
                                                corresponding to intersecting features
    """

    # Constants used to refer to parameters and outputs. They will be
    # used when calling the algorithm from another algorithm, or when
    # calling from the QGIS console.
    INPUT_POLY = "INPUT_POLY"
    INPUT_ARC = "INPUT_ARC"
    INPUT_NODE = "INPUT_NODE"
    INPUT_CROP = "INPUT_CROP"
    INPUT_CROP_ID = "INPUT_CROP_ID"
    INPUT_RIVER = "INPUT_RIVER"
    INPUT_RIVER_ID = "INPUT_RIVER_ID"
    INPUT_ROAD = "INPUT_ROAD"
    INPUT_ROAD_ID = "INPUT_ROAD_ID"
    INPUT_RAILWAY = "INPUT_RAILWAY"
    INPUT_RAILWAY_ID = "INPUT_RAILWAY_ID"
    INPUT_BUILDING = "INPUT_BUILDING"
    INPUT_BUILDING_ID = "INPUT_BUILDING_ID"
    OUTPUT_POLY = "OUTPUT_POLY"

    def initAlgorithm(self, config):
        """
        Here we define the inputs and output of the algorithm, along
        with some other properties.
        """

        # We add the input vector polygons features source.
        self.addParameter(
            QgsProcessingParameterFeatureSource(
                self.INPUT_POLY,
                self.tr("Polygons vector layer"),
                [QgsProcessing.TypeVectorPolygon]
            )
        )

        # We add the input vector lines features source.
        self.addParameter(
            QgsProcessingParameterFeatureSource(
                self.INPUT_ARC,
                self.tr("Arcs vector layer"),
                [QgsProcessing.TypeVectorLine]
            )
        )

        # We add the input vector nodes features source.
        self.addParameter(
            QgsProcessingParameterFeatureSource(
                self.INPUT_NODE,
                self.tr("Nodes vector layer"),
                [QgsProcessing.TypeVectorPoint]
            )
        )

        # Crop layer and crop id
        self.addParameter(
            QgsProcessingParameterFeatureSource(
                self.INPUT_CROP,
                self.tr("Crop vector layer (optionnal)"),
                [QgsProcessing.TypeVectorPolygon],
                optional=True
            )
        )

        self.addParameter(
            QgsProcessingParameterField(
                self.INPUT_CROP_ID,
                self.tr("Crop id field (optionnal)"),
                type=QgsProcessingParameterField.String,
                parentLayerParameterName="INPUT_CROP",
                optional=True
            )
        )

        # River layer and river id
        self.addParameter(
            QgsProcessingParameterFeatureSource(
                self.INPUT_RIVER,
                self.tr("River vector layer (optionnal)"),
                [QgsProcessing.TypeVectorLine, QgsProcessing.TypeVectorPolygon],
                optional=True

            )
        )

        self.addParameter(
            QgsProcessingParameterField(
                self.INPUT_RIVER_ID,
                self.tr("River id field (optionnal)"),
                type=QgsProcessingParameterField.String,
                parentLayerParameterName="INPUT_RIVER",
                optional=True
            )
        )

        # Road layer and road id
        self.addParameter(
            QgsProcessingParameterFeatureSource(
                self.INPUT_ROAD,
                self.tr("Road vector layer (optionnal)"),
                [QgsProcessing.TypeVectorLine, QgsProcessing.TypeVectorPolygon],
                optional=True

            )
        )

        self.addParameter(
            QgsProcessingParameterField(
                self.INPUT_ROAD_ID,
                self.tr("Road id field (optionnal)"),
                type=QgsProcessingParameterField.String,
                parentLayerParameterName="INPUT_ROAD",
                optional=True
            )
        )

        # Railway layer and railway id
        self.addParameter(
            QgsProcessingParameterFeatureSource(
                self.INPUT_RAILWAY,
                self.tr("Railway vector layer (optionnal)"),
                [QgsProcessing.TypeVectorLine, QgsProcessing.TypeVectorPolygon],
                optional=True

            )
        )

        self.addParameter(
            QgsProcessingParameterField(
                self.INPUT_RAILWAY_ID,
                self.tr("Railway id field (optionnal)"),
                type=QgsProcessingParameterField.String,
                parentLayerParameterName="INPUT_RAILWAY",
                optional=True
            )
        )

        # Building layer and building id
        self.addParameter(
            QgsProcessingParameterFeatureSource(
                self.INPUT_BUILDING,
                self.tr("Building vector layer (optionnal)"),
                [QgsProcessing.TypeVectorPolygon],
                optional=True
            )
        )

        self.addParameter(
            QgsProcessingParameterField(
                self.INPUT_BUILDING_ID,
                self.tr("Building id field (optionnal)"),
                type=QgsProcessingParameterField.String,
                parentLayerParameterName="INPUT_BUILDING",
                optional=True
            )
        )

    def processAlgorithm(self, parameters, context, feedback):
        """
        Here is where the processing itself takes place.
        """

        poly_layer = self.parameterAsVectorLayer(parameters, self.INPUT_POLY, context)
        arc_layer = self.parameterAsVectorLayer(parameters, self.INPUT_ARC, context)
        node_layer = self.parameterAsVectorLayer(parameters, self.INPUT_NODE, context)

        feedback.pushInfo("Starting processing")

        # Check for cancellation
        if feedback.isCanceled():
            return {}

        parameters_list = [parameters[self.INPUT_CROP], parameters[self.INPUT_RIVER],
                           parameters[self.INPUT_ROAD], parameters[self.INPUT_RAILWAY],
                           parameters[self.INPUT_BUILDING]]

        if any(param is not None for param in parameters_list):
            layer_dict = {}

            if parameters[self.INPUT_CROP] is not None:
                layer_dict["crop"] = \
                    [self.parameterAsVectorLayer(parameters, self.INPUT_CROP, context),
                     parameters[self.INPUT_CROP_ID]]

            if parameters[self.INPUT_RIVER] is not None:
                layer_dict["river"] = \
                    [self.parameterAsVectorLayer(parameters, self.INPUT_RIVER, context),
                     parameters[self.INPUT_RIVER_ID]]

            if parameters[self.INPUT_ROAD] is not None:
                layer_dict["road"] = \
                    [self.parameterAsVectorLayer(parameters, self.INPUT_ROAD, context),
                     parameters[self.INPUT_ROAD_ID]]

            if parameters[self.INPUT_RAILWAY] is not None:
                layer_dict["railway"] = \
                    [self.parameterAsVectorLayer(parameters, self.INPUT_RAILWAY, context),
                     parameters[self.INPUT_RAILWAY_ID]]

            if parameters[self.INPUT_BUILDING] is not None:
                layer_dict["building"] = \
                    [self.parameterAsVectorLayer(parameters, self.INPUT_BUILDING, context),
                     parameters[self.INPUT_BUILDING_ID]]

            # Init hedge class and retrieve interface type
            hedges = h.Hedges(poly_layer, arc_layer, node_layer)
            hedges.interface_type(context, feedback, layer_dict)

            return {"OUTPUT_POLY": parameters[self.INPUT_POLY]}

        else:
            feedback.reportError("Interface type algorithm needs \
                                  at least one interface layer and \
                                  his corresponding id field to perform.")

    def icon(self):
        """
        Should return a QIcon which is used for your provider inside
        the Processing toolbox.
        """
        return QIcon(":/plugins/hedge_tools/images/hedge_tools.png")
    
    def shortHelpString(self):
        """
        Returns a localised short help string for the algorithm.
        """
        return self.tr("Retrieve hedges adjacent interfaces \
                        and store their unique id in a set of fields \
                        in the polygon attribute table. \
                        Interface types includes : crops, rivers, roads, railways and buildings \
                        At least one interface layer and their corresponding id must be informed.")

    def name(self):
        """
        Returns the algorithm name, used for identifying the algorithm. This
        string should be fixed for the algorithm, and must not be localised.
        The name should be unique within each provider. Names should contain
        lowercase alphanumeric characters only and no spaces or other
        formatting characters.
        """
        return "interfacetype"

    def displayName(self):
        """
        Returns the translated algorithm name, which should be used for any
        user-visible display of the algorithm name.
        """
        return self.tr("Interface type")

    def group(self):
        """
        Returns the name of the group this algorithm belongs to. This string
        should be localised.
        """
        return self.tr("Geographic context")

    def groupId(self):
        """
        Returns the unique ID of the group this algorithm belongs to. This
        string should be fixed for the algorithm, and must not be localised.
        The group id should be unique within each provider. Group id should
        contain lowercase alphanumeric characters only and no spaces or other
        formatting characters.
        """
        return 'geographiccontext'

    def tr(self, string):
        return QCoreApplication.translate('Processing', string)

    def createInstance(self):
        return InterfaceTypeAlgorithm()
