# -*- coding: utf-8 -*-

"""
/***************************************************************************
 Floor proportion
                                 A QGIS plugin
 Compute the proportion of strata in a hedge from a top view perspective.
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2022-01-22
        copyright            : (C) 2022 by Gabriel Marquès
        email                : gabriel.marques@toulouse-inp.fr
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

__author__ = 'Gabriel Marquès'
__date__ = '2022-01-22'
__copyright__ = '(C) 2022 by Gabriel Marquès'

# This will get replaced with a git SHA1 when you do a git archive

__revision__ = '$Format:%H$'

from qgis.PyQt.QtCore import (QCoreApplication,
                              QVariant)
from qgis.core import (QgsProject,
                       QgsProcessing,
                       QgsProcessingUtils,
                       QgsFeatureSink,
                       QgsProcessingAlgorithm,
                       QgsProcessingParameterFeatureSource,
                       QgsProcessingParameterFeatureSink,
                       QgsProcessingParameterRasterLayer,
                       QgsProcessingParameterBoolean,
                       QgsProcessingParameterString,
                       QgsVectorFileWriter,
                       QgsVectorLayer)
from qgis.PyQt.QtGui import QIcon
from hedge_tools import resources # Only need in hedge_tools.py normaly but just to keep track of import

from hedge_tools.tools.vector import attribute_table as at
from hedge_tools.tools.vector import utils
from hedge_tools.tools.classes import class_hedge as h

import os

class StrataProportionFromDHMAlgorithm(QgsProcessingAlgorithm):
    """
    Compute the proportion of strata in a hedge from a top view perspective.

    Parameters
    ---
    INPUT_POLY (QgisObject : QgsVectorLayer) : Polygon layer path. Contains hedges
    INPUT_ARC (QgisObject : QgsVectorLayer) : Linestring layer path. Contains arc (polyline) hedges
    INPUT_NODE (QgisObject : QgsVectorLayer) : Node layer path. Contains nodes to delimit the hedges
    INPUT_MNH (QgisObject : QgsRasterLayer) : Height raster.
    INPUT_BINS (int) : Bins representing the strata boundary

    Return
    ---
    OUTPUT_POLY (QgisObject : QgsVectorLayer) : Polygon : Polygons layer with
    a field for each bin representing the percentage of overlap.
    """

    # Constants used to refer to parameters and outputs. They will be
    # used when calling the algorithm from another algorithm, or when
    # calling from the QGIS console.
    INPUT_POLY = "INPUT_POLY"
    INPUT_ARC = "INPUT_ARC"
    INPUT_NODE = "INPUT_NODE"
    INPUT_MNH = "INPUT_MNH"
    INPUT_BINS = "INPUT_BINS"
    INPUT_SPATIAL = "INPUT_SPATIAL"
    OUTPUT_POLY = "OUTPUT_POLY"
    OUTPUT_SPATIAL = "OUTPUT_SPATIAL"

    def initAlgorithm(self, config):
        """
        Here we define the inputs and output of the algorithm, along
        with some other properties.
        """

        # We add the input vector polygons features source.
        self.addParameter(
            QgsProcessingParameterFeatureSource(
                self.INPUT_POLY,
                self.tr("Polygons vector layer"),
                [QgsProcessing.TypeVectorPolygon]
            )
        )

        # We add the input vector lines features source.
        self.addParameter(
            QgsProcessingParameterFeatureSource(
                self.INPUT_ARC,
                self.tr("Arcs vector layer"),
                [QgsProcessing.TypeVectorLine]
            )
        )

        # We add the input vector nodes features source.
        self.addParameter(
            QgsProcessingParameterFeatureSource(
                self.INPUT_NODE,
                self.tr("Nodes vector layer"),
                [QgsProcessing.TypeVectorPoint]
            )
        )

        # We add the input vector mnc/mnh
        self.addParameter(
            QgsProcessingParameterRasterLayer(
                self.INPUT_MNH,
                self.tr("Height raster (DHM/MNH)"),
                [QgsProcessing.TypeRaster]
            )
        )

        self.addParameter(
            QgsProcessingParameterString(
                name=self.INPUT_BINS,
                description=self.tr("Boundary for each height category (separated by comma)"),
                defaultValue="0.3, 2, 7"
            )
        )

        # Handle distance parameter
        self.addParameter(
            QgsProcessingParameterBoolean(
                name=self.INPUT_SPATIAL,
                description=self.tr("Output category as feature"),
                defaultValue=False
            )
        )

        # We add a feature sink in which to store our processed features (this
        # usually takes the form of a newly created vector layer when the
        # algorithm is run in QGIS).

        # Output polygons sink
        self.addParameter(
            QgsProcessingParameterFeatureSink(
                self.OUTPUT_SPATIAL,
                self.tr("Output category as feature (optionnal)"),
            )
        )

    def flags(self):
        """
        Algorithm manipulating project (toggle layer) or using external library are not thread safe
        See : https://api.qgis.org/api/classQgsProcessingAlgorithm.html#a6a8c21fab75e03f50f45e41a9d67dbc3a229dea6cedf8c59132196dee09d4f2f6
        """
        return super().flags() | QgsProcessingAlgorithm.FlagNoThreading
    
    def processAlgorithm(self, parameters, context, feedback):
        """
        Here is where the processing itself takes place.
        """

        poly_layer = self.parameterAsVectorLayer(parameters, self.INPUT_POLY, context)
        arc_layer = self.parameterAsVectorLayer(parameters, self.INPUT_ARC, context)
        node_layer = self.parameterAsVectorLayer(parameters, self.INPUT_NODE, context)
        mnh = self.parameterAsRasterLayer(parameters, self.INPUT_MNH, context)
        bins = self.parameterAsString(parameters, self.INPUT_BINS, context)

        temp_layer = QgsProcessingUtils.generateTempFilename("temp_hedges.gpkg")

        feedback.pushInfo("Starting processing")

        # Check for cancellation
        if feedback.isCanceled():
            return {}

        # Max value of raster
        max_value = mnh.dataProvider().bandStatistics(1).maximumValue
        # Add to bins as upper limit
        if len(bins) != 0:
            bins = [float(b) for b in bins.split(",")]
        else:
            bins = [0.3, 2, 7]
        bins.append(int(max_value))

        # Fields creation
        fields_list = []
        for b in range(0, len(bins)):
            field = ("%(1)s/%(2)s"
                     %{"1": bins[b - 1] if bins[b] != bins[0] else "0",
                       "2": bins[b] if bins[b] != 9999 else "+"},
                     QVariant.Double)
            fields_list.append(field)
        fields_list.extend((("nb_strata", QVariant.Int),
                            ("dom_strata", QVariant.String)))
        idx_list = at.create_fields(poly_layer, fields_list)

        # Check if vector layer is temporary
        vector_driver = poly_layer.dataProvider().storageType()
        if vector_driver == "Memory storage":
            transform_context = QgsProject.instance().transformContext()
            options = QgsVectorFileWriter.SaveVectorOptions()
            options.fileEncoding = "utf-8"
            options.driverName = "GPKG"
            error = QgsVectorFileWriter.writeAsVectorFormatV3(poly_layer,
                                                              temp_layer,
                                                              transform_context,
                                                              options)

            if error[0] == QgsVectorFileWriter.NoError:
                feedback.pushInfo("Memory layer succesfully converted to OGR layer")
                poly_temp = QgsVectorLayer(temp_layer, "temp", "ogr")
            else:
                feedback.pushInfo("Error, couldn't convert memory layer to an OGR layer. \
                                   Please save the polygons layer before using this tool.")

        # Init hedge class and compute length
        #     fields = [f for f in poly_temp.fields().names()]
        #     print(fields)
            pr = poly_layer.dataProvider()
            hedges = h.Hedges(poly_temp, arc_layer, node_layer)
        else:
            hedges = h.Hedges(poly_layer, arc_layer, node_layer)

        if parameters[self.INPUT_SPATIAL]:
            output_layer = hedges.strata_overlap_from_DHM(context, feedback, mnh, bins, idx_list, True)
            # Fill the sink
            (sink_poly, poly_id) = self.parameterAsSink(parameters,
                                                        self.OUTPUT_SPATIAL,
                                                        context,
                                                        output_layer.fields(),
                                                        output_layer.wkbType(),
                                                        output_layer.sourceCrs())

            features = output_layer.getFeatures()
            for feature in features:
                sink_poly.addFeature(feature, QgsFeatureSink.FastInsert)

            # Copy floor fields in self.poly_layer
            if vector_driver == "Memory storage":
                # Populate
                attr_map = {feat.id(): dict(zip(idx_list,
                            feat.attributes()[min(idx_list): max(idx_list) + 1]))
                            for feat in poly_temp.getFeatures()}
                pr.changeAttributeValues(attr_map)

            # Delete temp layer
            poly_temp = None
            QgsVectorFileWriter.deleteShapeFile(temp_layer)

            return {"OUTPUT_POLY": parameters[self.INPUT_POLY],
                    "OUTPUT_SPATIAL": poly_id}

        else:
            hedges.strata_overlap_from_DHM(context, feedback, mnh, bins, idx_list, False)

            # Copy floor fields in self.poly_layer
            if vector_driver == "Memory storage":
                # Populate
                attr_map = {feat.id(): dict(zip(idx_list,
                            feat.attributes()[min(idx_list): max(idx_list) + 1]))
                            for feat in poly_temp.getFeatures()}
                pr.changeAttributeValues(attr_map)

            # Delete temp_layer
            poly_temp = None
            QgsVectorFileWriter.deleteShapeFile(temp_layer)

            return {"OUTPUT_POLY": parameters[self.INPUT_POLY]}
    
    def postProcessAlgorithm(self, context, feedback):
        """
        Tasks done when processAlgorithm is finished
        """
        utils.delete_processing_workspace()

        return {}
    
    def icon(self):
        """
        Should return a QIcon which is used for your provider inside
        the Processing toolbox.
        """
        return QIcon(":/plugins/hedge_tools/images/hedge_tools.png")
    
    def shortHelpString(self):
        """
        Returns a localised short help string for the algorithm.
        """
        return self.tr("Please note that the algorithm is currently freezing QGIS interface until completion. \n\
                        Return each height category area as a percentage in the polygon layer's fields. \
                        It is also possible to ask for a spatial representation of each category. \n\
                        Categories must be separated by commas. The lower boundary and upper boundary \
                        are assumed to be the minimum and maximum values of the height raster, respectively. \n\
                        By default, AFAC strata are used: [min value - 0.3[, [0.3 - 2[, [2 - 7[, [7 - max value].")

    def name(self):
        """
        Returns the algorithm name, used for identifying the algorithm. This
        string should be fixed for the algorithm, and must not be localised.
        The name should be unique within each provider. Names should contain
        lowercase alphanumeric characters only and no spaces or other
        formatting characters.
        """
        return "computestrataproportionDHM"

    def displayName(self):
        """
        Returns the translated algorithm name, which should be used for any
        user-visible display of the algorithm name.
        """
        return self.tr("Strata proportion from DHM")

    def group(self):
        """
        Returns the name of the group this algorithm belongs to. This string
        should be localised.
        """
        return self.tr("4 - Hedges level: physiognomy")

    def groupId(self):
        """
        Returns the unique ID of the group this algorithm belongs to. This
        string should be fixed for the algorithm, and must not be localised.
        The group id should be unique within each provider. Group id should
        contain lowercase alphanumeric characters only and no spaces or other
        formatting characters.
        """
        return 'hedgesphysiognomy'

    def tr(self, string):
        return QCoreApplication.translate('Processing', string)

    def createInstance(self):
        return StrataProportionFromDHMAlgorithm()
