# -*- coding: utf-8 -*-

"""
/***************************************************************************
 Height metrics
                                 A QGIS plugin
 Compute the hedge height median, mean, minimum, maximum and variance
 by overlapping with a mnh and store values in fields of attribute table
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2022-01-22
        copyright            : (C) 2022 by Gabriel Marquès
        email                : gabriel.marques@toulouse-inp.fr
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

__author__ = 'Gabriel Marquès'
__date__ = '2022-01-22'
__copyright__ = '(C) 2022 by Gabriel Marquès'

# This will get replaced with a git SHA1 when you do a git archive

__revision__ = '$Format:%H$'

from qgis.PyQt.QtCore import QCoreApplication
from qgis.core import (QgsProcessing,
                       QgsFeatureSink,
                       QgsProcessingAlgorithm,
                       QgsProcessingParameterFeatureSource,
                       QgsProcessingParameterFeatureSink,
                       QgsProcessingParameterNumber,
                       QgsProcessingParameterRasterLayer)
from qgis.PyQt.QtGui import QIcon
from hedge_tools import resources # Only need in hedge_tools.py normaly but just to keep track of import

from hedge_tools.tools.classes import class_hedge as h

class HeightMetricsFromDHMAlgorithm(QgsProcessingAlgorithm):
    """
    Compute height statistics for features based on a mnh/dhm.

    Parameters
    ---
    INPUT_POLY (QgisObject : QgsVectorLayer) : Polygon layer path. Contains hedges
    INPUT_ARC (QgisObject : QgsVectorLayer) : Linestring layer path. Contains arc (polyline) hedges
    INPUT_NODE (QgisObject : QgsVectorLayer) : Node layer path. Contains nodes to delimit the hedges
    INPUT_MNH (QgisObject : QgsRasterLayer) : Height raster.

    Return
    ---
    OUTPUT_POLY (QgisObject : QgsVectorLayer) : Polygon : Polygons layer with 5 height attributes.
    """

    # Constants used to refer to parameters and outputs. They will be
    # used when calling the algorithm from another algorithm, or when
    # calling from the QGIS console.
    INPUT_POLY = "INPUT_POLY"
    INPUT_ARC = "INPUT_ARC"
    INPUT_NODE = "INPUT_NODE"
    INPUT_MNH = "INPUT_MNH"
    OUTPUT_POLY = "OUTPUT_POLY"

    def initAlgorithm(self, config):
        """
        Here we define the inputs and output of the algorithm, along
        with some other properties.
        """

        # We add the input vector polygons features source.
        self.addParameter(
            QgsProcessingParameterFeatureSource(
                self.INPUT_POLY,
                self.tr("Polygons vector layer"),
                [QgsProcessing.TypeVectorPolygon]
            )
        )

        # We add the input vector lines features source.
        self.addParameter(
            QgsProcessingParameterFeatureSource(
                self.INPUT_ARC,
                self.tr("Arcs vector layer"),
                [QgsProcessing.TypeVectorLine]
            )
        )

        # We add the input vector nodes features source.
        self.addParameter(
            QgsProcessingParameterFeatureSource(
                self.INPUT_NODE,
                self.tr("Nodes vector layer"),
                [QgsProcessing.TypeVectorPoint]
            )
        )

        # We add the input vector mnc/mnh
        self.addParameter(
            QgsProcessingParameterRasterLayer(
                self.INPUT_MNH,
                self.tr("Height raster (DHM)"),
                [QgsProcessing.TypeRaster]
            )
        )

    def processAlgorithm(self, parameters, context, feedback):
        """
        Here is where the processing itself takes place.
        """

        poly_layer = self.parameterAsVectorLayer(parameters, self.INPUT_POLY, context)
        arc_layer = self.parameterAsVectorLayer(parameters, self.INPUT_ARC, context)
        node_layer = self.parameterAsVectorLayer(parameters, self.INPUT_NODE, context)
        mnh = self.parameterAsRasterLayer(parameters, self.INPUT_MNH, context)

        feedback.pushInfo("Starting processing")

        # Check for cancellation
        if feedback.isCanceled():
            return {}

        # Init hedge class and compute length
        hedges = h.Hedges(poly_layer, arc_layer, node_layer)
        hedges.compute_height_metrics_from_DHM(context, feedback, mnh)

        return {"OUTPUT_POLY": parameters[self.INPUT_POLY]}

    def icon(self):
        """
        Should return a QIcon which is used for your provider inside
        the Processing toolbox.
        """
        return QIcon(":/plugins/hedge_tools/images/hedge_tools.png")
    
    def shortHelpString(self):
        """
        Returns a localised short help string for the algorithm.
        """
        return self.tr("Summarize the height of the hedges using a height raster (DHM/MNH). \
                       Five fields are created and stored in the polygon layer: \
                       h_mean, h_med, h_min, h_max, and h_std. \
                       They represent, respectively, the mean, median, minimum, maximum, \
                       and standard deviation values of the hedges' height.")

    def name(self):
        """
        Returns the algorithm name, used for identifying the algorithm. This
        string should be fixed for the algorithm, and must not be localised.
        The name should be unique within each provider. Names should contain
        lowercase alphanumeric characters only and no spaces or other
        formatting characters.
        """
        return "computeheightmetricsDHM"

    def displayName(self):
        """
        Returns the translated algorithm name, which should be used for any
        user-visible display of the algorithm name.
        """
        return self.tr("Height metrics from DHM")

    def group(self):
        """
        Returns the name of the group this algorithm belongs to. This string
        should be localised.
        """
        return self.tr("4 - Hedges level: physiognomy")

    def groupId(self):
        """
        Returns the unique ID of the group this algorithm belongs to. This
        string should be fixed for the algorithm, and must not be localised.
        The group id should be unique within each provider. Group id should
        contain lowercase alphanumeric characters only and no spaces or other
        formatting characters.
        """
        return 'hedgesphysiognomy'

    def tr(self, string):
        return QCoreApplication.translate('Processing', string)

    def createInstance(self):
        return HeightMetricsFromDHMAlgorithm()
