#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
/***************************************************************************
 GenerateDHM
                                 A QGIS plugin
 From a DEM and a DSM generate a DHM by substracting the DSM to the DEM.
 Resolution of either the DEM or the DSM can be chosen.
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2024-01-05
        copyright            : (C) 2022 by Gabriel Marques / Dynafor
        email                : gabriel.marques@toulouse-inp.fr
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

__author__ = "Gabriel Marques / Dynafor"
__date__ = "2024-01-05"
__copyright__ = "(C) 2024 by Gabriel Marques / Dynafor"

# This will get replaced with a git SHA1 when you do a git archive

__revision__ = "$Format:%H$"

from qgis.PyQt.QtCore import QCoreApplication
from qgis.core import (QgsProcessing,
                       QgsProcessingAlgorithm,
                       QgsProcessingException,
                       QgsProcessingParameterRasterLayer,
                       QgsProcessingParameterEnum,
                       QgsProcessingParameterBoolean,
                       QgsProcessingParameterNumber,
                       QgsProcessingParameterExtent,
                       QgsProcessingParameterRasterDestination,
                       QgsRasterLayer,
                       QgsFeedback)

from hedge_tools.tools.classes import GenerateNDVI as gNDVI

from qgis.PyQt.QtGui import QIcon
from hedge_tools import resources

class GenerateNDVIProcessingAlgorithm(QgsProcessingAlgorithm):
    """
    From a raster with infrared and red band generate a NDVI raster.
    En extent can be specified to cut the sources.

    Parameters
    ---

    Return
    ---
    
    """
    # Constants used to refer to parameters and outputs. They will be
    # used when calling the algorithm from another algorithm, or when
    # calling from the QGIS console.

    RASTER = "RASTER"
    EXTENT = "EXTENT"
    RED_BAND = "RED_BAND"
    IR_BAND = "IR_BAND"
    RESOLUTION = "RESOLUTION"
    RESAMPLING = "RESAMPLING"
    MASK_NDVI = "MASK_NDVI"
    THRESHOLD_MASK = "THRESHOLD_MASK"
    SIEVE = "SIEVE"
    THRESHOLD_SIEVE = "THRESHOLD_SIEVE"
    CONNECTEDNESS = "CONNECTEDNESS"
    OVERWRITE = "OVERWRITE"
    OUTPUT = "OUTPUT"
    OUTPUT_MASK = "OUTPUT_MASK"
    OUTPUT_SIEVE = "OUTPUT_SIEVE"

    def initAlgorithm(self, config=None):
        """
        Here we define the inputs and output of the algorithm, along
        with some other properties.
        """

        self.addParameter(
            QgsProcessingParameterRasterLayer(
                self.RASTER,
                self.tr("Input raster")
            )
        )

        self.addParameter(
            QgsProcessingParameterExtent(
                self.EXTENT,
                self.tr("Extent of the study area"),
                optional=True
            )
        )

        self.addParameter(
            QgsProcessingParameterNumber(
                name=self.RED_BAND,
                description=self.tr("Red band"),
                type=QgsProcessingParameterNumber.Integer,
                defaultValue=2,
                minValue=1,
                maxValue=100
            )
        )

        self.addParameter(
            QgsProcessingParameterNumber(
                name=self.IR_BAND,
                description=self.tr("Infrared band"),
                type=QgsProcessingParameterNumber.Integer,
                defaultValue=1,
                minValue=1,
                maxValue=100
            )
        )
            
        self.addParameter(
            QgsProcessingParameterNumber(
                name=self.RESOLUTION,
                description=self.tr("Output resolution (meters)"),
                type=QgsProcessingParameterNumber.Double,
                defaultValue=None,
                minValue=0.0,
                optional=True
            )
        )

        resampling_opt_param = QgsProcessingParameterEnum(
            name=self.RESAMPLING,
            description=self.tr("Resampling method"),
            options=["Nearest neighbour ", "Bilinear", 
                     "Cubic", "Cubic spline ", 
                     "Lanczos windowed sinc", "Average", "Mode", 
                     "Maximum", "Minimum", "Median",
                     "First quartile", "Third quartile"],
            optional=True,
            allowMultiple=False)
        resampling_opt_param.setHelp(self.tr('This option determines \
                                             how to resample the output DHM'))
        self.addParameter(resampling_opt_param)

        self.addParameter(
            QgsProcessingParameterBoolean(
                self.MASK_NDVI,
                self.tr("Compute NDVI mask (required for tree cover tool)"),
                defaultValue=True
            )
        )

        self.addParameter(
            QgsProcessingParameterNumber(
                name=self.THRESHOLD_MASK,
                description=self.tr("Threshold NDVI value (NDVI mask)"),
                type=QgsProcessingParameterNumber.Double,
                defaultValue=0.2,
                optional=True,
                minValue=-1,
                maxValue=1
            )
        )
        
        self.addParameter(
            QgsProcessingParameterBoolean(
                self.SIEVE,
                self.tr("Compute sieved NDVI mask"),
                defaultValue=True
            )
        )

        self.addParameter(
            QgsProcessingParameterNumber(
                name=self.THRESHOLD_SIEVE,
                description=self.tr("Threshold sieve value (pixel number)"),
                type=QgsProcessingParameterNumber.Integer,
                defaultValue=100,
                optional=True,
                minValue=0
            )
        )
        
        self.addParameter(
            QgsProcessingParameterEnum(
                name=self.CONNECTEDNESS,
                description=self.tr("Connectedness parameter for sieved output"),
                options=["4","8"],
                defaultValue=1,
                optional=True,
                allowMultiple=False
            )
        )

        self.addParameter(
            QgsProcessingParameterBoolean(
                self.OVERWRITE,
                self.tr("Overwrite"),
                defaultValue=True
            )
        )

        self.addParameter(
            QgsProcessingParameterRasterDestination(
                self.OUTPUT,
                self.tr('NDVI')))
        
        self.addParameter(
            QgsProcessingParameterRasterDestination(
                self.OUTPUT_MASK,
                self.tr('Mask')))

        self.addParameter(
            QgsProcessingParameterRasterDestination(
                self.OUTPUT_SIEVE,
                self.tr('Sieve')))
    
    def __init__(self):
        super().__init__()
        
        self.steps = 2
        self.step_per_alg = 100//self.steps
        self.value = 0

    def progress_bar(self, feedback):
        """
        Incrementation of progress bar and cancellation
        """
        # Set progress
        feedback.setProgress(self.value)
        self.value += self.step_per_alg
        # Check for cancellation
        if feedback.isCanceled():
            return {}
        
    def processAlgorithm(self, parameters, context, feedback):
        """
        Here is where the processing itself takes place.
        """

        raster = self.parameterAsRasterLayer(parameters, self.RASTER, context)
        extent = self.parameterAsExtent(parameters, self.EXTENT, context)
        red_band =  self.parameterAsInt(parameters, self.RED_BAND, context)
        ir_band =  self.parameterAsInt(parameters, self.IR_BAND, context)
        resolution = self.parameterAsDouble(parameters, self.RESOLUTION, 
                                            context)
        resampling = self.parameterAsEnum(parameters, self.RESAMPLING, context)
        compute_mask = self.parameterAsBoolean(parameters, self.MASK_NDVI, context)
        threshold_mask = self.parameterAsDouble(parameters, self.THRESHOLD_MASK, 
                                                context)
        compute_sieve = self.parameterAsBoolean(parameters, self.SIEVE, context)
        threshold_sieve = self.parameterAsDouble(parameters, self.THRESHOLD_SIEVE, 
                                                 context)
        connectedness = self.parameterAsInt(parameters, self.CONNECTEDNESS, 
                                            context)
        overwrite = self.parameterAsBoolean(parameters, self.OVERWRITE, context)
        output = self.parameterAsOutputLayer(parameters, self.OUTPUT, context)
        if compute_mask:
            output_mask = self.parameterAsOutputLayer(parameters, 
                                                      self.OUTPUT_MASK, 
                                                      context)
        else:
            output_mask = None
        if compute_sieve:
            output_sieve = self.parameterAsOutputLayer(parameters, 
                                                       self.OUTPUT_SIEVE, 
                                                       context)
        else:
            output_sieve = None
        outputs = {}
        
        NDVI_builder = gNDVI.GenerateNDVI((raster, red_band), 
                                          (raster, ir_band), 
                                          feedback, context, 
                                          self.displayName())
        NDVI_builder.process_algorithm(extent, None, resolution, resampling, 
                                       compute_mask, threshold_mask, 
                                       compute_sieve, threshold_sieve,
                                       connectedness, overwrite, 
                                       output, output_mask, output_sieve)

        # Return path to avoid a qgis crash
        if NDVI_builder.out_resample is not None:
            outputs[self.OUTPUT] = NDVI_builder.out_resample.dataProvider(
            ).dataSourceUri()
        else:
            outputs[self.OUTPUT] = NDVI_builder.out_index.dataProvider(
            ).dataSourceUri()

        if compute_mask:
            outputs[self.OUTPUT_MASK] = NDVI_builder.out_mask.dataProvider(
            ).dataSourceUri()
        if compute_sieve:
            outputs[self.OUTPUT_SIEVE] = NDVI_builder.out_sieve.dataProvider(
            ).dataSourceUri()

        return outputs
            
    def icon(self):
        """
        Should return a QIcon which is used for your provider inside
        the Processing toolbox.
        """
        return QIcon(":/plugins/hedge_tools/images/hedge_tools.png")

    def shortHelpString(self):
        """
        Returns a localised short helper string for the algorithm. This string
        should provide a basic description about what the algorithm does and 
        the parameters and outputs associated with it..
        """
        return self.tr("Generate NDVI raster from a raster with a \
                       red band and an infrared band. The band numbers use the \
                       GDAL system, where the first band is 1. \n\
                       Optionally, users can specify an extent, \
                       in which case only the common extent between the raster \
                       and the given extent will be used.\n\
                       The algorithm allows you to choose a target resolution. \n\
                       You can also select the resampling method to use \
                       when adjusting the resolution. Some methods yield better \
                       results for downsampling while others are better for \
                       upsampling. Be aware that some methods \
                       (like nearest neighbor) might create gaps between tiles. \n\
                       Users can threshold the NDVI to create a binary mask. \
                       This option is mandatory if the tree cover tools \
                       will be used.\n\
                       It is also possible and advised to use the sieve option \
                       to remove small objects (connected pixels) in the masks. \
                       The sieve value is in number of pixels. \n\
                       The connectedness parameter specifies if 8 neighboring \
                       pixels (diagonals included) should be tested for \
                       connection or only 4.")
    
    def name(self):
        """
        Returns the algorithm name, used for identifying the algorithm. This
        string should be fixed for the algorithm, and must not be localised.
        The name should be unique within each provider. Names should contain
        lowercase alphanumeric characters only and no spaces or other
        formatting characters.
        """
        return "generatendvi"

    def displayName(self):
        """
        Returns the translated algorithm name, which should be used for any
        user-visible display of the algorithm name.
        """
        return self.tr("2 - Generate NDVI")

    def group(self):
        """
        Returns the name of the group this algorithm belongs to. This string
        should be localised.
        """
        return self.tr("0 - Extraction [optional]")

    def groupId(self):
        """
        Returns the unique ID of the group this algorithm belongs to. This
        string should be fixed for the algorithm, and must not be localised.
        The group id should be unique within each provider. Group id should
        contain lowercase alphanumeric characters only and no spaces or other
        formatting characters.
        """
        return "extraction"
    
    def tr(self, string):
        """
        Returns a translatable string with the self.tr() function.
        """
        return QCoreApplication.translate("Processing", string)
    
    def checkParameterValues(self, parameters, context):
        
        if parameters[self.SIEVE] and not parameters[self.MASK_NDVI]:
            return (False, "NDVI masks must be toggle if you want to use the sieve option.")
        
        return (True, '')

    def createInstance(self):
        return GenerateNDVIProcessingAlgorithm()
                    
