#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
/***************************************************************************
 GenerateDHM
                                 A QGIS plugin
 From a DEM and a DSM generate a DHM by substracting the DSM to the DEM.
 Resolution of either the DEM or the DSM can be chosen.
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2024-01-05
        copyright            : (C) 2022 by Gabriel Marques / Dynafor
        email                : gabriel.marques@toulouse-inp.fr
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

__author__ = "Gabriel Marques / Dynafor"
__date__ = "2024-01-05"
__copyright__ = "(C) 2024 by Gabriel Marques / Dynafor"

# This will get replaced with a git SHA1 when you do a git archive

__revision__ = "$Format:%H$"

from qgis.PyQt.QtCore import QCoreApplication
from qgis.core import (QgsProcessing,
                       QgsProcessingAlgorithm,
                       QgsProcessingParameterNumber,
                       QgsProcessingParameterBoolean,
                       QgsProcessingParameterRasterLayer,
                       QgsProcessingParameterRasterDestination,
                       QgsProcessingParameterEnum,
                       QgsRasterLayer,
                       QgsProcessingException,
                       QgsGeometry)

from processing.algs.gdal.GdalUtils import GdalUtils

from hedge_tools.tools.classes import GenerateDHM as gDHM

from qgis.PyQt.QtGui import QIcon
from hedge_tools import resources

class GenerateDHMProcessingAlgorithm(QgsProcessingAlgorithm):
    DSM = "DSM"
    DEM = "DEM"
    RESOLUTION = "RESOLUTION"
    RESAMPLING = "RESAMPLING"
    EXTENT = "EXTENT"
    MASK_DHM = "MASK_DHM"
    THRESHOLD_MASK = "THRESHOLD_MASK"
    OVERWRITE = "OVERWRITE"
    OUTPUT = "OUTPUT"
    OUTPUT_MASK = "OUTPUT_MASK"

    def initAlgorithm(self, config=None):

        self.addParameter(
            QgsProcessingParameterRasterLayer(
                self.DSM,
                self.tr("DSM"),
            )
        )

        self.addParameter(
            QgsProcessingParameterRasterLayer(
                self.DEM,
                self.tr("DEM"),
            )
        )
            
        self.addParameter(
            QgsProcessingParameterNumber(
                name=self.RESOLUTION,
                description=self.tr("Output resolution (meters)"),
                type=QgsProcessingParameterNumber.Double,
                defaultValue=None,
                minValue=0.0,
                optional=True
            )
        )

        resampling_opt_param = QgsProcessingParameterEnum(
            name=self.RESAMPLING,
            description=self.tr("Resampling method"),
            options=["Nearest neighbour ", "Bilinear", 
                     "Cubic", "Cubic spline ", 
                     "Lanczos windowed sinc", "Average", "Mode", 
                     "Maximum", "Minimum", "Median",
                     "First quartile", "Third quartile"],
            optional=True,
            allowMultiple=False)
        resampling_opt_param.setHelp(self.tr('This option determines how to \
                                             resample the output DHM'))
        self.addParameter(resampling_opt_param)

        if GdalUtils.version() >= 3030000:
            extent_opt_param = QgsProcessingParameterEnum(
                self.EXTENT,
                self.tr('Handling of extent differences'),
                options= ['Ignore', 'Fail', 'Union', 'Intersect'],
                defaultValue=0)
            extent_opt_param.setHelp(self.tr('This option determines how to \
                                             handle rasters with different extents'))
            self.addParameter(extent_opt_param)

        self.addParameter(
             QgsProcessingParameterBoolean(
                self.MASK_DHM,
                self.tr("Compute DHM mask (required for tree cover tool)"),
                defaultValue=True
            )
        )

        self.addParameter(
            QgsProcessingParameterNumber(
                name=self.THRESHOLD_MASK,
                description=self.tr("Threshold height value (DHM mask)"),
                type=QgsProcessingParameterNumber.Double,
                defaultValue=2.5,
                optional=True,
                minValue=-999,
                maxValue=1000
            )
        )
        
        self.addParameter(
            QgsProcessingParameterBoolean(
                self.OVERWRITE,
                self.tr("Overwrite"),
                defaultValue=True
            )
        )

        self.addParameter(
            QgsProcessingParameterRasterDestination(
                self.OUTPUT,
                self.tr('DHM')))
        
        self.addParameter(
            QgsProcessingParameterRasterDestination(
                self.OUTPUT_MASK,
                self.tr('Mask')))
    
    def __init__(self):
        super().__init__()
        
        self.steps = 2
        self.step_per_alg = 100//self.steps
        self.value = 0

    def progress_bar(self, feedback):
        """
        Incrementation of progress bar and cancellation
        """
        # Set progress
        feedback.setProgress(self.value)
        self.value += self.step_per_alg
        # Check for cancellation
        if feedback.isCanceled():
            return {}
        
    def processAlgorithm(self, parameters, context, feedback):
        """
        Here is where the processing itself takes place.
        """
        dsm = self.parameterAsRasterLayer(parameters, self.DSM, context)
        dem = self.parameterAsRasterLayer(parameters, self.DEM, context)
        resolution = self.parameterAsDouble(parameters, self.RESOLUTION, 
                                            context)
        resampling = self.parameterAsEnum(parameters, self.RESAMPLING, context)
        extent = self.parameterAsEnum(parameters, self.EXTENT, context)
        compute_mask = self.parameterAsBoolean(parameters, self.MASK_DHM, 
                                               context)
        threshold = self.parameterAsDouble(parameters, self.THRESHOLD_MASK, 
                                           context)
        overwrite = self.parameterAsBoolean(parameters, self.OVERWRITE, context)
        output = self.parameterAsOutputLayer(parameters, self.OUTPUT, context)
        if compute_mask:
            output_mask = self.parameterAsOutputLayer(parameters, 
                                                      self.OUTPUT_MASK, 
                                                      context)
        else:
            output_mask = None
        outputs = {}

        # In batch mod layers should be loaded in a sorted folder 
        # and using a selection not folder. otherwise they are not matched 
        # even if they have the same name. Could load using an expression like : 
        # 'File location/Max_'|| base_file_name( (at)inputfilename)||'.tif'
        # But that's the user problem
        DHM_builder = gDHM.GenerateDHM((dem,1), (dsm,1), 
                                       feedback, context, 
                                       self.displayName())
        DHM_builder.process_algorithm(extent, resolution, resampling, 
                                      compute_mask, threshold, overwrite, 
                                      output, output_mask)

        # Return path to avoid a qgis crash
        if DHM_builder.out_resample is not None:
            outputs[self.OUTPUT] = DHM_builder.out_resample.dataProvider(
            ).dataSourceUri()
        else:
            outputs[self.OUTPUT] = DHM_builder.out_index.dataProvider(
            ).dataSourceUri()

        if compute_mask:
            outputs[self.OUTPUT_MASK] = DHM_builder.out_mask.dataProvider(
            ).dataSourceUri()

        return outputs
                
    def icon(self):
        """
        Should return a QIcon which is used for your provider inside
        the Processing toolbox.
        """
        return QIcon(":/plugins/hedge_tools/images/hedge_tools.png")

    def shortHelpString(self):
        """
        Returns a localised short helper string for the algorithm. This string
        should provide a basic description about what the algorithm does and 
        the parameters and outputs associated with it.
        """
        return self.tr("Generate a DHM from any intersecting DEM and DSM \
                        that have the same resolution. \
                        The algorithm allows you to choose a target resolution. \n\
                        You can also select the resampling method to use \
                        when adjusting the resolution. \
                        Some methods yield better results \
                        for downsampling while others are better for upsampling. \
                        Beware that some methods (like nearest neighbor) \
                        might create gaps between tiles. \n\
                        You have the option to threshold the generated DHM. \
                        This option is mandatory if the tree cover tools \
                        will be used.")
    
    def name(self):
        """
        Returns the algorithm name, used for identifying the algorithm. This
        string should be fixed for the algorithm, and must not be localised.
        The name should be unique within each provider. Names should contain
        lowercase alphanumeric characters only and no spaces or other
        formatting characters.
        """
        return "generatedhm"

    def displayName(self):
        """
        Returns the translated algorithm name, which should be used for any
        user-visible display of the algorithm name.
        """
        return self.tr("1 - Generate DHM")
    
    def group(self):
        """
        Returns the name of the group this algorithm belongs to. This string
        should be localised.
        """
        return self.tr("0 - Extraction [optional]")

    def groupId(self):
        """
        Returns the unique ID of the group this algorithm belongs to. This
        string should be fixed for the algorithm, and must not be localised.
        The group id should be unique within each provider. Group id should
        contain lowercase alphanumeric characters only and no spaces or other
        formatting characters.
        """
        return "extraction"
    
    def tr(self, string):
        """
        Returns a translatable string with the self.tr() function.
        """
        return QCoreApplication.translate("Processing", string)
    
    def checkParameterValues(self, parameters, context):
                
        if (parameters["RESOLUTION"] is not None 
            and parameters["RESOLUTION"] != QgsRasterLayer(self.DSM).extent()) \
                and parameters["RESAMPLING"] is None:
            return (False, "Please select a resampling algorithm.")
        
                
        if (parameters["RESOLUTION"] is None 
            or parameters["RESOLUTION"] == QgsRasterLayer(self.DSM).extent()) \
                and parameters["RESAMPLING"] is not None:
            return (False, "Please select a target resolution.")
        

        return (True, '')

    def createInstance(self):
        return GenerateDHMProcessingAlgorithm()
                    
