# -*- coding: utf-8 -*-

"""
/***************************************************************************
 SplitHedges
                                 A QGIS plugin
 Split the hedge polygon given by the arcs.
 Splitting angle is the bissector between two arcs. Post-processing of the angle
 to ensure no intersection with other cut line or linear hedge feature can modify it.
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2022-01-22
        copyright            : (C) 2022 by Gabriel Marques / Terranis
        email                : gabriel.marques@toulouse-inp.fr
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

__author__ = 'Gabriel Marques / Terranis'
__date__ = '2022-01-22'
__copyright__ = '(C) 2022 by Gabriel Marques / Terranis'

# This will get replaced with a git SHA1 when you do a git archive

__revision__ = '$Format:%H$'

from qgis.PyQt.QtCore import (QCoreApplication,
                              QVariant)
from qgis.core import (QgsProcessing,
                       QgsProcessingUtils,
                       QgsFeatureSink,
                       QgsProcessingAlgorithm,
                       QgsProcessingParameterFeatureSource,
                       QgsProcessingParameterFeatureSink,
                       QgsProcessingParameterField,
                       QgsFeatureRequest)
from qgis.PyQt.QtGui import QIcon
from hedge_tools import resources # Only need in hedge_tools.py normaly but just to keep track of import 

import processing
import os
# from collections import defaultdict
# from itertools import combinations
from hedge_tools.tools.vector import geometry as g
from hedge_tools.tools.vector import utils
from hedge_tools.tools.vector import attribute_table as at

class CreatePolygonsAlgorithm(QgsProcessingAlgorithm):
    """
    Cut hedges polygon by their arc with a specific angle
    to ensure correct transition from linear representation to surfacic representation.

    Parameters
    ---
    INPUT_POLY (QgisObject : QgsVectorLayer) : Polygon layer path. Contains hedges
    INPUT_ARC (QgisObject : QgsVectorLayer) : Linestring layer path. Contains arc (polyline) hedges
    INPUT_NODE (QgisObject : QgsVectorLayer) : Node layer path. Contains nodes to delimit the hedges
    INPUT_FIELD (str) : Node classification field

    Return
    ---
    OUTPUT_POLY (QgisObject : QgsVectorLayer) : Polygon : Path of the layer containing the clipped hedges polygons
    OUTPUT_ARC (QgisObject : QgsVectorLayer) : Linestring : Path of the layer containing the arcs with the foreign key
    linking to the parent polygon
    """

    # Constants used to refer to parameters and outputs. They will be
    # used when calling the algorithm from another algorithm, or when
    # calling from the QGIS console.
    INPUT_POLY = "INPUT_POLY"
    INPUT_ARC = "INPUT_ARC"
    INPUT_NODE = "INPUT_NODE"
    INPUT_FIELD = "INPUT_FIELD"
    OUTPUT_POLY = "OUTPUT_POLY"
    OUTPUT_ARC = "OUTPUT_ARC"

    def initAlgorithm(self, config):
        """
        Here we define the inputs and output of the algorithm, along
        with some other properties.
        """

        # We add the input vector polygons features source.
        self.addParameter(
            QgsProcessingParameterFeatureSource(
                self.INPUT_POLY,
                self.tr("Polygons vector layer"),
                [QgsProcessing.TypeVectorPolygon]
            )
        )

        # We add the input vector lines features source.
        self.addParameter(
            QgsProcessingParameterFeatureSource(
                self.INPUT_ARC,
                self.tr("Arcs vector layer"),
                [QgsProcessing.TypeVectorLine]
            )
        )

        # We add the input vector nodes features source.
        self.addParameter(
            QgsProcessingParameterFeatureSource(
                self.INPUT_NODE,
                self.tr("Nodes vector layer"),
                [QgsProcessing.TypeVectorPoint]
            )
        )

        # We add the input field nodes type.
        self.addParameter(
            QgsProcessingParameterField(
                self.INPUT_FIELD,
                self.tr("Nodes type field"),
                type=QgsProcessingParameterField.String,
                parentLayerParameterName='INPUT_NODE'
            )
        )

        # We add a feature sink in which to store our processed features (this
        # usually takes the form of a newly created vector layer when the
        # algorithm is run in QGIS).

        # Output polygons sink
        self.addParameter(
            QgsProcessingParameterFeatureSink(
                self.OUTPUT_POLY,
                self.tr("Topological polygons"),
            )
        )

        # Output arcs sink
        self.addParameter(
            QgsProcessingParameterFeatureSink(
                self.OUTPUT_ARC,
                self.tr("Topological arcs"),
            )
        )

    def processAlgorithm(self, parameters, context, feedback):
        """
        Here is where the processing itself takes place.
        """
        # Init input as vector layer
        poly_layer = self.parameterAsVectorLayer(parameters, self.INPUT_POLY, context)
        arc_layer = self.parameterAsVectorLayer(parameters, self.INPUT_ARC, context)
        node_layer = self.parameterAsVectorLayer(parameters, self.INPUT_NODE, context)

        # Init progress bar
        alg_number = 14
        step_per_alg = int(100/alg_number)

        feedback.pushInfo("Starting processing")

        # Check for cancellation
        if feedback.isCanceled():
            return {}
        
        input_field_list = poly_layer.fields().names()
        arc_uri = QgsProcessingUtils.generateTempFilename("arc_layer.gpkg")
        arc_layer = utils.create_layer(arc_layer, copy_feat=True, copy_field=True,
                                       data_provider="ogr", path=arc_uri)
        
        cutlines_uri = QgsProcessingUtils.generateTempFilename("cutlines.gpkg")
        cutlines = utils.create_layer(arc_layer, data_provider="ogr", path=cutlines_uri)

        request = QgsFeatureRequest().setFilterExpression("%s != 'O'" %parameters["INPUT_FIELD"])
        cutline_layer = g.make_cutlines(cutlines, poly_layer, arc_layer, node_layer, request)

        # Set progress
        feedback.setProgress(step_per_alg * 4)
        # Check for cancellation
        if feedback.isCanceled():
            return {}

        # Validate cutLine
        feedback.pushInfo("Cut lines validation")
        cutline_valid = g.validate_cutlines(cutline_layer, poly_layer, arc_layer)
        #del(cutline_layer)

        ###TEMP?####
        expression = "isValid == False"
        request = QgsFeatureRequest().setFilterExpression(expression)
        idx_error = at.create_fields(node_layer, [("Error", QVariant.Bool)])
        attr_map = {}
        del_list = []
        for cutline in cutline_valid.getFeatures(expression):
            exp = f"vid == {cutline['vid']}"
            req = QgsFeatureRequest().setFilterExpression(exp)
            node = next(node_layer.getFeatures(exp))
            attr_map[node.id()]={idx_error:True}
            del_list.append(cutline.id())

        node_layer.dataProvider().changeAttributeValues(attr_map)
        cutline_valid.dataProvider().deleteFeatures(del_list)
        
        ###TEMP?####
        # Set progress
        feedback.setProgress(step_per_alg * 8)
        # Check for cancellation
        if feedback.isCanceled():
            return {}

        feedback.pushInfo("Cutting polygons with cut lines")
        hedge_cut, arc_final = g.use_cutlines(cutline_valid, poly_layer, arc_layer, node_layer)
        del(cutline_valid)

        # Set progress
        feedback.setProgress(step_per_alg * 12)
        # Check for cancellation
        if feedback.isCanceled():
            return {}

        feedback.pushInfo("Creating and formatting outputs")

        # Create sinkx  
        (sink_arcs, arc_id) = self.parameterAsSink(parameters,
                                                   self.OUTPUT_ARC,
                                                   context,
                                                   arc_final.fields(),
                                                   arc_final.wkbType(),
                                                   arc_final.sourceCrs())
        
        for arc in arc_final.getFeatures():
            # Fill the sink
            sink_arcs.addFeature(arc, QgsFeatureSink.FastInsert)

        # Set progress
        feedback.setProgress(step_per_alg * 13)
        # Check for cancellation
        if feedback.isCanceled():
            return {}

        # Delete useless fields
        new_field_list = hedge_cut.fields().names()
        field_del_list = list(set(new_field_list) - set(input_field_list))
        add_del_list = ["id", "ORIG_FID", "len"]

        for fields in add_del_list:
            if fields not in field_del_list:
                field_del_list.append(fields)

        # Delete duplicated fields
        alg_name = "native:deletecolumn"
        params = {"INPUT": hedge_cut,
                  "COLUMN": field_del_list,
                  "OUTPUT": "TEMPORARY_OUTPUT"}
        hedge_final = processing.run(alg_name, params,
                                     is_child_algorithm=False,
                                     context=context,
                                     feedback=feedback)["OUTPUT"]
        del(hedge_cut)

        # Create sink
        (sink_polys, poly_id) = self.parameterAsSink(parameters,
                                                   self.OUTPUT_POLY,
                                                   context,
                                                   hedge_final.fields(),
                                                   hedge_final.wkbType(),
                                                   hedge_final.sourceCrs())

        for poly in hedge_final.getFeatures():
            # Fill the sink
            sink_polys.addFeature(poly, QgsFeatureSink.FastInsert)

        # Set progress
        feedback.setProgress(step_per_alg * 14)
        # Check for cancellation
        if feedback.isCanceled():
            return {}

        # utils.delete_temp_workspace(self.folder_object)

        return {"OUTPUT_ARC": arc_id,
                "OUTPUT_POLY": poly_id}

        # return {"OUTPUT_POLY": poly_id}

    def postProcessAlgorithm(self, context, feedback):
        """
        Tasks done when processAlgorithm is finished
        """
        utils.delete_processing_workspace()

        return {}

    def icon(self):
        """
        Should return a QIcon which is used for your provider inside
        the Processing toolbox.
        """
        return QIcon(":/plugins/hedge_tools/images/hedge_tools.png")
    
    def shortHelpString(self):
        """
        Returns a localised short help string for the algorithm.
        """
        return self.tr("This algorithm transforms an initial polygonal hedge \
                        layer (with aggregated objects) into a new one based \
                        on the topological graph. Aggregated objects in the \
                        input layer are splitted to be consistent with the \
                        graph representation.\n \
                        The polygon ID (pid) is used as the primary key for \
                        topology. This 'pid' field is also transferred to the \
                        arc layer (as foreign key) in order to create the \
                        relations between the polygons and the arcs of the \
                        topological map. This explains why two output files \
                        are required (topological polygons and arcs).")

    def name(self):
        """
        Returns the algorithm name, used for identifying the algorithm. This
        string should be fixed for the algorithm, and must not be localised.
        The name should be unique within each provider. Names should contain
        lowercase alphanumeric characters only and no spaces or other
        formatting characters.
        """
        return 'createpolygons'

    def displayName(self):
        """
        Returns the translated algorithm name, which should be used for any
        user-visible display of the algorithm name.
        """
        return self.tr("3 - Create polygons from median axis")

    def group(self):
        """
        Returns the name of the group this algorithm belongs to. This string
        should be localised.
        """
        return self.tr("1 - Data preparation")

    def groupId(self):
        """
        Returns the unique ID of the group this algorithm belongs to. This
        string should be fixed for the algorithm, and must not be localised.
        The group id should be unique within each provider. Group id should
        contain lowercase alphanumeric characters only and no spaces or other
        formatting characters.
        """
        return 'datapreparation'

    def tr(self, string):
        return QCoreApplication.translate('Processing', string)

    def createInstance(self):
        return CreatePolygonsAlgorithm()
