# -*- coding: utf-8 -*-

"""
/***************************************************************************
 Aggregation
                                 A QGIS plugin
 Merge near by geometries depending of relative orientation
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2022-01-11
        copyright            : (C) 2022 by Gabriel Marquès
        email                : gabriel.marques@toulouse-inp.fr
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

__author__ = 'Gabriel Marquès'
__date__ = '2023-02-27'
__copyright__ = '(C) 2022 by Gabriel Marquès'

# This will get replaced with a git SHA1 when you do a git archive

__revision__ = '$Format:%H$'

from qgis.PyQt.QtCore import (QCoreApplication, 
                              QVariant)
from qgis.core import (QgsProcessing,
                       QgsFeatureSink,
                       QgsProcessingAlgorithm,
                       QgsProcessingParameterFeatureSource,
                       QgsProcessingParameterFeatureSink,
                       QgsProcessingParameterNumber,
                       QgsProcessingParameterMultipleLayers,
                       QgsProcessingContext,
                       QgsFeatureRequest,
                       QgsVectorLayer,
                       QgsFeature,
                       QgsFields,
                       QgsWkbTypes,
                       QgsGeometry,
                       QgsPoint,
                       QgsPointXY,
                       NULL)

from qgis.PyQt.QtGui import QIcon
from hedge_tools import resources # Only need in hedge_tools.py normaly but just to keep track of import 

from hedge_tools.tools.vector import qgis_wrapper as qw
from hedge_tools.tools.vector import geometry as g
from hedge_tools.tools.vector import attribute_table as at
from hedge_tools.tools.vector import utils as vutils
from hedge_tools.tools import utils

from typing import Union
import math
import numpy as np

class AggregationAlgorithm(QgsProcessingAlgorithm):
    """
    Split a list of polygon vector layer (usually hedges and tree) 
    by a list of linestring vector layer such as roads, railways and rivers.
    
    Parameters
    ---
    FOREST (QgsVectorLayer): Layer input from users.
    GROVE (QgsVectorLayer): Layer input from users.
    HEDGES (QgsVectorLayer): Layer input from users.
    TREES (QgsVectorLayer): Layer input from users.
    OVERLAYS (list[QgsVectorLayer]): Network layers from users.
    OFFSET (float): Offset value around the network layers. 
                    Should be the same as the buffer value in split_by_network.
    BUFFER (float): Buffer value around the network layers.
    MAX_DIST (float): Distance between two geometries for aggregation.
    DELTA (float): Upper and lower difference allowed in orientation \
                   between two geometries for aggregation.
    BYPASS_DIST: (float): Minimal distance between two geometries to bypass MAX_DIST and DELTA.
    
    Return
    ---
    AGR_LAYER (QgsVectorLayer): Aggregated vegetation structure.
    """

    # Constants used to refer to parameters and outputs. They will be
    # used when calling the algorithm from another algorithm, or when
    # calling from the QGIS console.
    FORESTS = "FORESTS"
    GROVES = "GROVES"
    HEDGES = "HEDGES"
    TREES = "TREES"
    OVERLAYS = "OVERLAYS"
    OFFSET = "OFFSET"
    BUFFER = "BUFFER"
    MAX_DIST = "MAX_DIST"
    DELTA = "DELTA"
    BYPASS_DIST = "BYPASS_DIST"
 
    AGR_LAYER = "AGR_LAYER"

    def initAlgorithm(self, config):
        """
        Here we define the inputs and output of the algorithm, along
        with some other properties.
        """
        self.addParameter(
            QgsProcessingParameterFeatureSource(
                self.HEDGES,
                self.tr("Hedge layer"),
                [QgsProcessing.TypeVectorPolygon],
                optional=False
            )
        )

        self.addParameter(
            QgsProcessingParameterFeatureSource(
                self.FORESTS,
                self.tr("Forest layer"),
                [QgsProcessing.TypeVectorPolygon],
                optional=True
            )
        )

        self.addParameter(
            QgsProcessingParameterFeatureSource(
                self.GROVES,
                self.tr("Grove layer"),
                [QgsProcessing.TypeVectorPolygon],
                optional=True
            )
        )

        self.addParameter(
            QgsProcessingParameterFeatureSource(
                self.TREES,
                self.tr("Tree layer"),
                [QgsProcessing.TypeVectorPolygon],
                optional=True
            )
        )

        self.addParameter(
            QgsProcessingParameterMultipleLayers(
                self.OVERLAYS,
                self.tr("Network layers"),
                QgsProcessing.TypeVectorLine,
                optional=True
            )
        )
        
        self.addParameter(
            QgsProcessingParameterNumber(
                name=self.OFFSET,
                description=self.tr("Offset value for the network buffer"),
                type=QgsProcessingParameterNumber.Double,
                defaultValue=1,
                optional=False,
                minValue=0
            )
        )

        self.addParameter(
            QgsProcessingParameterNumber(
                name=self.BUFFER,
                description=self.tr("Network buffer value"),
                type=QgsProcessingParameterNumber.Double,
                defaultValue=6,
                optional=False,
                minValue=2
            )
        )

        self.addParameter(
            QgsProcessingParameterNumber(
                name=self.MAX_DIST,
                description=self.tr("Aggregation distance (m)"),
                type=QgsProcessingParameterNumber.Double,
                defaultValue=25,
                optional=True,
                minValue=1
            )
        )

        self.addParameter(
            QgsProcessingParameterNumber(
                name=self.DELTA,
                description=self.tr("Angle variation (°)"),
                type=QgsProcessingParameterNumber.Double,
                defaultValue=25,
                optional=True,
                minValue=0,
                maxValue=180
            )
        )

        self.addParameter(
            QgsProcessingParameterNumber(
                name=self.BYPASS_DIST,
                description=self.tr("Bypass distance (m)"),
                type=QgsProcessingParameterNumber.Double,
                defaultValue=2,
                optional=True,
                minValue=0
            )
        )

        self.addParameter(
            QgsProcessingParameterFeatureSink(
                self.AGR_LAYER,
                self.tr("Aggregated vegetation structure")
            )
        )    

    def __init__(self):
        super().__init__()
        
        self.steps = 6
        self.step_per_alg = None
        self.value = 0
        # Init results storage
        self.feat_list = []
        self.attr_map = {}
        self.checked = []
        
    def processAlgorithm(self, parameters, context, feedback):
        """
        Here is where the processing itself takes place.
        """
        hedge_layer = self.parameterAsVectorLayer(parameters, self.HEDGES, context)  
        if parameters[self.FORESTS] is not None:
            forest_layer = self.parameterAsVectorLayer(parameters, self.FORESTS, context)
        else: 
            forest_layer = None
        if parameters[self.GROVES] is not None:
            grove_layer= self.parameterAsVectorLayer(parameters, self.GROVES, context)
        else :
            grove_layer = None
        if parameters[self.TREES] is not None:
            tree_layer = self.parameterAsVectorLayer(parameters, self.TREES, context)
        else:
            tree_layer = None
        if bool(parameters[self.OVERLAYS]):
            self.steps = 7
            networks = self.parameterAsLayerList(parameters, self.OVERLAYS, context)

        offset = self.parameterAsDouble(parameters, self.OFFSET, context)
        buffer = self.parameterAsDouble(parameters, self.BUFFER, context)
        max_dist = self.parameterAsDouble(parameters, self.MAX_DIST, context)
        delta = self.parameterAsDouble(parameters, self.DELTA, context)
        bypass_dist = self.parameterAsDouble(parameters, self.BYPASS_DIST, context)

        # Starting processing
        self.step_per_alg = 100//self.steps
        feedback.pushInfo("Merging vegetation layers...")
        self.progress_bar(feedback)
        str_layer = self.construct_input_layer(hedge_layer,tree_layer, 
                                               forest_layer, grove_layer)

        # Store input fields
        self.fields = str_layer.fields()

        # Handle network layers
        feedback.pushInfo("Merging network layers...")
        self.progress_bar(feedback)
        if bool(parameters[self.OVERLAYS]):
            network = qw.merge_layers(networks)
            feedback.pushInfo("Creating junction from network...")
            self.progress_bar(feedback)
            hedge_parts = self.create_hedge_parts(network, str_layer, buffer, offset)
            success = self.aggregate_with_network(hedge_parts, str_layer, network, max_dist)

            if not success:
                feedback.pushWarning("An error occured while aggregating using network.")
        else:
            network = None
        # Handle aggregation without network
        feedback.pushInfo("Creating junction without network...")
        self.progress_bar(feedback)
        success = self.aggregate_without_network(str_layer, network, 
                                                 max_dist, delta, bypass_dist)
        if not success:
            feedback.pushWarning("An error occured while aggregating without network.")

        self.progress_bar(feedback)
        agr_layer = self.add_junctions(str_layer, offset)

        feedback.pushInfo("Computing discontinuity (gap) and hedge typology...")
        self.progress_bar(feedback)
        agr_layer = self.estimate_discontinuity_and_typology(agr_layer, str_layer)

        feedback.pushInfo("Formating output...")
        self.progress_bar(feedback)
        agr_layer = self.processing_fields_deletion(agr_layer)
        fid_idx = self.fields.indexFromName("fid")
        _, agr_layer = vutils.update_unique_constraint(feedback, agr_layer, fid_idx)
        sink_id = self.create_sink(agr_layer, "AGR_LAYER", parameters, context)

        return {"AGR_LAYER": sink_id}
    
    def progress_bar(self, feedback):
        """
        Incrementation of progress bar and cancellation
        """
        # Set progress
        feedback.setProgress(self.value)
        self.value += self.step_per_alg
        # Check for cancellation
        if feedback.isCanceled():
            return {}
    
    def construct_input_layer(self, hedge_layer:QgsVectorLayer, 
                              tree_layer:Union[QgsVectorLayer,None]=None, 
                              forest_layer:Union[QgsVectorLayer,None]=None, 
                              grove_layer:Union[QgsVectorLayer,None]=None
                              )-> QgsVectorLayer:
        """
        Will fuse hedge, tree and forest layer togetehr and keep the 
        class of each one in a field.
        If fid is not in the layer one will be created.
        Various fields used for discontinuity and typology 
        of fused hedges will also be created.

        Will store max_fid and processing fields as class variable : 
        max_fid: Int
            Maximum value of fid in str_layer
        idx_discontinuity: Int
            Position of the discontinuity field in the attribute table of str_layer
        idx_typo: Int
            Position of the typology field in the attribute table of str_layer
        idx_over: Int
            Position of the over field in the attribute table of str_layer
            This field is only for storing the estimated area of the new part
            and will be deleted once the overall discontinuity is calculated
        idx_type: Int
            Position of the type field in the attribute table of str_layer
            This field is only for storing the junction type of two features
            (i.e hedge/forest, ...) and will be deleted once the overall 
            hedge typology is estimated.

        Parameters
        ---
        hedge_layer: QgsVectorLayer
        tree_layer : QgsVectorLayer
        forest_layer: QgsVectorLayer

        Return
        ---
        str_layer: QgsVectorLayer
            Vegetation structure layer made of forests, trees and hedges.
        """

        # Creation of class field for merging layers
        idx = at.create_fields(hedge_layer, [("class", QVariant.String)])[0]
        attr_map = {f.id(): {idx:"Hedge"} for f in hedge_layer.getFeatures()}
        hedge_layer.dataProvider().changeAttributeValues(attr_map)
        if tree_layer is not None:
            idx = at.create_fields(tree_layer, [("class", QVariant.String)])[0]
            attr_map = {f.id(): {idx:"Tree"} for f in tree_layer.getFeatures()}
            tree_layer.dataProvider().changeAttributeValues(attr_map)
        if forest_layer is not None:
            idx = at.create_fields(forest_layer, [("class", QVariant.String)])[0]
            attr_map = {f.id(): {idx:"Forest"} for f in forest_layer.getFeatures()}
            forest_layer.dataProvider().changeAttributeValues(attr_map)
        if grove_layer is not None:
            idx = at.create_fields(grove_layer, [("class", QVariant.String)])[0]
            attr_map = {f.id(): {idx:"Grove"} for f in grove_layer.getFeatures()}
            grove_layer.dataProvider().changeAttributeValues(attr_map)

        # Merge
        layers = [hedge_layer, tree_layer, forest_layer, grove_layer]
        layers = [layer for layer in layers if layer is not None]
        str_layer = qw.merge_layers(layers)
        # Just in case
        null_geom = [f.id() for f in str_layer.getFeatures() if f.geometry().area() == 0]
        str_layer.dataProvider().deleteFeatures(null_geom)
        # Correct topology
        str_layer = qw.closing(str_layer, 0.1, join_style=1, segments=5)
        str_layer = qw.multipart_to_singleparts(str_layer)

        # Create fid if not already in the attribute table
        fid_idx = str_layer.fields().indexFromName("fid")
        if fid_idx == -1:
            fid_idx = at.create_fields(str_layer, [("fid", QVariant.Int)])[0]
            attr_map = {f.id():{fid_idx: i+1} for (i,f) 
                        in enumerate(str_layer.getFeatures())}
            str_layer.dataProvider().changeAttributeValues(attr_map)
        self.max_fid = str_layer.dataProvider().maximumValue(fid_idx) + 1

        # Processing fields
        self.idx_discontinuity, self.idx_typo, self.idx_over, self.idx_type = \
            at.create_fields(str_layer, [("gap", QVariant.Double), 
                                        ("typo", QVariant.String), 
                                        ("over_area", QVariant.Double),
                                        ("part_type", QVariant.Int)])

        return str_layer

    def prepare_hedge_parts(self, network:QgsVectorLayer, buffered:QgsVectorLayer,
                            buffer_width:float=6.0, side:int=0)-> QgsVectorLayer:
        """
        Prepare hedge parts from network
        
        Parameters
        ---
        network : QgsVectorLayer : LineString : network layer
        buffered : QgsVectorLayer : Polygon : network layer buffered with same width used for the split by network
        buffer_width : int : Default 2 : width of buffer used to create hedge parts
        
        Return
        ---
        hedge_parts : QgsVectorLayer : potential hedge parts
        """
        side_buffer = qw.single_sided_buffer(network, buffer_width, 
                                            side, segments=8)
        dissolved = qw.dissolve(side_buffer)
        hedge_parts = qw.difference(dissolved, buffered)

        return hedge_parts

    def create_hedge_parts(self, network: QgsVectorLayer, str_layer:QgsVectorLayer, 
                           buffer:float=6, offset:float=1)-> QgsVectorLayer:
        """
        From a linestring layer will create buffer of both side and 
        substract the parts where there is geometries of str_layer
        The remaining part will be used to fetch geometries along the network and perform a 
        graph analysis to check the distance between them along the network.

        Parameters
        ---
        network: QgsVectorLayer
            Linestring layer used to create hedge parts.
            Commonly a network layer of roads, rivers and railway.
        str_layer : QgsVectorLayer
            Vegetation structure layer
        buffer: Float
            Buffer value used to create hedge parts
        offset: Float
            Negative buffer value used to split the structure layer 
            by the network in a previous algorithm.
            Allow to handle better the grpah analysis.
            [Still usefull with new method ?]

        Return
        ---
        hedge_parts: QgsVectorLayer
            Layer of parts connecting vegetation structure geometries
        """
        # Dissolve all networks layer
        layer_list = []

        # Diff with buffer used to split_by_network
        buffered = qw.buffer(network, offset, 
                            dissolve=True, end_style=2, 
                            join_style=0,segments=5)
        # Create single sided bfufer and correct invalid geometries
        left_side = self.prepare_hedge_parts(network, buffered, buffer, side=0)
        right_side = self.prepare_hedge_parts(network, buffered, buffer, side=1)
        layer_list += [left_side, right_side]

        # Merge and dissolve
        regroup = qw.merge_layers(layer_list)
        diss = qw.dissolve(regroup)

        diff = qw.difference(diss, str_layer)
        # To correct topological error small opening
        explode = qw.multipart_to_singleparts(diff)
        opened = qw.opening(explode, 0.1, False, end_style=2, 
                            join_style=0, segments=5)
        # hedge_parts = qw.snap(opened, hedge_layer, 0.3, 2)
        hedge_parts = qw.fix_geometries(opened)

        return hedge_parts
    
    def create_sub_network(self, network:QgsVectorLayer, 
                           geom_1:QgsGeometry, geom_2:QgsGeometry=None
                           )->QgsVectorLayer:
        """
        Create a sub network based on boundingbox of union of 
        boundingbox from input geoms.

        Parameters
        ---
        network : QgsVectorLayer : LineString
        geom_1 : QgsGeometry : Any
        geom_2 : QgsGeometry : Any

        Return
        ---
        sub_network ; QgsVectorLayer : LineString
        """
        sub_network = vutils.create_layer(network)
        bb_1 = geom_1.boundingBox()
        if geom_2 is not None:
            bb_2 = geom_2.boundingBox()
            union = QgsGeometry().fromRect(bb_1).combine(QgsGeometry().fromRect(bb_2))
            bb_of_union = union.boundingBox()
        else:
            bb_of_union = bb_1
        expression = QgsFeatureRequest().setFilterRect(bb_of_union)
        features_list = [feat for feat in network.getFeatures(expression)]
        sub_network.dataProvider().addFeatures(features_list)
        
        return sub_network

    def add_centroid_to_network(self, network: QgsVectorLayer, 
                                elements:Union[QgsFeature, QgsGeometry]
                                )-> QgsGeometry:
        """
        Add centroid or point on surface if the centroid is 
        not included inside the geometry in a network layer.

        Parameters
        ---
        network: QgsVectorLayer
            Network to add the centroids to
        elements: QgsFeature | QgsGeometry
            Features to fetch centroids from
        
        Return
        ---
        out_geom: QgsGeometry
            Dissolved geometries of the network where 
            the elements centroid have been added
        """
        if isinstance(elements[0], QgsFeature):
            elements = [f.geometry() for f in elements]

        dissolved = qw.dissolve(network)
        feat = next(dissolved.getFeatures())
        out_geom = feat.geometry()
        centroids = []
        for geom in elements:
            centroid = geom.centroid()
            if centroid.intersects(geom):
                centroids.append(centroid)
            else:
                centroids.append(geom.pointOnSurface())

        for centroid in centroids:
            x,y = centroid.asPoint().x(), centroid.asPoint().y()
            out_geom = g.vertex_add(out_geom, x,y)

        return out_geom
    
    def pairwise_ordering(self, count:int, features:list[QgsFeature], 
                          network_line:QgsVectorLayer
                          )->list[tuple[QgsFeature, QgsFeature]]:
        """
        From a list of features along a network will combine them 
        two by two given their respective position.

        Parameters
        ---
        count: Int
            Number of features to order
        features: list[QgsFeature]
            List of features to order
        network_line: QgsVectorLayer
            Linestring layer used to perform a "graph" analysis

        Return
        ---
        combi: list[tuple[QgsFeature, QgsFeature]]
            Each tuple contains associated geometries ordered 
            by the network distance between them.
            None if no hedge was found
        """
        if count > 2:
            id_list = [f.id() for f in features]
            combine_geom = g.combine_geometries(features)
            sub_network = self.create_sub_network(network_line, combine_geom)
            geom = self.add_centroid_to_network(sub_network, features)
        
            order = []        
            added = []
            # Retrieve the feature order in the network
            for vertex in geom.vertices():
                if vertex.wkbType() == QgsWkbTypes.PointZ:
                    x,y = vertex.x(), vertex.y()
                    pnt = QgsGeometry(QgsPoint(x,y))
                else:
                    pnt = QgsGeometry(vertex)

                intersecting = []
                for feature in features:
                    if feature.geometry().intersects(pnt.buffer(0.1, 5)):
                        intersecting.append(feature)

                if len(intersecting) == 1 \
                    and (pnt.intersects(intersecting[0].geometry().pointOnSurface()) \
                        or pnt.intersects(intersecting[0].geometry().centroid())):
                    idx = id_list.index(intersecting[0].id())
                    if idx not in added:
                        order.append(features[idx])
                        added.append(idx)
            # Create association by proximity of the features.
            combi = []
            for i in range(0, len(order)):
                if i < len(order) - 1:
                    combi.append((order[i], order[i+1]))

        elif count == 2:
            # If only two features just create a list of list for 
            # the next part iteration.
            combi = [features]

        else:
            combi = None

        return combi

    def find_start_and_end(self, network:QgsVectorLayer, 
                           geom_1:QgsGeometry, geom_2:QgsGeometry, 
                           part:QgsGeometry):
        """
        Find the start and end of the path to be computed in network
        Start and end are defined as closest linestring vertex of the input geoms interseciton with part.
        To find the closest linestring vertex, an iteration over geoms intersecting vertices is done.
        
        Parameters
        ---
        network : QgsVectorLayer : LineString
        geom_1 : QgsGeometry : Any
        geom_2 : QgsGeometry : Any
        part : QgsGeometry : Polygon : 
            Potential hedge part between geom_1 and geom_2
        Return
        ---
        start, end : QgsPointXY
        """

        inter_1 = geom_1.intersection(part)
        if inter_1.isEmpty():
            inter_1 = geom_1.intersection(part.buffer(0.1, 5))
        inter_2 = geom_2.intersection(part)  
        if inter_2.isEmpty():
            inter_2 = geom_2.intersection(part.buffer(0.1, 5))

        start = inter_1.closestVertex(inter_2.pointOnSurface().asPoint())[0]
        end = inter_2.closestVertex(start)[0]

        min_dist_start = min_dist_end = 99999
        for line in network.getFeatures():
            line = line.geometry().densifyByDistance(3)
            pnt_start, _, _, _, distance_start = line.closestVertex(start)
            pnt_end, _, _, _, distance_end = line.closestVertex(end)
            if distance_start < min_dist_start:
                min_dist_start = distance_start
                fake_start = pnt_start
                
            if distance_end < min_dist_end:
                min_dist_end = distance_end
                fake_end = pnt_end
                
        return start, end, fake_start, fake_end

    def check_distance(self, network_line: QgsVectorLayer, 
                       geom_1:QgsGeometry, geom_2:QgsGeometry, 
                       part:QgsGeometry, max_dist:float=25)->bool:
        """
        Check the euclidian distance between two geom.
        If this check is succefull it'll check the distance 
        along the associated network and if the shortest 
        euclidian path betwwen the two geom is intrsecting the network.

        Parameters
        ---
        network_line: QgsVectorLayer
            Linestring network to use for distance analysis
        geom_1: QgsGeometry
            Vegetation structure polygon to check 
        geom_2: QgsGeometry
            Vegetation structure polygon to check
        part: QgsGeometry
            Geometyr of the buffered network between geom_1 and geom_2
        max_dist: float
            Maximum distance allowed first euclidian, then by the network

        Return
        ---
        success: boolean
            True if the distance between the geometry is below max_dist
        start: None | QgsPointXY
            QgsPointXY if True
            Closest point of the intersection between the part and geom_1 and the network
        end: None | QgsPointXY
            QgsPointXY if True
            Closest point of the intersection between the part and geom_2 and the network
        """
        success = False
        start = end = None
        if 0 < geom_1.distance(geom_2) <= max_dist:
            # Create sub_network
            sub_network = self.create_sub_network(network_line, geom_1.buffer(max_dist,5), 
                                                  geom_2.buffer(max_dist,5))
            # Find start and end of the path
            start, end, fake_start, fake_end = self.find_start_and_end(sub_network, geom_1, geom_2, 
                                                                       part)
            # Compute shortest path
            route = g.shortest_path(sub_network, fake_start, fake_end)
            line = QgsGeometry().fromPolyline([QgsPoint(start), QgsPoint(end)])
            distance = route.length() if route is not None else 9999
            inter, _ = g.get_clementini(sub_network, line)

            if distance <= max_dist and inter == 0: 
                success = True

        return success, start, end

    def construct_polygon_link(self, start:QgsPointXY, end:QgsPointXY, 
                               geom_1:QgsGeometry, geom_2:QgsGeometry,
                               network_line:Union[QgsVectorLayer, None]=None, 
                               part:Union[QgsFeature, None]=None)->QgsGeometry:
        """
        Construct a new polygon that'll be touching 
        two geometry and fused with them.
        Two sides are constrcuted by retrieveing vertices of 
        each geometry until a length is reached.
        The two over sides are constructed between the first 
        and last vertices retrieved for the first step.

        Parameters
        ---
        start: QgsPointXY
            Starting point for the new polygon side with geom_1
        end: QgsPointXY
            Starting point for the new polygon side with geom_2
        geom_1: QgsGeometry
            Geometry to retrieve vertices around start from
        geom_2: QgsGeometry
            Geometry to retrieve vertices around end from
        network_line: QgsVectorLayer
            Linestring network to use to check if the created polygon is intersecting it
        part: QgsFeature
            If the new polygon intersect the network use the part geometry to clip it

        Return
        ---
        success: boolean
            True if the polygon was successfully created
        new_part: QgsGeometry 
        """
        success = False
        new_part = None
        start_id = geom_1.closestVertex(start)[1]
        end_id = geom_2.closestVertex(end)[1]

        vtces_1 = g.construct_short_side(geom_1, start, start_id)
        vtces_2 = g.construct_short_side(geom_2, end, end_id)
        if len(vtces_1) > 1 and len(vtces_2) > 1:
            long_sides = g.construct_long_sides(vtces_1, vtces_2)
            new_part = g.construct_polygon(vtces_1, vtces_2, long_sides)
            success = True

            # Handle network intersection 
            if network_line is not None:
                count, _ = g.get_clementini(network_line, new_part)
                if count >= 1:
                    new_part = new_part.intersection(part.geometry())
                    if len(new_part.asGeometryCollection()) != 1:
                        success = False
            
            # Handle case where part is overlapping 
            # or containing one of the two input geom
            comb_geom = geom_1.combine(geom_2)
            if new_part.buffer(0.1, 5).contains(comb_geom) \
                    or new_part.overlaps(comb_geom):
                new_part = new_part.difference(comb_geom)
                gc = new_part.asGeometryCollection()
                if len(gc) > 1:
                    max_area = 0
                    for geom in gc:
                        if geom.area() > max_area:
                            max_area = geom.area()
                            new_part = geom
                
        return success, new_part

    def compute_discontinuity(self, feat_1:QgsFeature, feat_2:QgsFeature, 
                         new_feat:QgsFeature)-> QgsFeature:
        """
        Estimate the real area of the new part by taking 
        the mean of the two geometry used to create it.
        Store the current area of the new_part and the 
        estimated area minus the current area in his field.

        Parameters
        ---
        feat_1:QgsFeature
        feat_2:QgsFeature
        new_feat: QgsFeature
            Feature containing the new part
        
        Return
        ---
        new_feat: QgsFeature
                  Same feature as input.
        """
        #Porosity - make function
        width_1 = feat_1.geometry().orientedMinimumBoundingBox()[3]
        width_2 = feat_2.geometry().orientedMinimumBoundingBox()[3]
        if feat_1["class"] == "Forest":
            mean_width = width_2
        elif feat_2["class"] == "Forest":
            mean_width = width_1
        else:
            mean_width = (width_1 + width_2)/2
        length = new_feat.geometry().orientedMinimumBoundingBox()[4]
        # Mettre une limite supérieur à mean_width ? Du genre 15 m?
        part_area = mean_width * length
        over_area = part_area - new_feat.geometry().area() if part_area != 0 else 0
        new_feat[self.idx_discontinuity] = part_area
        new_feat[self.idx_over] = over_area

        return new_feat

    def attribute_junction_type(self, feat_1:QgsFeature, feat_2:QgsFeature, 
                                new_feat:QgsFeature)->QgsFeature:
        """
        Attribute the junction type of the feat containing 
        the new part by comparing the class of the two geometry touching it.

        Parameters
        ---
        feat_1: QgsFeature
            Feature of one of the two geometry touching the new part
        feat_2: QgsFeature
            Feature of one of the two geometry touching the new part
        new_feat: QgsFeature
            Feature containing the new part
        
        Return
        ---
        feat: QgsFeature
            Same as input

        """
        types = [feat_1["class"], feat_2["class"]]
        if "Forest" in types or "Grove" in types:
            new_feat[self.idx_type] = 4
        elif types.count("Tree") == 2:
            new_feat[self.idx_type] = 3
        elif "Tree" in types and "Hedge" in types:
            new_feat[self.idx_type] = 2
        elif types.count("Hedge") == 2:
            new_feat[self.idx_type] = 1
        
        return new_feat
    
    def aggregate_with_network(self, hedge_parts:QgsVectorLayer, str_layer: QgsVectorLayer,
                               network:QgsVectorLayer, max_dist:float=25.0)->bool:
        """
        Use network layer and hedge parts created from it 
        to aggregate vegetation structure together.

        Parameters
        ---
        hedge_parts: QgsVectorLayer
            Buffered left and right side from network
        str_layer: QgsVectorLayer
            Vegetation structure layer to aggregate
        network: QgsVectorLayer
            Network layer to use for graph analysis
        max_dist: float
            Default 25.0 m
            Maximum distance to aggregate vegetation structure
        
        Return
        ---
        success: bool
            True if new feature were added to self.feat_list
            False otherwise

        """
        start_count = len(self.feat_list)
        dist_check = False
        link_creation = False
        for part in hedge_parts.getFeatures():
            count, features = g.get_clementini(str_layer, part.geometry())
            combi = self.pairwise_ordering(count, features, network)
            if count >= 2:
                for feat_1, feat_2 in combi:
                    geom_1 = feat_1.geometry()
                    geom_2 = feat_2.geometry()
                    
                    # Handle hedge directly connected to forest
                    if math.floor(geom_1.distance(geom_2)) == 0 \
                            and "Forest" in [feat_1["class"], feat_2["class"]]:
                        if feat_1["class"] == "Forest" and feat_2["class"] != "Forest":
                            self.attr_map[feat_2.id()] = {self.idx_type: 4}
                            
                        elif feat_1["class"] != "Forest" and feat_2["class"] == "Forest":
                            self.attr_map[feat_1.id()] = {self.idx_type: 4}
                        
                    elif feat_1["class"] == "Forest" and feat_2["class"] == "Forest":
                        continue
                        
                    else:
                        dist_check, start, end = self.check_distance(network, geom_1, geom_2, 
                                                                    part.geometry(), max_dist)
                        if dist_check:
                            dist_check = False
                            link_creation, new_part = \
                                self.construct_polygon_link(start, end, geom_1, geom_2, 
                                                            network, part)

                        if link_creation:
                            link_creation = False
                            # Create new_part feature
                            new_feat = at.create_feature(self.fields, new_part, self.max_fid)
                            new_feat = self.compute_discontinuity(feat_1, feat_2, new_feat)
                            new_feat = self.attribute_junction_type(feat_1, feat_2, new_feat)
                            self.feat_list.append(new_feat)
                            self.max_fid += 1
                    self.checked.append(utils.concat(feat_1.id(), feat_2.id()))

        end_count = len(self.feat_list)

        if end_count - start_count > 0:
            success = True
        else:
            success = False

        return success
    
    def fetch_closest_point(self, geom_1:QgsGeometry, geom_2:QgsGeometry
                            )->tuple[QgsPointXY, QgsPointXY]:
        """
        Iteratively fetch the closest point between 
        two geometries by using the previous closest point.
        Will stop when the distance is not mooving anymore.
        First point is the center of gravity of one geometry.

        Parameters
        ---
        geom_1: QgsGeometry
            Geometry to fetch the closest point from
        geom_2: QgsGeometry
            Geometry to fetch the closest point from

        Return
        ---
        pnt_1: QgsPointXY
            Closest vertex from geom_1 to geom_2
        pnt_2: QgsPointXY
            Closest vertex from geom_2 to geom_1
        """
        pnt_2 = geom_2.pointOnSurface().asPoint()
        min_dist = 9999
        dist = 9998
        while dist < min_dist:
            min_dist = dist
            pnt_1 = geom_1.closestVertex(pnt_2)[0]
            pnt_2 = geom_2.closestVertex(pnt_1)[0]
            
            dist = pnt_1.distance(pnt_2)
        
        return pnt_1, pnt_2

    def handle_simple_junction(self, start:QgsPointXY, end:QgsPointXY, 
                               geom_1:QgsGeometry, geom_2:QgsGeometry,
                               ori_1:float, ori_2: float,
                               network:Union[QgsVectorLayer, None],
                               delta:float=25.0)->Union[QgsFeature, None]:
        """
        Create a geometry that'll serves as junction between two polygons.
        Put it inside a QgsFeature and a feature list.

        Parameters
        ---
        start: QgsPointXY
            Closest vertex from geom_1 to geom_2
        end: QgsPointXY
            Closest vertex from geom_2 to geom_1
        geom_1: QgsGeometry
        geom_2 : QgsGeometry
        network_line: QgsVectorLayer|None
            Network to use to prevent junction to cross a geometry from this layer.
        
        Return
        ---
        feat: QgsFeature
            Feature containing the junction between geom_1 and geom_2
            Return None if a junction was not created
        """
        feat = None

        _, new_part = self.construct_polygon_link(start, end, geom_1, geom_2)
        ori_new_part = new_part.orientedMinimumBoundingBox()[2]

        if network is not None:
            count, _ = g.get_clementini(network, new_part)
        else:
            count = 0

        if count == 0 and ((g.compare_orientation(ori_new_part, ori_2, delta)) \
            or (g.compare_orientation(ori_new_part, ori_1, delta))):
            feat = at.create_feature(self.fields, new_part, self.max_fid)

        return feat

    def handle_complex_junction(self, start:QgsPointXY, end:QgsPointXY, 
                                geom_1:QgsGeometry, geom_2:QgsGeometry,
                                network:Union[QgsVectorLayer, None],
                                max_dist:float=25.0, delta:float=25.0, 
                                bypass_dist:float=2.0)->Union[QgsFeature, None]:
        """
        Create a geometry that'll serves as junction between two polygons.
        Put it inside a QgsFeature and a feature list.

        Parameters
        ---
        start: QgsPointXY
            Closest vertex from geom_1 to geom_2
        end: QgsPointXY
            Closest vertex from geom_2 to geom_1
        geom_1: QgsGeometry
        geom_2 : QgsGeometry
        network_line: QgsVectorLayer|None
            Network to use to prevent junction to cross a geometry from this layer.
        max_dist: float
            Default 25.0 m
            Max distance to aggregate features together.
        delta: float
            Default 25.0 °
            Orientation delta between both geometry to consider similar.
        bypass_dist: float
            Default 2.0 m
            Bypass distance to avoid checking delta and max_dist before aggregation.

        Return
        ---
        feat: QgsFeature
            Feature containing the junction between geom_1 and geom_2
            Return None if a junction was not created
        """
        feat = None

        if bypass_dist < start.distance(end) <= max_dist:
            success_ori, ori_1, ori_2 = g.compare_orientation_advanced(start, end, 
                                                                       geom_1, geom_2, 
                                                                       delta)
            if success_ori:
                success, new_part = self.construct_polygon_link(start, end, geom_1, geom_2)
                ori_new_part = new_part.orientedMinimumBoundingBox()[2]

                if network is not None:
                    count, _ = g.get_clementini(network, new_part)
                else:
                    count = 0
                
                if count == 0 and ((g.compare_orientation(ori_new_part, ori_2, delta)) \
                    or (g.compare_orientation(ori_new_part, ori_1, delta))):
                    feat = at.create_feature(self.fields, new_part, self.max_fid)

        elif 0 < start.distance(end) <= bypass_dist:
            success, new_part = self.construct_polygon_link(start, end, geom_1, geom_2)
            if network is not None:
                count, _ = g.get_clementini(network, new_part)
                if count == 0:
                    feat = at.create_feature(self.fields, new_part, self.max_fid)
            else:
                feat = at.create_feature(self.fields, new_part, self.max_fid)

        return feat

    def aggregate_without_network(self, str_layer:QgsVectorLayer, network:Union[QgsVectorLayer,None], 
                                  max_dist:float=25.0, delta:float=25.0, bypass_dist:float=2.0):
        """
        Create junctions between vegetation structure without the help of a network.
        However if a network is provided it is ensured that the junction does not intersect it.

        Parameters
        ---
        str_layer: QgsVectorLayer
            Vegetation structure layer to aggregate
        network: QgsVectorLayer
            Network layer to use for graph analysis
        max_dist: float
            Default 25.0 m
            Maximum distance to aggregate vegetation structure
        delta: float
            Default 25.0 °
            Orientation delta between both geometry to consider similar.
        bypass_dist: float
            Default 2.0 m
            Bypass distance to avoid checking delta and max_dist before aggregation.

        Return
        ---
        success: bool
            True if new feature were added to self.feat_list
            False otherwise
        """
        buffer = qw.buffer(str_layer, max_dist)
        # Intersection avec buffer
        request = QgsFeatureRequest().setFilterExpression("class != 'Forest'")
        # To avoid having forest/forets interface
        start_count = len(self.feat_list)

        for feature in buffer.getFeatures(request):
            count, features = g.get_clementini(str_layer, feature)
            if count > 1:
                feat_1 = [f for f in features if f["fid"] == feature["fid"]][0]
                features = [f for f in features if f["fid"] != feature["fid"]]
                geom_1 = feat_1.geometry()
                for feat_2 in features:
                    if utils.concat(feat_1.id(), feat_2.id()) not in self.checked \
                        and utils.concat(feat_2.id(), feat_1.id()) not in self.checked:
                            
                        geom_2 = feat_2.geometry()

                        if math.floor(geom_1.distance(geom_2)) != 0:
                            start, end = self.fetch_closest_point(geom_1, geom_2)
                            ori_1 = geom_1.orientedMinimumBoundingBox()[2]
                            ori_2 = geom_2.orientedMinimumBoundingBox()[2]

                            if g.compare_orientation(ori_1, ori_2, delta):
                                new_feat = self.handle_simple_junction(start, end, 
                                                                       geom_1, geom_2, 
                                                                       ori_1, ori_2,
                                                                       network)
                            else:
                                new_feat = self.handle_complex_junction(start, end, 
                                                                        geom_1, geom_2,
                                                                        network, max_dist, 
                                                                        delta, bypass_dist)
                            if new_feat is not None:
                                self.max_fid += 1
                                new_feat = self.compute_discontinuity(feat_1, feat_2, new_feat)
                                new_feat = self.attribute_junction_type(feat_1, feat_2, new_feat)
                                self.feat_list.append(new_feat)
                        self.checked.append(utils.concat(feat_1.id(), feat_2.id()))

        end_count = len(self.feat_list)

        if end_count - start_count > 0:
            success = True
        else:
            success = False

        return success
    
    def add_junctions(self, str_layer:QgsVectorLayer, offset:Union[None,float]=None)->QgsVectorLayer:
        """
        Add self.feat_list. Populate discontinuity and topology 
        fields with self.attr_map.
        Delete forest and grove from vegetation structure 
        to not merge them with junctions.
        Then perform various spatial operation to merge 
        junctions with vegetation structure and correct topology.

        Parameters
        ---
        str_layer: QgsVectorLayer
            Vegetation structure layer
        offset: Union[None,float] : None
            Value used to split by network.
            It'll be to used correct the junction width without aggregating parallel hedges
        
        Return
        ---
        agr_layer: QgsVectorLayer
            Hedge and tree merged with junctions
        """
        str_layer.dataProvider().addFeatures(self.feat_list)
        str_layer.dataProvider().changeAttributeValues(self.attr_map)
        expression = "class in ('Forest', 'Grove')"
        # Delete forest as we do not want to fuse them with the new parts
        request = QgsFeatureRequest().setFilterExpression(expression)
        del_id = [f.id() for f in str_layer.getFeatures(request)]
        str_layer.dataProvider().deleteFeatures(del_id)
        #snapped = qw.snap(str_layer, str_layer, 0.1, 1)
        # Fuse new part and perform a little closing to avoid narrow junction
        fix = qw.buffer(str_layer, 0.0001, 5, end_style=1)
        single = qw.dissolve(fix, separate_disjoint=True)
        agr_layer = qw.closing(single, offset/1.3333, dissolve=False, end_style=1, join_style=1)

        return agr_layer
    
    def estimate_discontinuity_and_typology(self, agr_layer:QgsVectorLayer, 
                                       str_layer:QgsVectorLayer)->QgsVectorLayer:
        """
        Estimate fused hedge discontinuity with the stored "real" area 
        of the parts and the real parts area.
        Also estimate the typology of the fused hedge 
        given the most occuring junction type.

        Parameters
        ---
        agr_layer: QgsVectorLayer
             Vegetation structure layer merged new parts
        str_layer: QgsVectorLayer
            Vegetation structure layer containing the unmerged new parts
        """
        deletion = set()
        exp_1 = "part_type is not NULL"
        exp_2 = "part_type is NULL"
        req_1 = QgsFeatureRequest().setFilterExpression(exp_1)
        req_2 = QgsFeatureRequest().setFilterExpression(exp_2)

        for feature in str_layer.getFeatures(req_1):
            _, parts = g.get_clementini(str_layer, feature, req_1)
            for part in parts:
                count, _ = g.get_clementini(str_layer, part, req_2)
                if count > 2:
                    deletion.add(part.id())

        str_layer.dataProvider().deleteFeatures(list(deletion))

        attr_map = {}
        for hedge in agr_layer.getFeatures():
            main_typo = NULL
            discontinuity = NULL
            _, feats = g.get_clementini(str_layer, hedge)
            tot_part_area = sum([f[self.idx_discontinuity] for f in feats 
                                 if f[self.idx_discontinuity] != NULL])
            tot_over_area = sum([f[self.idx_over] for f in feats if f[self.idx_over] != NULL])
            if hedge.geometry().area() != 0:
                discontinuity = round(tot_part_area / (hedge.geometry().area() + tot_over_area), 2)
            discontinuity = NULL if tot_part_area == 0 else discontinuity
            
            #if discontinuity is not NULL:
            main_typo = []
            typo = [f[self.idx_type] for f in feats if f[self.idx_type] != NULL]
            if bool(typo):
                occ = np.bincount(typo)
                maximum = max(occ)
                for i in range(len(occ)): 
                    if occ[i] == maximum: 
                        main_typo.append(i)
            
            if len(main_typo) >= 1:
                # Always proritize hedge/hedge if same occurence
                main_typo = min(main_typo)
            else:
                main_typo = NULL

            if main_typo == 1:
                main_typo = "Hedge/Hedge"
            elif main_typo == 2:
                main_typo = "Hedge/Tree"
            elif main_typo == 3:
                main_typo = "Tree/Tree"
            elif main_typo == 4:
                main_typo = "Forest/*"

            attr_map[hedge.id()] = {self.idx_discontinuity: discontinuity, self.idx_typo: main_typo}
        agr_layer.dataProvider().changeAttributeValues(attr_map)
        at.delete_fields(agr_layer, ["over_area", "part_type"])

        return agr_layer

    def processing_fields_deletion(self, layer:QgsVectorLayer)-> QgsVectorLayer:
        """
        Will use self.layers and the current label to fetch input fields 
        and delete new fields created by the processing.

        Parameters
        ---
        layer: QgsVectorLayer
            Current splitted layer.
        label: str
            Current label.

        Return
        ---
        output: QgsVectorLayer
        """
        del_list = list(set(layer.fields().names()) - set(self.fields.names()))
        output = qw.delete_column(layer, del_list)

        return output
    
    def create_sink(self, layer:QgsVectorLayer, sink_name:str,
                    parameters:dict, context:QgsProcessingContext)-> int:
        """
        Create a sink and store the id in self.output

        Parameters
        ---
        layer: QgsVectorLayer
            Layer to duplicate into a sink
        
        Return
        ---
        id: int
            Identifier of the sink layer
        """
        (sink, id) = self.parameterAsSink(parameters,
                                          sink_name,
                                          context,
                                          layer.fields(),
                                          layer.wkbType(),
                                          layer.sourceCrs())
        for feature in layer.getFeatures():
            sink.addFeature(feature, QgsFeatureSink.FastInsert)

        return id
            
    def icon(self):
        """
        Should return a QIcon which is used for your provider inside
        the Processing toolbox.
        """
        return QIcon(":/plugins/hedge_tools/images/hedge_tools.png")
    
    def shortHelpString(self):
        """
        Returns a localised short help string for the algorithm.
        """
        return self.tr("Create new geometries linking near by and aligned geometries. \
                        Geometries can include several layer such as : \
                        forest, grove, hedge and tree. \n\
                        Hedges and trees will be fused with the new link but \
                        junction with forest or grove will not. \
                        In all case the new hedges will have two new attributes : \n\
                        - Discontinuity will be estimated ; \n\
                        - A junction type will assigned based on the most occuring features. \
                          Grove is considered as Forest in the junction type attribute. \n\
                        Several parameters condition the aggregation. \
                        First, if a network is used there is a buffer value and an offset value. \
                        This algorithm yield better results if networks layer such as rivers, \
                        railways or roads are used. If this is the case it is strongly advised \
                        to use the split by network algorithm before and to use the buffer value \
                        from split by network as the offset value in this tool. \
                        Regardless, the buffer value should be strictly greater than the offset value. \n\
                        There is also the aggregation distance and the delta who respectively condition \
                        the distance and the difference in orientation allowed between \
                        two geometries to consider the fusion. \n\
                        Last, a bypass distance is used to bypass the delta and aggregation distance \
                        threshold in case two geometries are close together. \
                        The bypass distance should be smaller than the aggregation distance.")

    def name(self):
        """
        Returns the algorithm name, used for identifying the algorithm. This
        string should be fixed for the algorithm, and must not be localised.
        The name should be unique within each provider. Names should contain
        lowercase alphanumeric characters only and no spaces or other
        formatting characters.
        """
        return 'aggregation'

    def displayName(self):
        """
        Returns the translated algorithm name, which should be used for any
        user-visible display of the algorithm name.
        """
        return self.tr("7 - Aggregation [optional]")

    def group(self):
        """
        Returns the name of the group this algorithm belongs to. This string
        should be localised.
        """
        return self.tr("0 - Extraction [optional]")

    def groupId(self):
        """
        Returns the unique ID of the group this algorithm belongs to. This
        string should be fixed for the algorithm, and must not be localised.
        The group id should be unique within each provider. Group id should
        contain lowercase alphanumeric characters only and no spaces or other
        formatting characters.
        """
        return 'extraction'

    def tr(self, string):
        return QCoreApplication.translate('Processing', string)

    def checkParameterValues(self, parameters, context):
        
        if parameters[self.HEDGES] is None and parameters[self.TREES] is None:
            return (False, "At least one input layer is needed.")
        
        if parameters[self.BUFFER] <= parameters[self.OFFSET]:
            return (False, "Buffer value should be strictly bigger than offset value.")
        
        if parameters[self.MAX_DIST] < parameters[self.BYPASS_DIST]:
            return (False, "Aggregation distance should be bigger than bypass distance.")
        
        return (True, '')

    def createInstance(self):
        return AggregationAlgorithm()