# -*- coding: utf-8 -*-

"""
/***************************************************************************
 Connecitivity metrics
                                 A QGIS plugin
 Network analysis at subgraph scale.
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2022-01-22
        copyright            : (C) 2022 by Dynafor
        email                : gabriel.marques@toulouse-inp.fr
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

__author__ = "Dynafor"
__date__ = "2022-01-22"
__copyright__ = "(C) 2022 by Dynafor"

# This will get replaced with a git SHA1 when you do a git archive

__revision__ = "$Format:%H$"

from qgis.PyQt.QtCore import QCoreApplication, QVariant
from qgis.core import (
    QgsProcessing,
    QgsProcessingAlgorithm,
    QgsProcessingParameterFeatureSource,
    QgsProcessingParameterBoolean,
    QgsProcessingParameterField,
    QgsProcessingParameterEnum,
    QgsFeatureRequest,
    NULL,
)
from qgis.PyQt.QtGui import QIcon
from hedge_tools import (
    resources,
)  # Only need in hedge_tools.py normaly but just to keep track of import

from hedge_tools.tools.classes import HedgeNetwork as hN
from hedge_tools.tools.vector import attribute_table as at


class ConnectivityMetricsAlgorithm(QgsProcessingAlgorithm):
    """
    Compute connectivity metrics for each subgraphs such as :
    - Links/edges number ;
    - Vertices number ;
    - Cycle number ;
    - Total length ;
    - Alpha index ;
    - Bêta index ;
    - Gamme index ;
    - Eta index ;
    - Detour index ;
    - Density index

    See HedgeNetwork docstring for more info and potential development

    Most metrics only make sense in a network but
    some such as Detour index can be used as curvature index for single hedges.

    Parameters
    ----------
    NETWORK_EDGES (QgisObject : QgsVectorLayer) : MultiLineString layer.
    NETWORK_VERTICES (QgisObject : QgsVectorLayer) : Node layer. Contains nodes with a comp field
    EDGES_ID : QgsField : Int
    VERTICES_ID : QgsField :Int
    NODE_TYPE : QgsField : Str
    REMOVE_SINGLE : Bool : Allow to set single hedges useless index to NULL
    METRICS : ite[int]

    Return
    ------
    OUTPUT_NET_EDGES (QgisObject : QgsVectorLayer) : MultiLineString layer.
    """

    # Constants used to refer to parameters and outputs. They will be
    # used when calling the algorithm from another algorithm, or when
    # calling from the QGIS console.
    NETWORK_EDGES = "NETWORK_EDGES"
    NETWORK_VERTICES = "NETWORK_VERTICES"
    EDGES_ID = "EDGES_ID"
    VERTICES_ID = "VERTICES_ID"
    NODE_TYPE = "NODE_TYPE"
    REMOVE_SINGLE = "REMOVE_SINGLE"
    METRICS = "METRICS"
    OUTPUT_NET_EDGES = "OUTPUT_NET_EDGES"

    def initAlgorithm(self, config):
        """
        Here we define the inputs and output of the algorithm, along
        with some other properties.
        """
        # We add the input vector lines features source.
        self.addParameter(
            QgsProcessingParameterFeatureSource(
                self.NETWORK_EDGES,
                self.tr("Subgraph edges layer"),
                [QgsProcessing.TypeVectorLine],
            )
        )

        # We add the input vector nodes features source.
        self.addParameter(
            QgsProcessingParameterFeatureSource(
                self.NETWORK_VERTICES,
                self.tr("Subgraph vertices layer"),
                [QgsProcessing.TypeVectorPoint],
            )
        )

        # Unique id
        self.addParameter(
            QgsProcessingParameterField(
                self.EDGES_ID,
                self.tr("Subgraph edges id"),
                type=QgsProcessingParameterField.Numeric,
                parentLayerParameterName="NETWORK_EDGES",
                defaultValue="comp",
            )
        )

        self.addParameter(
            QgsProcessingParameterField(
                self.VERTICES_ID,
                self.tr("Subgraph vertices id"),
                type=QgsProcessingParameterField.Numeric,
                parentLayerParameterName="NETWORK_VERTICES",
                defaultValue="comp",
            )
        )

        # Node type
        self.addParameter(
            QgsProcessingParameterField(
                self.NODE_TYPE,
                self.tr("Subgraph vertices type"),
                type=QgsProcessingParameterField.String,
                parentLayerParameterName="NETWORK_VERTICES",
                defaultValue="Node_type",
            )
        )

        # Handle unconnected hedges
        self.addParameter(
            QgsProcessingParameterBoolean(
                name=self.REMOVE_SINGLE,
                description=self.tr("Ignore single edge network"),
                defaultValue=True,
            )
        )

        # Which width to compute
        self.addParameter(
            QgsProcessingParameterEnum(
                name=self.METRICS,
                description=self.tr("Connectivity metrics"),
                options=[
                    "Number of links",
                    "Number of nodes",
                    "Cyclometric number",
                    "Network length",
                    "Alpha index",
                    "Bêta index",
                    "Gamma index",
                    "Eta index",
                    "Detour index",
                    "Network density",
                ],
                optional=False,
                allowMultiple=True,
            )
        )

    def __init__(self):
        super().__init__()

        # key = index of the metrics, value : (name, type) of field to create
        self.field_map = {
            0: ("Nb_links", QVariant.Int),
            1: ("Nb_nodes", QVariant.Int),
            2: ("Nb_cycles", QVariant.Int),
            3: ("Tot_lgth", QVariant.Double),
            4: ("Alpha", QVariant.Double),
            5: ("Bêta", QVariant.Double),
            6: ("Gamma", QVariant.Double),
            7: ("Eta", QVariant.Double),
            8: ("Detour", QVariant.Double),
            9: ("Density", QVariant.Double),
        }

    def processAlgorithm(self, parameters, context, feedback):
        """
        Here is where the processing itself takes place.
        """

        network_edges = self.parameterAsVectorLayer(
            parameters, self.NETWORK_EDGES, context
        )
        network_vertices = self.parameterAsVectorLayer(
            parameters, self.NETWORK_VERTICES, context
        )
        edges_id = self.parameterAsFields(parameters, self.EDGES_ID, context)[0]
        vertices_id = self.parameterAsFields(parameters, self.VERTICES_ID, context)[0]
        node_type = self.parameterAsFields(parameters, self.NODE_TYPE, context)[0]
        remove_single = self.parameterAsBool(parameters, self.REMOVE_SINGLE, context)
        metrics = self.parameterAsEnums(parameters, self.METRICS, context)

        feedback.pushInfo("Starting processing")

        # Check for cancellation
        if feedback.isCanceled():
            return {}

        count = network_edges.featureCount()

        index_map = self.create_output_fields(metrics, network_edges)

        attr_map = {}
        for current, subgraph in enumerate(network_edges.getFeatures()):
            request = QgsFeatureRequest().setFilterExpression(
                f"{node_type} != 'L' and {vertices_id} = {subgraph[edges_id]}"
            )
            nodes = list(network_vertices.getFeatures(request))
            network = hN.HedgeNetwork(subgraph, nodes)
            attr_map[subgraph.id()] = self.compute_metrics(
                index_map, network, remove_single
            )

            feedback.setProgress(int((current / count) * 100))
            # Check for cancellation
            if feedback.isCanceled():
                return {}

        network_edges.dataProvider().changeAttributeValues(attr_map)

        return {"OUTPUT_NET_EDGES": parameters[self.NETWORK_EDGES]}

    def create_output_fields(self, metrics, subgraph_layer):
        """
        Create the metrics field wanted by user and
        return a map of field position for each index

        Parameters
        ----------
        metrics : ite[str]
        subgraph_layer : QgsVectorLayer:MultiLineString

        Return
        ------
        index_map : dict{metrics index: field index}

        TODO: Remove index_map and use create_fields output
        """
        fields = [self.field_map[key] for key in metrics]
        _ = at.create_fields(subgraph_layer, fields)

        index_map = {
            k: subgraph_layer.fields().indexFromName(name)
            for k, (name, _) in self.field_map.items()
            if k in metrics
        }

        return index_map

    def compute_metrics(self, index_map, network, remove_single=True):
        """
        Compute the metrics wanted by the users
        and build the attr_map for the current feature

        Parameters
        ----------
        index_map : dict{metrics index: field index}
        network : HedgeNetwork instance
        remove_single : boolean : True

        Return
        ------
        results : dict{field_index: value} : Attr map for the current feature
        """
        results = {}
        for k, v in index_map.items():
            if remove_single is False or network.nb_link > 1:
                if k == 0:
                    results[v] = network.nb_link
                if k == 1:
                    results[v] = network.nb_node
                if k == 2:
                    results[v] = network.cyclomatic_number
                if k == 3:
                    results[v] = network.compute_network_length()
                if k == 4:
                    results[v] = network.compute_alpha_index()
                if k == 5:
                    results[v] = network.compute_beta_index()
                if k == 6:
                    results[v] = network.compute_gamma_index()
                if k == 7:
                    results[v] = network.compute_eta_index()
                if k == 8:
                    results[v] = network.compute_detour_index()
                if k == 9:
                    results[v] = network.compute_network_density()
            else:
                results[v] = NULL

        return results

    def icon(self):
        """
        Should return a QIcon which is used for your provider inside
        the Processing toolbox.
        """
        return QIcon(":/plugins/hedge_tools/images/hedge_tools.png")

    def shortHelpString(self):
        """
        Returns a localised short help string for the algorithm.
        """
        return self.tr(
            "Compute connectivity metrics for each subgraph such as : \n\
                        - Links/edges number: the number of arcs;\n\
                        - Vertices number: the number of nodes;\n\
                        - Cycle number: the number of closed circuits;\n\
                        - Total length: the total length of the network;\n\
                        - Alpha index: a ratio of existing circuits to the maximum possible circuits (0 to 1);\n\
                        - Beta index: an average number of edges per vertex (0 to 1);\n\
                        - Gamma index: a ratio between the number of observed links and \
                          the number of possible links (0 to 1);\n\
                        - Eta index: average length per link;\n\
                        - Detour index: a measure of network efficiency (0 to 1);\n\
                        - Density index: a measure of network development.\n\
                        Most metrics only make sense in a network, but some, such as the Detour index, \
                        can be used as curvature index for single hedges.\n\
                        Recommended use: ignore single-edge networks, as most metrics do not apply \
                        and can create statistical bias in further analysis."
        )

    def name(self):
        """
        Returns the algorithm name, used for identifying the algorithm. This
        string should be fixed for the algorithm, and must not be localised.
        The name should be unique within each provider. Names should contain
        lowercase alphanumeric characters only and no spaces or other
        formatting characters.
        """
        return "connectivitymetrics"

    def displayName(self):
        """
        Returns the translated algorithm name, which should be used for any
        user-visible display of the algorithm name.
        """
        return self.tr("2 - Connectivity metrics")

    def group(self):
        """
        Returns the name of the group this algorithm belongs to. This string
        should be localised.
        """
        return self.tr("6 - Network level")

    def groupId(self):
        """
        Returns the unique ID of the group this algorithm belongs to. This
        string should be fixed for the algorithm, and must not be localised.
        The group id should be unique within each provider. Group id should
        contain lowercase alphanumeric characters only and no spaces or other
        formatting characters.
        """
        return "hedgenetwork"

    def tr(self, string):
        return QCoreApplication.translate("Processing", string)

    def createInstance(self):
        return ConnectivityMetricsAlgorithm()
