# -*- coding: utf-8 -*-

"""
/***************************************************************************
 SplitByDistance
                                 A QGIS plugin
 Split the hedges data with adjacent interface changes.
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2022-01-22
        copyright            : (C) 2022 by Dynafor
        email                : gabriel.marques@toulouse-inp.fr
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

__author__ = "Dynafor"
__date__ = "2022-01-22"
__copyright__ = "(C) 2022 by Dynafor"

# This will get replaced with a git SHA1 when you do a git archive

__revision__ = "$Format:%H$"

from qgis.PyQt.QtCore import QCoreApplication
from qgis.core import (
    QgsProcessing,
    QgsFeatureSink,
    QgsProcessingAlgorithm,
    QgsProcessingParameterFeatureSource,
    QgsProcessingParameterFeatureSink,
    QgsProcessingParameterNumber,
    QgsProcessingParameterBoolean,
    QgsProcessingParameterField,
)
from qgis.PyQt.QtGui import QIcon
from hedge_tools import (
    resources,
)  # Only need in hedge_tools.py normaly but just to keep track of import

import processing
from hedge_tools.tools.classes import class_hedge as h
from hedge_tools.tools.vector import utils


class SplitByInterfaceAlgorithm(QgsProcessingAlgorithm):
    """
    Split the arc and polygons with adjcent interface changes (fields limit, ...).
    Create new start and end nodes for each new hedges.

    Parameters
    ---
    INPUT_POLY (QgisObject : QgsVectorLayer) : Polygon layer path. Contains hedges
    INPUT_ARC (QgisObject : QgsVectorLayer) : Linestring layer path. Contains arc (polyline) hedges
    INPUT_NODE (QgisObject : QgsVectorLayer) : Node layer path. Contains nodes to delimit the hedges
    INPUT_FIELD (str) : Field name for node type (O,L,T, ...)
    INPUT_ITF (QgisObject : QgsVectorLayer) : Layer of the interface to cut hedges with
    INPUT_ORI (int) : Orientation threshold between adjacent junction and hedges to allow cutting them.
         ori hedge < ori junction - INPUT_ORI  or ori hedge > ori junction + INPUT_ORI
        If the junction and the hedge are parallel or almost the junction is ignored.
    INPUT_HANDLE_DIST (bool) : IF true cancel creation of small hedges by removing one of the two cutlines
    INPUT_DIST (int) : Minimum distance for an hedges


    Return
    ---
    OUTPUT_POLY (QgisObject : QgsVectorLayer) : Polygon. Layer splited by orientation change
    OUTPUT_ARC (QgisObject : QgsVectorLayer) : Linestring. Layer splited by orientation change
    OUTPUT_NODE (QgisObject : QgsVectorLayer) : Node. Layer with L node (orientation change)
    """

    # Constants used to refer to parameters and outputs. They will be
    # used when calling the algorithm from another algorithm, or when
    # calling from the QGIS console.
    INPUT_POLY = "INPUT_POLY"
    INPUT_ARC = "INPUT_ARC"
    INPUT_NODE = "INPUT_NODE"
    INPUT_FIELD = "INPUT_FIELD"
    INPUT_ITF = "INPUT_ITF"
    INPUT_ORI = "INPUT_ORI"

    INPUT_HANDLE_DIST = "INPUT_HANDLE_DIST"
    INPUT_DIST = "INPUT_DIST"

    OUTPUT_POLY = "OUTPUT_POLY"
    OUTPUT_ARC = "OUTPUT_ARC"
    OUTPUT_NODE = "OUTPUT_NODE"

    def initAlgorithm(self, config):
        """
        Here we define the inputs and output of the algorithm, along
        with some other properties.
        """

        # We add the input vector polygons features source.
        self.addParameter(
            QgsProcessingParameterFeatureSource(
                self.INPUT_POLY,
                self.tr("Polygons vector layer"),
                [QgsProcessing.TypeVectorPolygon],
            )
        )

        # We add the input vector lines features source.
        self.addParameter(
            QgsProcessingParameterFeatureSource(
                self.INPUT_ARC,
                self.tr("Arcs vector layer"),
                [QgsProcessing.TypeVectorLine],
            )
        )

        # We add the input vector nodes features source.
        self.addParameter(
            QgsProcessingParameterFeatureSource(
                self.INPUT_NODE,
                self.tr("Nodes vector layer"),
                [QgsProcessing.TypeVectorPoint],
            )
        )

        # We add the input field nodes type.
        self.addParameter(
            QgsProcessingParameterField(
                self.INPUT_FIELD,
                self.tr("Nodes type field"),
                type=QgsProcessingParameterField.String,
                parentLayerParameterName="INPUT_NODE",
            )
        )

        # We add the input vector nodes features source.
        self.addParameter(
            QgsProcessingParameterFeatureSource(
                self.INPUT_ITF,
                self.tr("Interface layer"),
                [QgsProcessing.TypeVectorPolygon],
            )
        )

        # Orientation parameter for splitting
        self.addParameter(
            QgsProcessingParameterNumber(
                name=self.INPUT_ORI,
                description=self.tr("Offset tolerance (°)"),
                type=QgsProcessingParameterNumber.Integer,
                defaultValue=15,
                optional=False,
                minValue=0,
                maxValue=90,
            )
        )

        # Handle distance parameter
        self.addParameter(
            QgsProcessingParameterBoolean(
                name=self.INPUT_HANDLE_DIST,
                description=self.tr("Cancel small hedges creation"),
                defaultValue=False,
            )
        )

        # Distance value parameter
        self.addParameter(
            QgsProcessingParameterNumber(
                name=self.INPUT_DIST,
                description=self.tr("Minimum hedge length (meters)"),
                type=QgsProcessingParameterNumber.Integer,
                defaultValue=25,
                optional=True,
                minValue=0,
                maxValue=1000,
            )
        )

        # We add a feature sink in which to store our processed features (this
        # usually takes the form of a newly created vector layer when the
        # algorithm is run in QGIS).

        # Output polys sink
        self.addParameter(
            QgsProcessingParameterFeatureSink(
                self.OUTPUT_POLY,
                self.tr("Interface polygons"),
            )
        )

        # Output arcs sink
        self.addParameter(
            QgsProcessingParameterFeatureSink(
                self.OUTPUT_ARC,
                self.tr("Interface arcs"),
            )
        )

        # Output nodes sink
        self.addParameter(
            QgsProcessingParameterFeatureSink(
                self.OUTPUT_NODE,
                self.tr("Interface nodes"),
            )
        )

    def processAlgorithm(self, parameters, context, feedback):
        """
        Here is where the processing itself takes place.
        """

        poly_layer = self.parameterAsVectorLayer(parameters, self.INPUT_POLY, context)
        arc_layer = self.parameterAsVectorLayer(parameters, self.INPUT_ARC, context)
        node_layer = self.parameterAsVectorLayer(parameters, self.INPUT_NODE, context)
        itf_layer = self.parameterAsVectorLayer(parameters, self.INPUT_ITF, context)

        feedback.pushInfo("Starting processing")

        # Check for cancellation
        if feedback.isCanceled():
            return {}

        input_fields = set(poly_layer.fields().names())
        input_fields.add("o_pid")
        # Dupe as instanciation do not create deep copy of the layers
        # and QGIS crash if the tool return an input layer
        # poly_layer = utils.create_layer(poly_layer, copy_feat=True, copy_field=True)
        # arc_layer = utils.create_layer(arc_layer, copy_feat=True, copy_field=True)
        # node_layer = utils.create_layer(node_layer, copy_feat=True, copy_field=True)
        itf_layer = utils.create_layer(itf_layer, copy_feat=True, copy_field=True)

        # Init hedge class and compute length
        hedges = h.Hedges(poly_layer, arc_layer, node_layer)
        itf_poly_layer, itf_arc_layer, itf_node_layer = hedges.split_by_interface(
            context,
            feedback,
            itf_layer=itf_layer,
            ori_bound=parameters["INPUT_ORI"],
            ignore_dist=(
                parameters["INPUT_DIST"] if parameters["INPUT_HANDLE_DIST"] else 5
            ),  # to prevent small hedge in all case min hedge = 5m
            node_type_field=parameters["INPUT_FIELD"],
        )

        feedback.pushInfo("Creating and formatting output")

        # # Delete useless fields
        field_del_list = list(set(itf_poly_layer.fields().names()) - input_fields)

        alg_name = "native:deletecolumn"
        params = {
            "INPUT": itf_poly_layer,
            "COLUMN": field_del_list,
            "OUTPUT": "TEMPORARY_OUTPUT",
        }
        itf_poly_layer = processing.run(alg_name, params)["OUTPUT"]

        # Fill the sinks
        # Poly
        (sink_polys, poly_id) = self.parameterAsSink(
            parameters,
            self.OUTPUT_POLY,
            context,
            itf_poly_layer.fields(),
            itf_poly_layer.wkbType(),
            itf_poly_layer.sourceCrs(),
        )

        features = itf_poly_layer.getFeatures()
        for feature in features:
            sink_polys.addFeature(feature, QgsFeatureSink.FastInsert)

        # Arc
        (sink_arcs, arc_id) = self.parameterAsSink(
            parameters,
            self.OUTPUT_ARC,
            context,
            itf_arc_layer.fields(),
            itf_arc_layer.wkbType(),
            itf_arc_layer.sourceCrs(),
        )

        features = itf_arc_layer.getFeatures()
        for feature in features:
            sink_arcs.addFeature(feature, QgsFeatureSink.FastInsert)

        # Node
        (sink_nodes, node_id) = self.parameterAsSink(
            parameters,
            self.OUTPUT_NODE,
            context,
            itf_node_layer.fields(),
            itf_node_layer.wkbType(),
            itf_node_layer.sourceCrs(),
        )

        features = itf_node_layer.getFeatures()
        for feature in features:
            sink_nodes.addFeature(feature, QgsFeatureSink.FastInsert)

        # Set progress
        feedback.setProgress(100)
        # Check for cancellation
        if feedback.isCanceled():
            return {}

        return {"OUTPUT_POLY": poly_id, "OUTPUT_ARC": arc_id, "OUTPUT_NODE": node_id}

    def icon(self):
        """
        Should return a QIcon which is used for your provider inside
        the Processing toolbox.
        """
        return QIcon(":/plugins/hedge_tools/images/hedge_tools.png")

    def shortHelpString(self):
        """
        Returns a localised short help string for the algorithm.
        """
        return self.tr(
            "Modify the representation of the hedges by cutting \
                        them when there is a change in the adjacent interface (fields limit, ...). \
                        Offset tolerance allows deviation from the perpendicular between \
                        an hedge orientation and an interface limit direction. \
                        An offset value of 0 means only strictly perpendicular interfaces with respect \
                        to the hedge will cut it. An offset of 15 means that an interface \
                        whose orientation is 75 to 105° with respect to the hedge will cut it. \n\
                        The topology (pid, eid and vid) will be updated. But old identifiers will be kept.\n\
                        L nodes represent degree 2 nodes (one incoming and one outgoing arc) \
                        where the hedges are cut."
        )

    def name(self):
        """
        Returns the algorithm name, used for identifying the algorithm. This
        string should be fixed for the algorithm, and must not be localised.
        The name should be unique within each provider. Names should contain
        lowercase alphanumeric characters only and no spaces or other
        formatting characters.
        """
        return "splitbyinterface"

    def displayName(self):
        """
        Returns the translated algorithm name, which should be used for any
        user-visible display of the algorithm name.
        """
        return self.tr("Split by interface")

    def group(self):
        """
        Returns the name of the group this algorithm belongs to. This string
        should be localised.
        """
        return self.tr("2 - Data transformation")

    def groupId(self):
        """
        Returns the unique ID of the group this algorithm belongs to. This
        string should be fixed for the algorithm, and must not be localised.
        The group id should be unique within each provider. Group id should
        contain lowercase alphanumeric characters only and no spaces or other
        formatting characters.
        """
        return "datatransformation"

    def tr(self, string):
        return QCoreApplication.translate("Processing", string)

    def createInstance(self):
        return SplitByInterfaceAlgorithm()
