# -*- coding: utf-8 -*-

"""
/***************************************************************************
 Create subgraph
                                 A QGIS plugin
 Perform a connected components analysis on a linestring layer to regroup
 geometry that are connected in a multipolyline

 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2022-01-22
        copyright            : (C) 2022 by Dynafor
        email                : gabriel.marques@toulouse-inp.fr
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

__author__ = "Dynafor"
__date__ = "2022-01-22"
__copyright__ = "(C) 2022 by Dynafor"

# This will get replaced with a git SHA1 when you do a git archive

__revision__ = "$Format:%H$"

from qgis.PyQt.QtCore import QCoreApplication
from qgis.core import (
    QgsProcessing,
    QgsProcessingUtils,
    QgsFeatureSink,
    QgsProcessingAlgorithm,
    QgsProcessingParameterFeatureSource,
    QgsProcessingParameterFeatureSink,
)
from qgis.PyQt.QtGui import QIcon
import processing
import os

from hedge_tools import (
    resources,
)  # Only need in hedge_tools.py normaly but just to keep track of import

from hedge_tools.tools.vector import attribute_table as at
from hedge_tools.tools.vector import qgis_wrapper as qw
from hedge_tools.tools.vector import utils as vutils


class SubgraphCreationAlgorithm(QgsProcessingAlgorithm):
    """
    Perform a connected components analysis to regroup connected arc as multipolyline

    Parameters
    ----------
    INPUT_ARC (QgisObject : QgsVectorLayer) : Linestring layer path. Contains arc (polyline) hedges
    INPUT_NODE (QgisObject : QgsVectorLayer) : Node layer path. Contains nodes to delimit the hedges

    Return
    ------
    NETWORK_EDGES (QgisObject : QgsVectorLayer) : MultiLineString : Subgraphs layer
    NETWORK_VERTICES (QgisObject : QgsVectorLayer) : Point
    """

    # Constants used to refer to parameters and outputs. They will be
    # used when calling the algorithm from another algorithm, or when
    # calling from the QGIS console.
    INPUT_ARC = "INPUT_ARC"
    INPUT_NODE = "INPUT_NODE"
    NETWORK_EDGES = "NETWORK_EDGES"
    NETWORK_VERTICES = "NETWORK_VERTICES"

    def initAlgorithm(self, config):
        """
        Here we define the inputs and output of the algorithm, along
        with some other properties.
        """
        # We add the input vector lines features source.
        self.addParameter(
            QgsProcessingParameterFeatureSource(
                self.INPUT_ARC,
                self.tr("Arcs vector layer"),
                [QgsProcessing.TypeVectorLine],
            )
        )

        # We add the input vector nodes features source.
        self.addParameter(
            QgsProcessingParameterFeatureSource(
                self.INPUT_NODE,
                self.tr("Nodes vector layer"),
                [QgsProcessing.TypeVectorPoint],
            )
        )

        # Output
        self.addParameter(
            QgsProcessingParameterFeatureSink(
                name=self.NETWORK_EDGES,
                description=self.tr("Subgraphs edges layer"),
            )
        )

        self.addParameter(
            QgsProcessingParameterFeatureSink(
                name=self.NETWORK_VERTICES,
                description=self.tr("Subgraphs vertices layer"),
            )
        )

    def processAlgorithm(self, parameters, context, feedback):
        """
        Here is where the processing itself takes place.
        """
        arc_layer = self.parameterAsVectorLayer(parameters, self.INPUT_ARC, context)
        node_layer = self.parameterAsVectorLayer(parameters, self.INPUT_NODE, context)

        feedback.pushInfo("Starting processing")

        # Check for cancellation
        if feedback.isCanceled():
            return {}

        subgraph_arc, subgraph_nodes = self.create_subgraph(
            arc_layer, node_layer, feedback
        )

        feedback.setProgress(75)
        feedback.pushInfo("Formating output")

        (sink_edge, edge_id) = self.parameterAsSink(
            parameters,
            self.NETWORK_EDGES,
            context,
            subgraph_arc.fields(),
            subgraph_arc.wkbType(),
            subgraph_arc.sourceCrs(),
        )
        for feature in subgraph_arc.getFeatures():
            sink_edge.addFeature(feature, QgsFeatureSink.FastInsert)

        del subgraph_arc

        (sink_vertex, vertex_id) = self.parameterAsSink(
            parameters,
            self.NETWORK_VERTICES,
            context,
            subgraph_nodes.fields(),
            subgraph_nodes.wkbType(),
            subgraph_nodes.sourceCrs(),
        )
        for feature in subgraph_nodes.getFeatures():
            sink_vertex.addFeature(feature, QgsFeatureSink.FastInsert)

        del subgraph_nodes

        return {"NETWORK_EDGES": edge_id, "NETWORK_VERTICES": vertex_id}

    def postProcessAlgorithm(self, context, feedback):
        """
        Tasks done when processAlgorithm is finished
        """
        vutils.delete_processing_workspace()

        return {}

    def create_subgraph(self, line_layer, node_layer, feedback):
        """
        From a linestring layer perform connected components analysis and return subgraph
        A unique subgraph id will be added to the node layer for easier connectivity index computation

        Parameters
        ----------

        line_layer : QgsVectorLayer : LineString
        node_layer : QgsVectorLayer : Point

        Return
        ------
        subgraph_arc : QgsVectorLayer : MultiLineString
        subgraph_nodes : QgsVectorLayer : Point

        Note : Grass gis CC is used for easier/faster manipulation
        """
        uri = QgsProcessingUtils.generateTempFilename("subgraph_arc.gpkg")

        alg_name = "grass7:v.net.components"
        params = {
            "input": line_layer,
            "threshold": 0,
            "method": 0,
            "output": uri,
            "output_point": "TEMPORARY_OUTPUT",
        }
        subgraph_path = processing.run(alg_name, params)["output"]

        feedback.setProgress(50)

        subgraph_arc = qw.dissolve(subgraph_path, "comp", False)
        at.delete_fields(subgraph_arc, ["cat"])

        subgraph_nodes = qw.join_attributes_by_location(
            node_layer,
            subgraph_arc,
            predicate=[0],
            join_fields=["comp"],
            method=0,
            discard_nonmatching=False,
            prefix="",
        )
        return subgraph_arc, subgraph_nodes

    def icon(self):
        """
        Should return a QIcon which is used for your provider inside
        the Processing toolbox.
        """
        return QIcon(":/plugins/hedge_tools/images/hedge_tools.png")

    def shortHelpString(self):
        """
        Returns a localised short help string for the algorithm.
        """
        return self.tr(
            "This algorithm regroups connected arcs into a multilinestring that represents a connected graph. \n\
                        A 'comp' field is added to both layers as a network ID to link nodes and edges.\n\
                        Both of the output layers can then be used to compute connectivity metrics."
        )

    def name(self):
        """
        Returns the algorithm name, used for identifying the algorithm. This
        string should be fixed for the algorithm, and must not be localised.
        The name should be unique within each provider. Names should contain
        lowercase alphanumeric characters only and no spaces or other
        formatting characters.
        """
        return "subgraphcreation"

    def displayName(self):
        """
        Returns the translated algorithm name, which should be used for any
        user-visible display of the algorithm name.
        """
        return self.tr("1 - Subgraphs creation")

    def group(self):
        """
        Returns the name of the group this algorithm belongs to. This string
        should be localised.
        """
        return self.tr("6 - Network level")

    def groupId(self):
        """
        Returns the unique ID of the group this algorithm belongs to. This
        string should be fixed for the algorithm, and must not be localised.
        The group id should be unique within each provider. Group id should
        contain lowercase alphanumeric characters only and no spaces or other
        formatting characters.
        """
        return "hedgenetwork"

    def tr(self, string):
        return QCoreApplication.translate("Processing", string)

    def createInstance(self):
        return SubgraphCreationAlgorithm()
