# -*- coding: utf-8 -*-

"""
/***************************************************************************
 Width
                                 A QGIS plugin
 Compute the width of a the hedge spolygon by creating perpendicular transects.
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2022-01-22
        copyright            : (C) 2022 by Dynafor
        email                : gabriel.marques@toulouse-inp.fr
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

__author__ = "Dynafor"
__date__ = "2022-01-22"
__copyright__ = "(C) 2022 by Dynafor"

# This will get replaced with a git SHA1 when you do a git archive

__revision__ = "$Format:%H$"

from qgis.PyQt.QtCore import QCoreApplication, QVariant
from qgis.core import (
    QgsProcessing,
    QgsProcessingUtils,
    QgsFeatureSink,
    QgsProcessingAlgorithm,
    QgsProcessingParameterFeatureSource,
    QgsProcessingParameterFeatureSink,
    QgsProcessingParameterNumber,
    QgsProcessingParameterEnum,
    QgsFeedback,
    QgsFeatureRequest,
    QgsVectorLayer,
    QgsFeature,
    QgsGeometry,
    QgsWkbTypes,
    NULL,
)

from qgis.PyQt.QtGui import QIcon
from hedge_tools import (
    resources,
)  # Only need in hedge_tools.py normaly but just to keep track of import

from hedge_tools.tools.vector import utils
from hedge_tools.tools.vector import attribute_table as at
from hedge_tools.tools.vector import geometry as g
from hedge_tools.tools.vector import qgis_wrapper as qw

from statistics import median, stdev
import numpy as np


class WidthAlgorithm(QgsProcessingAlgorithm):
    """
    Compute width of polygonal features who have a median axis.

    Parameters
    ---
    POLYGONS (QgisObject : QgsVectorLayer) : Polygon layer path. Contains hedges
    ARCS (QgisObject : QgsVectorLayer) : Linestring layer path. Contains arc (polyline) hedges
    METHOD : Either compute width using transect or estimates width using area to length ratio.
    DISTANCE (float) : Distance between transect. A smaller distance will return
                       a more accurate width but will be slower to process.


    Return
    ---
    OUTPUT_POLY (QgisObject : QgsVectorLayer) : Polygon : Polygons layer with 2 width attributes.
    """

    # Constants used to refer to parameters and outputs. They will be
    # used when calling the algorithm from another algorithm, or when
    # calling from the QGIS console.
    POLYGONS = "POLYGONS"
    ARCS = "ARCS"
    METHOD = "METHOD"
    DISTANCE = "DISTANCE"
    OUTPUT_POLY = "OUTPUT_POLY"

    def initAlgorithm(self, config):
        """
        Here we define the inputs and output of the algorithm, along
        with some other properties.
        """

        # We add the input vector polygons features source.
        self.addParameter(
            QgsProcessingParameterFeatureSource(
                self.POLYGONS,
                self.tr("Polygons vector layer"),
                [QgsProcessing.TypeVectorPolygon],
            )
        )

        # We add the input vector lines features source.
        self.addParameter(
            QgsProcessingParameterFeatureSource(
                self.ARCS, self.tr("Arcs vector layer"), [QgsProcessing.TypeVectorLine]
            )
        )

        # Which width to compute
        self.addParameter(
            QgsProcessingParameterEnum(
                name=self.METHOD,
                description=self.tr("Width method"),
                options=["Transect", "Area/Length ratio"],
                optional=False,
                allowMultiple=True,
            )
        )

        # transect distance separation
        self.addParameter(
            QgsProcessingParameterNumber(
                name=self.DISTANCE,
                description=self.tr("Distance separation for transects (meters)"),
                type=QgsProcessingParameterNumber.Integer,
                defaultValue=3,
                optional=False,
                minValue=0,
                maxValue=20,
            )
        )

    def processAlgorithm(self, parameters, context, feedback):
        """
        Here is where the processing itself takes place.
        """
        poly_layer = self.parameterAsVectorLayer(parameters, self.POLYGONS, context)
        arc_layer = self.parameterAsVectorLayer(parameters, self.ARCS, context)
        method = self.parameterAsEnums(parameters, self.METHOD, context)
        distance = self.parameterAsInt(parameters, self.DISTANCE, context)

        if len(method) == 2:
            percent = 100
        else:
            percent = 50

        feedback.pushInfo("Starting processing")

        # Check for cancellation
        if feedback.isCanceled():
            return {}

        feedback.pushInfo("Computing width with transect")
        if 0 in method:
            poly_layer = self.width_transect(
                poly_layer, arc_layer, feedback, percent, distance
            )

        feedback.pushInfo("Computing width with shape")
        if 1 in method:
            poly_layer = self.width_ratio(poly_layer, arc_layer, feedback, percent)

        return {"OUTPUT_POLY": parameters[self.POLYGONS]}

    def postProcessAlgorithm(self, context, feedback):
        """
        Tasks done when processAlgorithm is finished
        """
        utils.delete_processing_workspace()

        return {}

    def arc_simplification(
        self, poly_layer: QgsVectorLayer, arc_layer: QgsVectorLayer, epsilon: int = 5
    ) -> QgsVectorLayer:
        """
        Simplify arcs to avoid overestimating narrow
        part of polygons where there is more vertices

        First perform a douglas constrained simplification
        If an arc is outside his polygon (part too narrow)
        or if there is no solution found it'll use a standart simplification
        with a small epsilon.

        Parameters
        ----------

        poly_layer:
        arc_layer:
        node_layer:
        epislon:
            Shift tolerance

        Returns
        -------
        arc_layer:
            As usual not mandatory but return for lisibility
        """
        geom_map = {}
        for arc in arc_layer.getFeatures():
            geometry = arc.geometry()
            line = geometry.constGet()

            expression = f"pid = {arc['pid']}"
            request = QgsFeatureRequest().setFilterExpression(expression)
            feature = next(poly_layer.getFeatures(request))
            polygon = feature.geometry()

            try:
                result = g.constrained_douglas_peucker(polygon, line, epsilon)
                result_geom = QgsGeometry().fromPolyline(result)
                # Add new geom to geom_map
                geom_map[arc.id()] = result_geom
            except RecursionError:
                result_geom = geometry.simplify(0.25)
                geom_map[arc.id()] = result_geom

        arc_layer.dataProvider().changeGeometryValues(geom_map)

        return arc_layer

    def correct_intersection(
        self, transect: QgsGeometry, polygon: QgsGeometry
    ) -> QgsGeometry:
        """
        From a line/polygon intersection get the part
        that is intersecting with the center of the transect.
        Allow to filter a geometry collection when
        there is multiple interseciton between the geometries

        A while loop is used in case the transect center
        is outside the polygon (i.e median axis is outside)
        allowing to buffer incrementally the center point.
        This allow the median axis to be outside his polygon to some extents

        Parameters
        ----------
        transect:
        polygon:

        Returns
        -------
        results:
        """
        size = 0.1
        center = transect.interpolate(100).buffer(size, 5)
        gc = transect.intersection(polygon).asGeometryCollection()

        results = None
        while True:
            for geometry in gc:
                if geometry.intersects(center):
                    results = geometry
            if results == None:
                size += 0.4
                center = transect.interpolate(100).buffer(size, 5)

            if size > 4 or results != None:
                break

        return results

    def clip_transect(
        self, transect_layer: QgsVectorLayer, poly_layer: QgsVectorLayer
    ) -> QgsVectorLayer:
        """
        Clip tansect by the closest polygon from the center of the transect.

        Parameters
        ----------
        transect_layer:
            Transect centered on the median axis of the polygons
        poly_layer:
            Polygons associated with the median axis which created the transects

        Returns
        -------
        clipped:
            Transects clipped
        """
        clip_uri = QgsProcessingUtils.generateTempFilename("clip_layer.gpkg")
        clipped = utils.create_layer(
            transect_layer, copy_field=True, data_provider="ogr", path=clip_uri
        )
        feat_list = []

        for feat in poly_layer.getFeatures():
            polygon = feat.geometry()
            pid = feat["pid"]
            expression = f"pid = {pid}"
            request = QgsFeatureRequest().setFilterExpression(expression)

            for feature in transect_layer.getFeatures(request):
                transect = feature.geometry()
                # Buffer to intersect center and polygons in case arc is a outside at a narrow gap
                geometry = self.correct_intersection(transect, polygon)
                # geom is None when arc is too far outside polygon
                # Ignore transect from first and last vertex
                if geometry != None and geometry.wkbType() != QgsWkbTypes.Point:
                    f = QgsFeature()
                    f.setGeometry(geometry)
                    feat_list.append(f)

        clipped.dataProvider().addFeatures(feat_list)

        return clipped

    def width_average(self, lengths: list[float]) -> float:
        """
        From a list of length, exclude the outliers
        using the interquartile range method and compute the average

        Parameters
        ----------
        lengths:
            Length of each transect

        Returns
        -------
        average:
            Average width of a polygon
        """

        # Get Q1 and Q3
        q1, q3 = np.percentile(lengths, 25), np.percentile(lengths, 75)
        iqr = q3 - q1

        # Get outliers thresh
        thresh = iqr * 1.5
        lower, upper = q1 - thresh, q3 + thresh

        # Remove outliers
        # Special case of small polygons where there is not enough transect
        if q1 != q3:
            lengths = [l for l in lengths if l > lower and l < upper]

        # Compute average
        if len(lengths) != 0:
            average = round(sum(lengths) / len(lengths), 2)
        else:
            average = NULL

        return average

    def width_statistics(
        self, poly_layer: QgsVectorLayer, clip_layer: QgsVectorLayer
    ) -> QgsVectorLayer:
        """
        Compute width average, median and standard deviation for each polygon
        with the help of transects

        Parameters
        ----------
        poly_layer:
        clip_layer:
            Transect clipped by their polygon

        Returns
        -------
        poly_layer:
        """
        # Create output fields
        fields = [
            ("width_med", QVariant.Double),
            ("width_avg", QVariant.Double),
            ("width_std", QVariant.Double),
        ]
        indexes = at.create_fields(poly_layer, fields)

        # Compute statistics
        attr_map = {}
        for feature in poly_layer.getFeatures():
            count, transects = g.get_clementini(clip_layer, feature.geometry())
            if count != 0:
                lengths = [transect.geometry().length() for transect in transects]
                width_med = round(median(lengths), 2)
                width_avg = self.width_average(lengths)
                width_std = round(stdev(lengths), 2) if count >= 2 else NULL
            attr_map[feature.id()] = {
                indexes[0]: width_med,
                indexes[1]: width_avg,
                indexes[2]: width_std,
            }
        poly_layer.dataProvider().changeAttributeValues(attr_map)

        return poly_layer

    def width_transect(
        self,
        poly_layer: QgsVectorLayer,
        arc_layer: QgsVectorLayer,
        feedback: QgsFeedback,
        percent: int = 100,
        distance: int = 3,
    ) -> QgsVectorLayer:
        """
        Estimation of polygons width by computing descriptive statistics
        of regularly spaced transect.

        Parameters
        ----------
        poly_layer:
        arc_layer:
        feedback:
            Used for progress bar
        percent:
            Used for progress bar, 100 if one method, 50 is 2 methods
        distance:
            Space between 2 transects
            Space could be lower if there arc is curved and with high vertex density

        Returns
        -------
        poly_layer:
            As usual not mandatory but return for lisibility
        """
        arc_uri = QgsProcessingUtils.generateTempFilename("arc_layer.gpkg")
        arc_layer = utils.create_layer(
            arc_layer,
            copy_feat=True,
            copy_field=True,
            data_provider="ogr",
            path=arc_uri,
        )
        arc_layer = self.arc_simplification(poly_layer, arc_layer)

        # Set progress
        feedback.setProgress(0.2 * percent)
        # Check for cancellation
        if feedback.isCanceled():
            return {}

        arc_layer = qw.densify_by_interval(arc_layer, distance)

        # Set progress
        feedback.setProgress(0.4 * percent)
        # Check for cancellation
        if feedback.isCanceled():
            return {}

        transect_layer = qw.transect(arc_layer, 200, 90, 2)

        # Set progress
        feedback.setProgress(0.6 * percent)
        # Check for cancellation
        if feedback.isCanceled():
            return {}

        clip_layer = self.clip_transect(transect_layer, poly_layer)

        # Set progress
        feedback.setProgress(0.8 * percent)
        # Check for cancellation
        if feedback.isCanceled():
            return {}

        poly_layer = self.width_statistics(poly_layer, clip_layer)

        # Set progress
        feedback.setProgress(1 * percent)
        # Check for cancellation
        if feedback.isCanceled():
            return {}

        return poly_layer

    def width_ratio(
        self,
        poly_layer: QgsVectorLayer,
        arc_layer: QgsVectorLayer,
        feedback: QgsFeedback,
        percent: int = 100,
    ) -> QgsVectorLayer:
        """
        Estimation of polygons width by polygon area divided by arc length

        Parameters
        ----------
        poly_layer:
        arc_layer:
        feedback:
            Used for progress bar
        percent:
            Used for progress bar, 100 if one method, 50 is 2 methods

        Returns
        -------
        poly_layer:
            As usual not mandatory but return for lisibility
        """
        if percent == 50:
            start = 50
        else:
            start = 0
        # Init output field
        fields_list = [("width_ratio", QVariant.Double)]
        idx_ratio = at.create_fields(poly_layer, fields_list)[0]

        # Init attributes map
        attr_map = {}

        # Building request to iterate in the same order over two layers
        request_poly = QgsFeatureRequest()
        clause_poly = QgsFeatureRequest.OrderByClause("pid", ascending=True)
        order_poly = QgsFeatureRequest.OrderBy([clause_poly])
        request_poly.setOrderBy(order_poly)
        polygons = poly_layer.getFeatures(request_poly)

        request_arc = QgsFeatureRequest()
        clause_arc = QgsFeatureRequest.OrderByClause("pid", ascending=True)
        order_arc = QgsFeatureRequest.OrderBy([clause_arc])
        request_arc.setOrderBy(order_arc)
        arcs = arc_layer.getFeatures(request_arc)

        total = poly_layer.featureCount()

        for current, (poly, arc) in enumerate(zip(polygons, arcs)):
            area = poly.geometry().area()
            arc_lgth = arc.geometry().length()

            if arc_lgth != 0:
                width = round(area / arc_lgth, 2)
            else:
                width = NULL

            attr_map[poly.id()] = {idx_ratio: width}

            # Set progress
            feedback.setProgress(start + int(current / total) * percent)
            # Check for cancellation
            if feedback.isCanceled():
                return {}

        poly_layer.dataProvider().changeAttributeValues(attr_map)

        return poly_layer

    def icon(self):
        """
        Should return a QIcon which is used for your provider inside
        the Processing toolbox.
        """
        return QIcon(":/plugins/hedge_tools/images/hedge_tools.png")

    def shortHelpString(self):
        """
        Returns a localised short help string for the algorithm.
        """
        return self.tr(
            "If the transect method is selected, this algorithm computes \
                        the median, average, and standard deviation of the hedges' width. \
                        The average excludes potential outliers with the interquartile range method. \n\
                        If the area/length method is selected, the computation will be faster but less accurate. \n\
                        Results are stored in fields of the polygon layer \
                        named respectively : width_med width_avg, width_std or width_ratio."
        )

    def name(self):
        """
        Returns the algorithm name, used for identifying the algorithm. This
        string should be fixed for the algorithm, and must not be localised.
        The name should be unique within each provider. Names should contain
        lowercase alphanumeric characters only and no spaces or other
        formatting characters.
        """
        return "computewidth"

    def displayName(self):
        """
        Returns the translated algorithm name, which should be used for any
        user-visible display of the algorithm name.
        """
        return self.tr("Width")

    def group(self):
        """
        Returns the name of the group this algorithm belongs to. This string
        should be localised.
        """
        return self.tr("3 - Hedges level: morphology")

    def groupId(self):
        """
        Returns the unique ID of the group this algorithm belongs to. This
        string should be fixed for the algorithm, and must not be localised.
        The group id should be unique within each provider. Group id should
        contain lowercase alphanumeric characters only and no spaces or other
        formatting characters.
        """
        return "hedgesmorphology"

    def tr(self, string):
        return QCoreApplication.translate("Processing", string)

    def createInstance(self):
        return WidthAlgorithm()
