# -*- coding: utf-8 -*-

"""
/***************************************************************************
 DiffForestHedge
                                 A QGIS plugin
 Allow to split forest from hedge and others tree area
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2022-01-11
        copyright            : (C) 2022 by Dynafor
        email                : gabriel.marques@toulouse-inp.fr
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

__author__ = "Dynafor"
__date__ = "2022-01-11"
__copyright__ = "(C) 2022 by Dynafor"

# This will get replaced with a git SHA1 when you do a git archive

__revision__ = "$Format:%H$"

from qgis.PyQt.QtCore import QCoreApplication, QVariant
from qgis.core import (
    QgsProcessing,
    QgsProcessingUtils,
    QgsFeatureSink,
    QgsProcessingAlgorithm,
    QgsProcessingParameterFeatureSource,
    QgsProcessingParameterFeatureSink,
    QgsProcessingParameterNumber,
    QgsFeatureRequest,
    QgsFeature,
    QgsGeometry,
    QgsVectorLayer,
    QgsWkbTypes,
)
from qgis.PyQt.QtGui import QIcon
from hedge_tools import (
    resources,
)  # Only need in hedge_tools.py normaly but just to keep track of import

from hedge_tools.tools.vector import qgis_wrapper as qw
from hedge_tools.tools.vector import geometry as g
from hedge_tools.tools.vector import utils


class PreProcessDiffForestHedgeAlgorithm(QgsProcessingAlgorithm):
    """
    Performs preprocess step to allow CWA classification and further hedge tools stpe to work

    It'll first delete small geometries with area below an area threshold
    Then it'll perform a douglas peucker simplification to remove some vertex
    as "raster" representation will slow down algorithms consequently
    It'll also remove holes in polygons below an area threshold
    Finally a geometry correction steps will be performed

    Parameters
    ---
    INPUT (QgisObject : QgsVectorLayer) : Layer input from users.
    DELETE (int) : Threshold for small geometry deletion
    SIMPL (int) : Shift tolerance for simplification.
    RM_HOLE (int) : Threshold for filling holes.

    Return
    ---
    OUTPUT (QgisObject : QgsVectorLayer) : Fixed layer.
    """

    # Constants used to refer to parameters and outputs. They will be
    # used when calling the algorithm from another algorithm, or when
    # calling from the QGIS console.
    INPUT = "INPUT"
    DELETE = "DELETE"
    TOLERANCE = "TOLERANCE"
    RM_HOLE = "RM_HOLE"
    OUTPUT = "OUTPUT"

    def initAlgorithm(self, config):
        """
        Here we define the inputs and output of the algorithm, along
        with some other properties.
        """

        # We add the input vector features source. It can have any kind of
        # geometry.
        self.addParameter(
            QgsProcessingParameterFeatureSource(
                self.INPUT,
                self.tr("Tree cover layer"),
                [QgsProcessing.TypeVectorPolygon],
            )
        )

        # We add the buffer distance to smooth the border of the polygone network
        self.addParameter(
            QgsProcessingParameterNumber(
                name=self.DELETE,
                description=self.tr(
                    "Area threshold for geometry deletion (square meters)"
                ),
                type=QgsProcessingParameterNumber.Integer,
                defaultValue=20,
                optional=False,
                minValue=0,
            )
        )

        # We add the area threshold for the spatial selection
        self.addParameter(
            QgsProcessingParameterNumber(
                name=self.TOLERANCE,
                description=self.tr("Shift tolerance for simplification (meters)"),
                type=QgsProcessingParameterNumber.Double,
                defaultValue=0.25,
                optional=False,
                minValue=0,
            )
        )

        # We add the area threshold for the spatial selection
        self.addParameter(
            QgsProcessingParameterNumber(
                name=self.RM_HOLE,
                description=self.tr(
                    "Area threshold for filling polygon holes (square meters)"
                ),
                type=QgsProcessingParameterNumber.Integer,
                defaultValue=50,
                optional=False,
                minValue=0,
            )
        )

        self.addParameter(
            QgsProcessingParameterFeatureSink(
                self.OUTPUT, self.tr("Preprocessed tree cover layer (output)")
            )
        )

    def __init__(self):
        super().__init__()
        self.step_per_alg = 100 // 5
        self.value = 0

    def processAlgorithm(self, parameters, context, feedback):
        """
        Here is where the processing itself takes place.
        """
        # Init
        input_layer = self.parameterAsVectorLayer(parameters, self.INPUT, context)
        max_area = self.parameterAsInt(parameters, self.DELETE, context)
        tolerance = self.parameterAsDouble(parameters, self.TOLERANCE, context)
        rm_hole = self.parameterAsInt(parameters, self.RM_HOLE, context)

        self.progress_bar(feedback)
        fix_layer = qw.fix_geometries(input_layer)

        feedback.pushInfo("Deleting small geometries")
        self.progress_bar(feedback)

        filter_layer = g.delete_geometries(fix_layer, max_area=max_area)

        feedback.pushInfo("Layer simplification")
        self.progress_bar(feedback)

        simpl_layer = qw.simplification(filter_layer, tolerance=tolerance)

        feedback.pushInfo("Removing holes inside polygons")
        self.progress_bar(feedback)

        fill_layer = qw.remove_holes(simpl_layer, rm_hole)

        self.progress_bar(feedback)

        output = qw.fix_geometries(fill_layer)

        self.progress_bar(feedback)

        # Forest output
        (sink, output_id) = self.parameterAsSink(
            parameters,
            self.OUTPUT,
            context,
            output.fields(),
            output.wkbType(),
            output.sourceCrs(),
        )
        for feature in output.getFeatures():
            sink.addFeature(feature, QgsFeatureSink.FastInsert)

        return {"OUTPUT": output_id}

    def progress_bar(self, feedback):
        """
        Incrementation of progress bar and cancellation
        """
        # Set progress
        feedback.setProgress(self.value)
        self.value += self.step_per_alg
        # Check for cancellation
        if feedback.isCanceled():
            return {}

    def icon(self):
        """
        Should return a QIcon which is used for your provider inside
        the Processing toolbox.
        """
        return QIcon(":/plugins/hedge_tools/images/hedge_tools.png")

    def shortHelpString(self):
        """
        Returns a localised short help string for the algorithm.
        """
        return self.tr(
            "A preprocessing step is required before to categorize \
                        wooded area. It consists in: \n \
                        - eliminating small polygons with an area lower than \
                        the threshold \n \
                        - simplifying geometry with a Douglas-Peucker \
                        algorithm to smooth the vector borders obtained from \
                        a raster data \n \
                        - removing holes in polygons with an area lower than \
                        the threshold \n \
                        - correcting final geometry"
        )

    def name(self):
        """
        Returns the algorithm name, used for identifying the algorithm. This
        string should be fixed for the algorithm, and must not be localised.
        The name should be unique within each provider. Names should contain
        lowercase alphanumeric characters only and no spaces or other
        formatting characters.
        """
        return "preprocesscwa"

    def displayName(self):
        """
        Returns the translated algorithm name, which should be used for any
        user-visible display of the algorithm name.
        """
        return self.tr("4 - Preprocessing: categorize wooded area")

    def group(self):
        """
        Returns the name of the group this algorithm belongs to. This string
        should be localised.
        """
        return self.tr("0 - Extraction [optional]")

    def groupId(self):
        """
        Returns the unique ID of the group this algorithm belongs to. This
        string should be fixed for the algorithm, and must not be localised.
        The group id should be unique within each provider. Group id should
        contain lowercase alphanumeric characters only and no spaces or other
        formatting characters.
        """
        return "extraction"

    def tr(self, string):
        return QCoreApplication.translate("Processing", string)

    def createInstance(self):
        return PreProcessDiffForestHedgeAlgorithm()
