# -*- coding: utf-8 -*-

"""
/***************************************************************************
 Relative orientation
                                 A QGIS plugin
 Compute hedge orientation inside the main slope.
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2022-01-22
        copyright            : (C) 2022 by Dynafor
        email                : gabriel.marques@toulouse-inp.fr
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

__author__ = "Dynafor"
__date__ = "2022-01-22"
__copyright__ = "(C) 2022 by Dynafor"

# This will get replaced with a git SHA1 when you do a git archive

__revision__ = "$Format:%H$"

from qgis.PyQt.QtCore import QCoreApplication, QVariant
from qgis.core import (
    QgsProcessing,
    QgsProcessingUtils,
    QgsFeatureSink,
    QgsProcessingAlgorithm,
    QgsProcessingParameterFeatureSource,
    QgsProcessingParameterFeatureSink,
    QgsProcessingParameterNumber,
    QgsProcessingParameterRasterLayer,
    QgsProcessingParameterEnum,
    QgsProcessingParameterField,
    NULL,
    QgsVectorLayer,
    QgsFeature,
)
import processing
from qgis.PyQt.QtGui import QIcon
from hedge_tools import (
    resources,
)  # Only need in hedge_tools.py normaly but just to keep track of import

from hedge_tools.tools.classes import class_hedge as h
from hedge_tools.tools.vector import utils
from hedge_tools.tools.vector import attribute_table as at
from hedge_tools.tools.raster import wrapper as rw

from typing import Union  # until python 3.10 we need to use that instead of |

import os


class RelativeOrientationAlgorithm(QgsProcessingAlgorithm):
    """
    Extract the relative orientation of the hedge inside the main slope.
    Hhedge can either be perpendicular, parallel or diagonal to the main slope.
    If there is no slope the algorithm return  the no slope

    Parameters
    ---
    INPUT_POLY (QgisObject : QgsVectorLayer) : Polygon layer path. Contains hedges
    INPUT_ARC (QgisObject : QgsVectorLayer) : Linestring layer path. Contains arc (polyline) hedges
    INPUT_NODE (QgisObject : QgsVectorLayer) : Node layer path. Contains nodes to delimit the hedges
    INPUT_MNT (QgisObject : QgsRasterLayer) : MNT path.
    INPUT_BUFFER (int) : Buffer size to determine main slope
    INPUT_BOUND (int) : Value determining lower bound and upper bound to consider
    the hedges relative position (for perpendicular hedge it will be
    90 - INPUT_BOUND < abs(slope_exposition - hedge_orientation) < 90 + INPUT_BOUND)
    INPUT_SLOPE (int) : Maximum slope for an hedge to be considered on flat ground

    Return
    ---
    OUTPUT_POLY (QgisObject : QgsVectorLayer) : Polygon : Polygon layer with a topographic position attribute.
    """

    # Constants used to refer to parameters and outputs. They will be
    # used when calling the algorithm from another algorithm, or when
    # calling from the QGIS console.
    POLYGON = "POLYGON"
    ARC = "ARC"
    NODE = "NODE"
    DEM = "DEM"
    RADIUS = "RADIUS"
    BOUND = "BOUND"
    MAX_SLOPE = "MAX_SLOPE"
    USE_GEOMORPHON = "USE_GEOMORPHON"
    GEOMORPHON_FIELD = "GEOMORPHON_FIELD"
    SEARCH_DIST = "SEARCH_DIST"
    SKIP_DIST = "SKIP_DIST"
    OUTPUT_POLY = "OUTPUT_POLY"

    def initAlgorithm(self, config):
        """
        Here we define the inputs and output of the algorithm, along
        with some other properties.
        """

        # We add the input vector polygons features source.
        self.addParameter(
            QgsProcessingParameterFeatureSource(
                self.POLYGON,
                self.tr("Polygons vector layer"),
                [QgsProcessing.TypeVectorPolygon],
            )
        )

        # We add the input vector lines features source.
        self.addParameter(
            QgsProcessingParameterFeatureSource(
                self.ARC, self.tr("Arcs vector layer"), [QgsProcessing.TypeVectorLine]
            )
        )

        # We add the input vector nodes features source.
        self.addParameter(
            QgsProcessingParameterFeatureSource(
                self.NODE,
                self.tr("Nodes vector layer"),
                [QgsProcessing.TypeVectorPoint],
            )
        )

        # We add the input vector mnt
        self.addParameter(
            QgsProcessingParameterRasterLayer(
                self.DEM, self.tr("Elevation raster (DEM)"), [QgsProcessing.TypeRaster]
            )
        )

        self.addParameter(
            QgsProcessingParameterNumber(
                name=self.RADIUS,
                description=self.tr(
                    "Kernel radius (must be odd) of the median filter (pixels)"
                ),
                type=QgsProcessingParameterNumber.Integer,
                defaultValue=7,
                optional=False,
                minValue=3,
            )
        )

        self.addParameter(
            QgsProcessingParameterNumber(
                name=self.BOUND,
                description=self.tr("Offset tolerance"),
                type=QgsProcessingParameterNumber.Integer,
                defaultValue=15,
                optional=False,
                minValue=5,
                maxValue=44,
            )
        )

        self.addParameter(
            QgsProcessingParameterNumber(
                name=self.MAX_SLOPE,
                description=self.tr("Maximum slope (%) to be considered as flat"),
                type=QgsProcessingParameterNumber.Integer,
                defaultValue=3,
                optional=False,
                minValue=0,
                maxValue=200,
            )
        )

        # Enum to ask for geomorhpon use and if we use a field or compute it
        self.addParameter(
            QgsProcessingParameterEnum(
                name=self.USE_GEOMORPHON,
                description=self.tr("Use topographic position for no slope category"),
                options=["Compute topographic position", "Use topographic field", "No"],
                optional=False,
                allowMultiple=False,
                defaultValue=0,
            )
        )

        # Either use an already calculated geomorphon or compute it
        self.addParameter(
            QgsProcessingParameterField(
                self.GEOMORPHON_FIELD,
                self.tr("Topographic position field"),
                type=QgsProcessingParameterField.String,
                parentLayerParameterName="POLYGON",
                optional=True,
            )
        )

        # Geomorphon parameter if no field used
        self.addParameter(
            QgsProcessingParameterNumber(
                name=self.SEARCH_DIST,
                description=self.tr("Search distance for terrain shapes (meters)"),
                type=QgsProcessingParameterNumber.Integer,
                defaultValue=100,
                optional=True,
                minValue=0,
                maxValue=200,
            )
        )

        self.addParameter(
            QgsProcessingParameterNumber(
                name=self.SKIP_DIST,
                description=self.tr("Skip distance for terrain shapes (meters)"),
                type=QgsProcessingParameterNumber.Integer,
                defaultValue=0,
                optional=True,
                minValue=0,
                maxValue=200,
            )
        )

    def __init__(self):
        super().__init__()

        self.shape_string_to_int = {
            "Flat": 1,
            "Peak": 2,
            "Ridge": 3,
            "Shoulder": 4,
            "Spur": 5,
            "Slope": 6,
            "Hollow": 7,
            "Footslope": 8,
            "Valley": 9,
            "Pit": 10,
            "No_data": 255,
        }

    def processAlgorithm(self, parameters, context, feedback):
        """
        Here is where the processing itself takes place.
        """
        poly_layer = self.parameterAsVectorLayer(parameters, self.POLYGON, context)
        arc_layer = self.parameterAsVectorLayer(parameters, self.ARC, context)
        node_layer = self.parameterAsVectorLayer(parameters, self.NODE, context)
        dem = self.parameterAsRasterLayer(parameters, self.DEM, context)
        radius = self.parameterAsInt(parameters, self.RADIUS, context)
        bound = self.parameterAsInt(parameters, self.BOUND, context)
        max_slope = self.parameterAsInt(parameters, self.MAX_SLOPE, context)
        use_geomorphon = self.parameterAsEnum(parameters, self.USE_GEOMORPHON, context)
        if use_geomorphon == 0:
            search_dist = self.parameterAsInt(parameters, self.SEARCH_DIST, context)
            skip_dist = self.parameterAsInt(parameters, self.SKIP_DIST, context)
            topo_field = "_majority"
        elif use_geomorphon == 1:
            topo_field = self.parameterAsFields(
                parameters, self.GEOMORPHON_FIELD, context
            )[0]
        else:
            topo_field = None

        # Defining progress bar step
        alg_number = 6
        step_per_alg = int(100 / alg_number)

        feedback.pushInfo("Starting processing")

        # Check for cancellation
        if feedback.isCanceled():
            return {}

        # Preprocessing
        if "direction" not in arc_layer.fields().names():
            feedback.pushInfo("Computing hedge orientation")
            # Qgis crash by loosing scope of arc_ayer if it is written
            # uri = QgsProcessingUtils.generateTempFilename("arc_layer.gpkg")
            arc_layer = utils.create_layer(
                arc_layer, copy_field=True, copy_feat=True, data_provider="memory"
            )
            self.compute_orientation(poly_layer, arc_layer, node_layer)

        # Set progress
        feedback.setProgress(step_per_alg)
        # Check for cancellation
        if feedback.isCanceled():
            return {}

        # Raster smoothing
        feedback.pushInfo("Raster smoothing")
        dem = rw.raster_neighbors(dem, 1, radius)

        # Set progress
        feedback.setProgress(step_per_alg * 2)
        # Check for cancellation
        if feedback.isCanceled():
            return {}

        feedback.pushInfo("Compute raster derivative")
        # Create raster derivative
        slope = rw.compute_slope(dem)
        aspect = rw.compute_aspect(dem)

        # Set progress
        feedback.setProgress(step_per_alg * 3)
        # Check for cancellation
        if feedback.isCanceled():
            return {}

        if use_geomorphon == 0:
            geomorphon = rw.compute_geomorphon(
                dem, search_dist, skip_dist, as_meters=True
            )

        # Set progress
        feedback.setProgress(step_per_alg * 4)
        # Check for cancellation
        if feedback.isCanceled():
            return {}

        feedback.pushInfo("Performs zonal statistics")

        poly_zonal = rw.zonal_statistics(poly_layer, slope, stats=[2])
        poly_zonal = rw.zonal_statistics(poly_zonal, aspect, stats=[3])
        if use_geomorphon == 0:
            poly_zonal = rw.zonal_statistics(poly_zonal, geomorphon, stats=[9])

        # Set progress
        feedback.setProgress(step_per_alg * 5)
        # Check for cancellation
        if feedback.isCanceled():
            return {}

        # Make table join
        fk_field = ["pid", "pid"]
        join_layer = [poly_zonal, arc_layer]
        if use_geomorphon == 0:
            join_field = [["_mean", "_median", "_majority"], ["direction"]]
        else:
            join_field = [["_mean", "_median"], ["direction"]]
        at.table_join(poly_layer, "pid", join_layer, fk_field, join_field)

        # Compute relative orientation
        poly_layer = self.compute_relative_orientation(
            poly_layer, use_geomorphon, bound, max_slope, topo_field
        )

        # Remove table join
        poly_layer.removeJoin(poly_zonal.id())
        poly_layer.removeJoin(arc_layer.id())

        # Set progress
        feedback.setProgress(step_per_alg * 6)
        # Check for cancellation
        if feedback.isCanceled():
            return {}

        return {"OUTPUT_POLY": parameters[self.POLYGON]}

    def postProcessAlgorithm(self, context, feedback):
        """
        Tasks done when processAlgorithm is finished
        """
        utils.delete_processing_workspace()

        return {}

    def compute_orientation(
        self,
        poly_layer: QgsVectorLayer,
        arc_layer: QgsVectorLayer,
        node_layer: QgsVectorLayer,
    ):
        """
        Duplicate arc layer and use hedge tools orientation computation
        to compute an orientation field on the duplicate

        Parameters
        ----------
        poly_layer : QgsVectorLayer
        arc_layer : QgsVectorLayer
        node_layer : QgsVectorLayer

        Return
        ------
        None
        """
        alg_name = "hedgetools:computeori"
        params = {
            "INPUT_POLY": poly_layer,
            "INPUT_ARC": arc_layer,
            "INPUT_NODE": node_layer,
        }
        processing.run(alg_name, params)

    def assess_orientation(
        self, feature: QgsFeature, field_idx: int, bound: int = 15
    ) -> dict[int, str]:
        """
        From a QgsFeature asses orientation of his geometry inside the slope
        using slope orientation and goemetry orientation difference

        Parameters
        ----------
        feature : QgsFeature
        field_idx : int
            index of the results field
        bound : int
            Allowed range for perpendicular and parallel assessment

        Return
        ------
        results : dict[int, str]
            As {idx: position}
        """
        results = {}
        # Halving median aspect to match hedge orientation
        slope_ori = (
            feature["_median"]
            if feature["_median"] <= 180
            else feature["_median"] - 180
        )

        if (
            abs(slope_ori - feature["direction"]) <= 0 + bound
            or abs(slope_ori - feature["direction"]) >= 180 - bound
        ):
            results[field_idx] = "Parallel"
        elif 90 - bound < abs(slope_ori - feature["direction"]) < 90 + bound:
            results[field_idx] = "Perpendicular"
        else:
            results[field_idx] = "Diagonal"

        return results

    def compute_relative_orientation(
        self,
        poly_layer: QgsVectorLayer,
        method: int = 0,
        bound: int = 15,
        max_slope: int = 3,
        topo_field: Union[str, None] = None,
    ):
        """
        Compute the orientation of a polygon inside the slope
        Orientation can be : diagonal, perpendicular or parellel
        When on ridge or valley or under a slope threshold polygon can be assigned as no slope.

        Parameters
        ----------
        poly_layer: QgsVectorLayer
        method : int
        topo_field : str or None depending oof the used method

        Return
        ------
        poly_layer: QgsVectorLayer

        """
        # Create output field
        idx = at.create_fields(poly_layer, [("slope_pos", QVariant.String)])[0]

        attr_map = {}
        for feature in poly_layer.getFeatures():
            if (
                feature["_mean"] != NULL
                and feature["_median"] != NULL
                and feature["direction"] != NULL
                and feature[topo_field] != NULL
            ):
                if method in [0, 1]:
                    shape = feature[topo_field]
                    if isinstance(shape, str):
                        shape = self.shape_string_to_int[shape]

                    if shape in [1, 2, 3, 9, 10] or feature["_mean"] <= max_slope:
                        attr_map[feature.id()] = {idx: "No slope"}
                    else:
                        attr_map[feature.id()] = self.assess_orientation(
                            feature, idx, bound
                        )
                elif method == 2:
                    if feature["_mean"] <= max_slope:
                        attr_map[feature.id()] = {idx: "No slope"}
                    else:
                        attr_map[feature.id()] = self.assess_orientation(
                            feature, idx, bound
                        )

        poly_layer.dataProvider().changeAttributeValues(attr_map)

        return poly_layer

    def icon(self):
        """
        Should return a QIcon which is used for your provider inside
        the Processing toolbox.
        """
        return QIcon(":/plugins/hedge_tools/images/hedge_tools.png")

    def shortHelpString(self):
        """
        Returns a localised short help string for the algorithm.
        """
        return self.tr(
            "Return hedges' position inside the main slope. \
                        It can either be perpendicular, parallel, diagonal, or no slope. \n\
                        The median filter allows reducing noise induced by special features (like steep river banks). \n\
                        Offset tolerance allows deviating from the perpendicular between \
                        the hedge orientation and a slope orientation. \n\
                        A minimum slope parameter determines if the hedge is on a slope or not, \
                        and topographic position can be used to automatically categorize hedges \
                        inside a ridge or a valley with no slope. Topographic position can either \
                        be computed in the algorithm or users can use the topo_pos field \
                        if they already have computed topographic_position. \n\
                        Note: The use of topographic position will increase the 'no slope' category. \
                        This will increase further as the skip distance increases. \
                        Recommended values for a 5 meters resolution DEM are to search 100 meters \
                        and skip either 0 or 50 meters."
        )

    def name(self):
        """
        Returns the algorithm name, used for identifying the algorithm. This
        string should be fixed for the algorithm, and must not be localised.
        The name should be unique within each provider. Names should contain
        lowercase alphanumeric characters only and no spaces or other
        formatting characters.
        """
        return "relativeorientation"

    def displayName(self):
        """
        Returns the translated algorithm name, which should be used for any
        user-visible display of the algorithm name.
        """
        return self.tr("Relative orientation inside the main slope")

    def group(self):
        """
        Returns the name of the group this algorithm belongs to. This string
        should be localised.
        """
        return self.tr("5 - Context level: geographic")

    def groupId(self):
        """
        Returns the unique ID of the group this algorithm belongs to. This
        string should be fixed for the algorithm, and must not be localised.
        The group id should be unique within each provider. Group id should
        contain lowercase alphanumeric characters only and no spaces or other
        formatting characters.
        """
        return "geographiccontext"

    def tr(self, string):
        return QCoreApplication.translate("Processing", string)

    def checkParameterValues(self, parameters, context):
        if (
            parameters[self.USE_GEOMORPHON] == 1
            and parameters[self.GEOMORPHON_FIELD] == NULL
        ):
            return (
                False,
                "A field with topographic position is required in 'Topographic position field\
                            if 'Use topographic field' choice is toggle",
            )

        dem = self.parameterAsRasterLayer(parameters, self.DEM, context)
        if dem.crs().isGeographic():
            return (False, "Your DEM must not be in a geographic projection system.")
        return (True, "")

    def createInstance(self):
        return RelativeOrientationAlgorithm()
