# -*- coding: utf-8 -*-

"""
/***************************************************************************
 SplitByNetwork
                                 A QGIS plugin
 Split a list of layer by a list of network layers.
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2022-01-11
        copyright            : (C) 2022 by Dynafor
        email                : gabriel.marques@toulouse-inp.fr
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

__author__ = "Dynafor"
__date__ = "2023-02-27"
__copyright__ = "(C) 2022 by Dynafor"

# This will get replaced with a git SHA1 when you do a git archive

__revision__ = "$Format:%H$"

from qgis.PyQt.QtCore import QCoreApplication
from qgis.core import (
    QgsProcessing,
    QgsFeatureSink,
    QgsProcessingAlgorithm,
    QgsProcessingParameterFeatureSource,
    QgsProcessingParameterFeatureSink,
    QgsProcessingParameterNumber,
    QgsProcessingParameterMultipleLayers,
    QgsProcessingParameterBoolean,
    QgsProcessingContext,
    QgsVectorLayer,
)
from qgis.PyQt.QtGui import QIcon
from hedge_tools import (
    resources,
)  # Only need in hedge_tools.py normaly but just to keep track of import

from hedge_tools.tools.vector import qgis_wrapper as qw
from hedge_tools.tools.vector import geometry as g
from hedge_tools.tools.vector import utils


class SplitByNetworkAlgorithm(QgsProcessingAlgorithm):
    """
    Split a list of polygon vector layer (usually hedges and tree)
    by a list of linestring vector layer such as roads, railways and rivers.

    Parameters
    ---
    HEDGES (QgsVectorLayer): Layer input from users.
    TREES (QgsVectorLayer): Layer input from users.
    OVERLAYS (list[QgsVectorLayer]): Network layers from users.
    BUFFER (float): Buffer value around the network layers.
    DELETION (bool): If True will use MIN_AREA parameter.
    MIN_AREA (float): Deletion of geometry below this area threshold after the split.

    Return
    ---
    OUTPUT (QgisObject : QgsVectorLayer) : Fixed layer.
    """

    # Constants used to refer to parameters and outputs. They will be
    # used when calling the algorithm from another algorithm, or when
    # calling from the QGIS console.
    HEDGES = "HEDGES"
    TREES = "TREES"
    OVERLAYS = "OVERLAYS"
    OFFSET = "OFFSET"
    TREE_MIN_AREA = "TREE_MIN_AREA"
    DELETION = "DELETION"
    MIN_AREA = "MIN_AREA"
    SPLIT_HEDGES = "SPLIT_HEDGES"
    SPLIT_TREES = "SPLIT_TREES"

    def initAlgorithm(self, config):
        """
        Here we define the inputs and output of the algorithm, along
        with some other properties.
        """

        self.addParameter(
            QgsProcessingParameterFeatureSource(
                self.HEDGES,
                self.tr("Hedge layer"),
                [QgsProcessing.TypeVectorPolygon],
                optional=True,
            )
        )

        self.addParameter(
            QgsProcessingParameterFeatureSource(
                self.TREES,
                self.tr("Tree layer"),
                [QgsProcessing.TypeVectorPolygon],
                optional=True,
            )
        )

        self.addParameter(
            QgsProcessingParameterMultipleLayers(
                self.OVERLAYS, self.tr("Network layers"), QgsProcessing.TypeVectorLine
            )
        )

        self.addParameter(
            QgsProcessingParameterNumber(
                name=self.OFFSET,
                description=self.tr("Buffer value around the networks geometries"),
                type=QgsProcessingParameterNumber.Double,
                defaultValue=1,
                optional=False,
                minValue=0,
            )
        )

        self.addParameter(
            QgsProcessingParameterNumber(
                name=self.TREE_MIN_AREA,
                description=self.tr("Tree area threshold (m²)"),
                type=QgsProcessingParameterNumber.Double,
                defaultValue=50,
                optional=True,
                minValue=0,
            )
        )

        self.addParameter(
            QgsProcessingParameterBoolean(
                name=self.DELETION,
                description=self.tr("Delete geometries below minimum area value"),
                defaultValue=True,
            )
        )

        self.addParameter(
            QgsProcessingParameterNumber(
                name=self.MIN_AREA,
                description=self.tr("Minimum area of a geometry (m²)"),
                type=QgsProcessingParameterNumber.Double,
                defaultValue=5,
                optional=True,
                minValue=0,
            )
        )

        self.addParameter(
            QgsProcessingParameterFeatureSink(
                self.SPLIT_HEDGES, self.tr("Splitted hedges")
            )
        )

        self.addParameter(
            QgsProcessingParameterFeatureSink(
                self.SPLIT_TREES, self.tr("Splitted trees")
            )
        )

    def __init__(self):
        super().__init__()

        self.len_alg = 0
        self.steps = 2
        self.step_per_alg = None
        self.value = 0
        self.layers = {"hedges": None, "trees": None}
        self.outputs = {"hedges": None, "trees": None}

    def processAlgorithm(self, parameters, context, feedback):
        """
        Here is where the processing itself takes place.
        """
        if parameters[self.HEDGES] is not None:
            self.layers["hedges"] = self.parameterAsVectorLayer(
                parameters, self.HEDGES, context
            )
            self.len_alg += 1
        if parameters[self.TREES] is not None:
            self.layers["trees"] = self.parameterAsVectorLayer(
                parameters, self.HEDGES, context
            )
            self.len_alg += 1
        networks = self.parameterAsLayerList(parameters, self.OVERLAYS, context)
        buffer = self.parameterAsDouble(parameters, self.OFFSET, context)
        min_tree_area = self.parameterAsDouble(parameters, self.TREE_MIN_AREA, context)
        if parameters[self.DELETION]:
            min_area = self.parameterAsDouble(parameters, self.MIN_AREA, context)
        else:
            min_area = None

        self.step_per_alg = 100 // (self.steps * self.len_alg)
        self.progress_bar(feedback)
        feedback.pushInfo("Merging network layers...")
        network = qw.merge_layers(networks)
        network = qw.dissolve(network)

        tree_list = []
        for label, layer in self.layers.items():
            name = f"SPLIT_{label.upper()}"
            if layer is not None:
                feedback.pushInfo(f"Splitting {label} layer...")
                del_list = list()
                output = self.split_by_network(layer, network, buffer)

                self.progress_bar(feedback)

                # Deletion of artifact
                if min_area is not None:
                    del_list.extend(
                        [
                            f.id()
                            for f in output.getFeatures()
                            if f.geometry().area() < min_area
                        ]
                    )
                    output.dataProvider().deleteFeatures(del_list)

                # Handle feature trasnfer between hedges and trees
                if label == "hedges" and self.layers["trees"] is not None:
                    tree_list = [
                        f
                        for f in output.getFeatures()
                        if f.geometry().area() < min_tree_area
                    ]
                    del_list = [f.id() for f in tree_list]
                    output.dataProvider().deleteFeatures(del_list)

                elif label == "hedges" and self.layers["trees"] is None:
                    feedback.pushWarning(
                        "There is no trees layer, therefore \
                                        the minimum tree area will not be used."
                    )
                elif label == "trees" and bool(tree_list):
                    output.dataProvider().addFeatures(tree_list)

                # Delete fields created by processing
                output = self.processing_fields_deletion(output, label)

                # Create and store sink
                fid_idx = output.fields().indexFromName("fid")
                if fid_idx != -1:
                    _, output = utils.update_unique_constraint(
                        feedback, output, fid_idx
                    )
                self.outputs[label] = self.create_sink(
                    output, name, parameters, context
                )

            # Update dict key to match alg output name
            self.outputs[name] = self.outputs.pop(label)
            self.progress_bar(feedback)

        return self.outputs

    def progress_bar(self, feedback):
        """
        Incrementation of progress bar and cancellation
        """
        # Set progress
        feedback.setProgress(self.value)
        self.value += self.step_per_alg
        # Check for cancellation
        if feedback.isCanceled():
            return {}

    def create_sink(
        self,
        layer: QgsVectorLayer,
        sink_name: str,
        parameters: dict,
        context: QgsProcessingContext,
    ) -> int:
        """
        Create a sink and store the id in self.output

        Parameters
        ---
        layer: QgsVectorLayer
            Layer to duplicate into a sink

        Return
        ---
        id: int
            Identifier of the sink layer
        """
        (sink, id) = self.parameterAsSink(
            parameters,
            sink_name,
            context,
            layer.fields(),
            layer.wkbType(),
            layer.sourceCrs(),
        )
        for feature in layer.getFeatures():
            sink.addFeature(feature, QgsFeatureSink.FastInsert)

        return id

    def processing_fields_deletion(
        self, layer: QgsVectorLayer, label: str
    ) -> QgsVectorLayer:
        """
        Will use self.layers and the current label to fetch input fields
        and delete new fields created by the processing.

        Parameters
        ---
        layer: QgsVectorLayer
            Current splitted layer.
        label: str
            Current label.

        Return
        ---
        output: QgsVectorLayer
        """
        input_fields = set(self.layers[label].fields().names())
        del_list = list(set(layer.fields().names()) - input_fields)
        output = qw.delete_column(layer, del_list)

        return output

    def split_by_network(
        self, layer: QgsVectorLayer, network: QgsVectorLayer, buffer: float = 1.0
    ) -> QgsVectorLayer:
        """
        Split a layer by a buffered linear network

        Parameters
        ---
        layer : QgsVectorLayer : Polygon.
        network : QgsVectorLayer : LineString.
        buffer : float :
            Default = 2
            Buffer to use on the network before the split of layer


        Return
        ---
        layer : QgsVectorLayer : Polygon.
            Splitted layer
        """
        buffered = qw.buffer(network, buffer, True, 2, 0, segments=5)
        layer = qw.difference(layer, buffered)
        layer = qw.multipart_to_singleparts(layer)
        layer = qw.remove_null_geometries(layer, True)

        return layer

    def icon(self):
        """
        Should return a QIcon which is used for your provider inside
        the Processing toolbox.
        """
        return QIcon(":/plugins/hedge_tools/images/hedge_tools.png")

    def shortHelpString(self):
        """
        Returns a localised short help string for the algorithm.
        """
        return self.tr(
            "Split hedges and/or trees layer with a list of network layers. \
                        A buffer value will be used around the nerworks \
                        to split the polygonal features. \n\
                        If both inputs are used the tree area threshold allows \
                        to transfer hedges with an area below the specified value \
                        to the tree layer after the split. \n\
                        The split can create various small geometries, \
                        therefore deletion of artifact is available with \
                        the minimum area parameter. \n\
                        It is strongly advised to keep the same buffer value \
                        in this tool and in the aggregation tool."
        )

    def name(self):
        """
        Returns the algorithm name, used for identifying the algorithm. This
        string should be fixed for the algorithm, and must not be localised.
        The name should be unique within each provider. Names should contain
        lowercase alphanumeric characters only and no spaces or other
        formatting characters.
        """
        return "splitbynetwork"

    def displayName(self):
        """
        Returns the translated algorithm name, which should be used for any
        user-visible display of the algorithm name.
        """
        return self.tr("6 - Split by network [optional]")

    def group(self):
        """
        Returns the name of the group this algorithm belongs to. This string
        should be localised.
        """
        return self.tr("0 - Extraction [optional]")

    def groupId(self):
        """
        Returns the unique ID of the group this algorithm belongs to. This
        string should be fixed for the algorithm, and must not be localised.
        The group id should be unique within each provider. Group id should
        contain lowercase alphanumeric characters only and no spaces or other
        formatting characters.
        """
        return "extraction"

    def tr(self, string):
        return QCoreApplication.translate("Processing", string)

    def checkParameterValues(self, parameters, context):

        if parameters[self.HEDGES] is None and parameters[self.TREES] is None:
            return (False, "At least one input layer is needed.")

        return (True, "")

    def createInstance(self):
        return SplitByNetworkAlgorithm()
